/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.model;

import jp.co.fujitsu.reffi.client.nexaweb.event.ModelProcessEvent;
import jp.co.fujitsu.reffi.client.nexaweb.listener.SubscribeManager;

import com.nexaweb.util.XmlSerializable;
import com.nexaweb.xml.Document;
import com.nexaweb.xml.Parser;
import com.nexaweb.xml.ParserException;
import com.nexaweb.xml.ParserFactory;

/**
 * <p>[概 要] </p>
 * トピックをクライアントで購読する為の機能モデルクラスです。
 * 
 * <p>[詳 細] </p>
 * 
 * <p>[備 考] </p>
 * 
 * <b>使用例）</b><br>
 * ・指定回数subscribeしたらunsubscribeする
 * 
 * <pre class="samplecode">
 *    public class ServerPushDemoAction extends BaseAction {
 *        &#064;Override
 *        protected void reserveModels(List<Class<? extends Model>> models) {
 *            models.add(SubscribeCore.class);
 *        }
 *
 *        &#064;Override
 *        public boolean nextModel(int index, ModelProcessEvent prev, Model next) throws Exception {
 *            switch(index){
 *                case 0:
 *                    ((SubscribeCore)next).setTopic("serverPushDemo");
 *                    ((SubscribeCore)next).setSubscribe(true);
 *                    break;
 *            }
 *            return true;
 *        }
 *
 *        &#064;Override
 *        public void successForward(int index, Model model, Object result) throws Exception {
 *            switch(index) {
 *                case 0:
 *                    // 3回subscribeしたらunsubscribeする
 *                    SubscribeCore subscribeCore = ((SubscribeCore)model);
 *                    if(subscribeCore.getSuccessCount() > 3) {
 *                        SubscriberManager.getInstance().unsubscribe(subscribeCore.getTopic());
 *                    }
 *                
 *                    ObjectDataSource odc = getObjectDataSourceById("hostDataSource");
 *                    odc.setSource(result);
 *                    break;
 *            }
 *        }
 *    }
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @see SubscribeManager
 * @author Project Reffi 
 */
public class SubscribeCore extends XalReturnPossibilityModel {

	/** 購読開始、中止対象トピック名です */
	private String topic;

	/** 開始、中止を判断するフラグです（true:購読開始、false:中止、デフォルト：true） */
	private boolean subscribe = true;

	/** トピックを受信中のSubscribeCoreを識別する為の識別子です。 */
	private String identifier;

	/**
	 * <p>[概 要] </p>
	 * 購読開始、中止対象トピック名を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * topicフィールド値を返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 購読開始、中止対象トピック名
	 */
	public String getTopic() {
		return this.topic;
	}

	/**
	 * <p>[概 要] </p>
	 * 購読開始、中止対象トピック名を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * topicフィールド値を引数topicで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param topic 購読開始、中止対象トピック名
	 */
	public void setTopic(String topic) {
		this.topic = topic;
	}

	/**
	 * <p>[概 要] </p>
	 * 開始、中止を判断するフラグを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * subscribeフィールド値を返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 開始、中止フラグ
	 */
	public boolean isSubscribe() {
		return this.subscribe;
	}

	/**
	 * <p>[概 要] </p>
	 * 開始、中止を判断するフラグを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * subscribeフィールド値を引数subscribeで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param subscribe 開始、中止フラグ
	 */
	public void setSubscribe(boolean subscribe) {
		this.subscribe = subscribe;
	}

	/**
	 * <p>[概 要] </p>
	 * トピックを受信中のSubscribeCoreを識別する為の識別子を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * identifierフィールド値を返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return トピックを受信中のSubscribeCoreを識別する為の識別子
	 */
	public String getIdentifier() {
		return this.identifier;
	}

	/**
	 * <p>[概 要] </p>
	 * トピックを受信中のSubscribeCoreを識別する為の識別子を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * identifierフィールド値を引数identifierで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param identifire トピックを受信中のSubscribeCoreを識別する為の識別子
	 */
	public void setIdentifier(String identifier) {
		this.identifier = identifier;
	}

	/**
	 * <p>[概 要] </p>
	 * subscribeの開始、中止処理を行います。
	 * 
	 * <p>[詳 細] </p>
	 * subscribe管理オブジェクト（SubscribeManager）を取得し、購読の開始、中止を委譲します。<br>
	 * 対象となるトピック名はフィールド「topic」から、購読開始、中止の判断を行うフラグは
	 * フィールド「subscribe」から取得します。<p>
	 * 
	 * 購読中止（isSubscribe() == false）時、unsubscribeメソッド処理が例外をスローしなかった
	 * 場合は、その時点でModel処理成功イベントを発火します。<br>
	 * 
	 * <p>[備 考] </p>
	 * 購読中止時のModel処理成功イベント内結果オブジェクトはnullです。
	 * 利用することは出来ません。
	 * 
	 */
	@Override
	protected void mainproc() throws Exception {
		if (isSubscribe()) {
			// 購読開始
			// subscribe管理オブジェクトにtopicの購読を委譲
			SubscribeManager.instance.subscribe(getTopic(), this);
		} else {
			// 購読中止
			// subscribe管理オブジェクトにtopicの購読中止を委譲
			SubscribeManager.instance.unsubscribe(getTopic(), getIdentifier());

			// 購読中止の場合、この時点で例外が発生していなければモデル処理成功と見做す
			ModelProcessEvent evt = new ModelProcessEvent(this);
			// モデル処理結果オブジェクトは無し
			evt.setResult(null);
			// 発火
			fireModelSuccess(evt);
			
			fireModelFinished(new ModelProcessEvent(this));
		}
	}

	/**
	 * <p>[概 要] </p>
	 * subscribe受信時にコールバックされるメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * SubscribeManagerに管理されているMessageListnerのonMessageイベントハンドラ
	 * からコールバックされます。<br>
	 * subscribe受信データ取得用オーバーライドメソッドであるpostSubscribeを
	 * テンプレートコールします。<br>
	 * postSubscribeメソッドが正常終了した後、Model処理成功イベントを発火します。<br>
	 * 
	 * <p>[備 考] </p>
	 * このメソッドはオーバーライド出来ません。<br>
	 * subscribe受信データをモデルレイヤで取得したい場合は、
	 * postSubscribeメソッドをオーバーライドして下さい。
	 * 
	 * @param result subscribeの結果受信したオブジェクト
	 */
	public final void subscribeCompleted(Object result) {
		// 汎用モデル処理結果オブジェクトに設定
		setResult(result);

		// オーバーライド用テンプレートメソッドをコール
		postSubscribe(result);

		// プッシュされたデータがxalだった場合、
		// xal内window、dialogに識別子（parentId, communicateId）を入れる
		if (result instanceof XmlSerializable) {
			try {
				String xml = ((XmlSerializable) result).toXml();
				Parser parser = ParserFactory.getParser();
				Document document = parser.parseXml(xml);
				addIdentifierToWindows(document);
				result = document;
			} catch (ParserException e) {
				// ここに入る契機は無い
			}
		}

		// Model処理成功イベントを生成
		ModelProcessEvent evt = new ModelProcessEvent(this);
		// subscribe結果を格納
		evt.setResult(result);
		// 発火
		fireModelSuccess(evt);
	}

	/**
	 * <p>[概 要] </p>
	 * subscribe結果受信用オーバーライドメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * デフォルト処理は有りません。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param result subscribeの結果受信したオブジェクト
	 */
	protected void postSubscribe(Object result) {
	}
}
