/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_board_2ch.c,v 1.31 2004/01/11 22:29:14 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_board_2ch.h"
#include "ochusha_thread_2ch.h"
#include "ochusha_utils_2ch.h"

#include "htmlutils.h"
#include "utils.h"

#include <glib-object.h>
#include <glib.h>

#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <unistd.h>

#include <zlib.h>


#define ANALYSIS_INTERVAL	50


static OchushaBulletinBoardClass *parent_class = NULL;


static void ochusha_board_2ch_class_init(OchushaBoard2chClass *klass);
static void ochusha_board_2ch_init(OchushaBoard2ch *board);
static void ochusha_board_2ch_finalize(GObject *object);

static void ochusha_board_2ch_read_boardlist_element(
						OchushaBulletinBoard *board,
						GHashTable *board_attributes);
static void ochusha_board_2ch_write_boardlist_element(
						OchushaBulletinBoard *board,
						gzFile *boardlist_xml);

static const char *ochusha_board_2ch_get_subject_txt_encoding(
						OchushaBulletinBoard *board);

static void ochusha_board_2ch_recover_threadlist(OchushaBulletinBoard *board,
						 OchushaConfig *config);
static char *ochusha_board_2ch_generate_base_path(OchushaBulletinBoard *board,
						  const char *url);
static char *ochusha_board_2ch_generate_board_id(OchushaBulletinBoard *board,
						 const char *url);
static OchushaBBSThread *ochusha_board_2ch_thread_new(
						OchushaBulletinBoard *board,
						const char *id,
						const gchar *title);
static OchushaBBSThread *ochusha_board_2ch_lookup_thread_by_url(
						OchushaBulletinBoard *board,
						const char *url);
static OchushaBBSThread *ochusha_board_2ch_lookup_kako_thread_by_url(
						OchushaBulletinBoard *board,
						OchushaNetworkBroker *broker,
						OchushaAsyncBuffer *buffer,
						const char *url);
static OchushaAsyncBuffer *ochusha_board_2ch_get_threadlist_source(
					OchushaBulletinBoard *board,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode);
static gboolean ochusha_board_2ch_refresh_threadlist(
						OchushaBulletinBoard *board,
						OchushaAsyncBuffer *buffer,
						EachThreadCallback *cb,
						gpointer callback_data);

static const char *ochusha_board_2ch_get_response_character_encoding(
						OchushaBulletinBoard *board);
static iconv_helper *ochusha_board_2ch_get_response_iconv_helper(
						OchushaBulletinBoard *board);
static gboolean ochusha_board_2ch_is_new_thread_supported(
						OchushaBulletinBoard *board);
static gboolean ochusha_board_2ch_create_new_thread(
					OchushaBulletinBoard *board,
					OchushaNetworkBroker *broker,
					const gchar *title,
					const OchushaBBSResponse *response);

GType
ochusha_board_2ch_get_type(void)
{
  static GType board_2ch_type = 0;

  if (board_2ch_type == 0)
    {
      static const GTypeInfo board_2ch_info =
	{
	  sizeof(OchushaBoard2chClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_board_2ch_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaBoard2ch),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_board_2ch_init,
	};

      board_2ch_type = g_type_register_static(OCHUSHA_TYPE_BULLETIN_BOARD,
					      "OchushaBoard2ch",
					      &board_2ch_info, 0);
    }

  return board_2ch_type;
}


static void
ochusha_board_2ch_class_init(OchushaBoard2chClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  OchushaBulletinBoardClass *b_class = OCHUSHA_BULLETIN_BOARD_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = ochusha_board_2ch_finalize;

  b_class->read_boardlist_element = ochusha_board_2ch_read_boardlist_element;
  b_class->write_boardlist_element = ochusha_board_2ch_write_boardlist_element;

  b_class->recover_threadlist = ochusha_board_2ch_recover_threadlist;
  b_class->generate_base_path = ochusha_board_2ch_generate_base_path;
  b_class->generate_board_id = ochusha_board_2ch_generate_board_id;
  b_class->thread_new = ochusha_board_2ch_thread_new;
  b_class->lookup_thread_by_url = ochusha_board_2ch_lookup_thread_by_url;
  b_class->lookup_kako_thread_by_url
    = ochusha_board_2ch_lookup_kako_thread_by_url;
  b_class->get_threadlist_source = ochusha_board_2ch_get_threadlist_source;
  b_class->refresh_threadlist = ochusha_board_2ch_refresh_threadlist;

  b_class->get_response_character_encoding
    = ochusha_board_2ch_get_response_character_encoding;
  b_class->get_response_iconv_helper
    = ochusha_board_2ch_get_response_iconv_helper;

  b_class->new_thread_supported
    = ochusha_board_2ch_is_new_thread_supported;
  b_class->create_new_thread
    = ochusha_board_2ch_create_new_thread;

  klass->get_read_cgi_url = NULL;
}


static void
ochusha_board_2ch_init(OchushaBoard2ch *board)
{
  OCHUSHA_BULLETIN_BOARD(board)->bbs_type = OCHUSHA_BBS_TYPE_2CH;

  board->read_cgi_url = NULL;
  board->last_modified = NULL;
  board->date = NULL;
  board->cookie = NULL;
}


static void
ochusha_board_2ch_finalize(GObject *object)
{
  OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(object);

  if (board_2ch->read_cgi_url != NULL)
    {
      G_FREE(board_2ch->read_cgi_url);
      board_2ch->read_cgi_url = NULL;
    }

  if (board_2ch->last_modified != NULL)
    {
      G_FREE(board_2ch->last_modified);
      board_2ch->last_modified = NULL;
    }

  if (board_2ch->date != NULL)
    {
      G_FREE(board_2ch->date);
      board_2ch->date = NULL;
    }

  if (board_2ch->cookie != NULL)
    {
      G_FREE(board_2ch->cookie);
      board_2ch->cookie = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
ochusha_board_2ch_read_boardlist_element(OchushaBulletinBoard *board,
					 GHashTable *board_attributes)
{
  OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(board);
  board_2ch->last_modified
    = ochusha_utils_get_attribute_string(board_attributes, "last_modified");
  board_2ch->cookie
    = ochusha_utils_get_attribute_string(board_attributes, "cookie");

  if (parent_class->read_boardlist_element != NULL)
    (*parent_class->read_boardlist_element)(board, board_attributes);
}


#define OUTPUT_BOARD_ATTRIBUTE_STRING(gzfile, board, attribute)		\
  do									\
    {									\
      if ((board)->attribute != NULL)					\
	{								\
	  gchar *text = g_markup_escape_text((board)->attribute, -1);	\
	  gzprintf(gzfile,						\
		   "        <attribute name=\"" #attribute	"\">\n"	\
		   "          <string>%s</string>\n"			\
		   "        </attribute>\n", text);			\
	  g_free(text);							\
	}								\
    } while (0)


static void
ochusha_board_2ch_write_boardlist_element(OchushaBulletinBoard *board,
					  gzFile *boardlist_xml)
{
  OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(board);

  if (parent_class->write_boardlist_element != NULL)
    (*parent_class->write_boardlist_element)(board, boardlist_xml);

  OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, board_2ch, last_modified);
  OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, board_2ch, cookie);
}


static const char *
ochusha_board_2ch_get_subject_txt_encoding(OchushaBulletinBoard *board)
{
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      return "CP932";

    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
      return "EUC-JP";

    default:
      return NULL;
    }
}


static gboolean
recover_title(OchushaBBSThread *thread, const char *title, gpointer unused)
{
  const char *encoding
    = ochusha_bbs_thread_get_response_character_encoding(thread);
  iconv_helper *helper
    = ochusha_bbs_thread_get_response_iconv_helper(thread);
  iconv_t converter = iconv_open(helper ? "UTF-8" : "UTF-8//IGNORE", encoding);
  gchar *recovered_title;

  if (converter == (iconv_t)-1)
    return FALSE;

  recovered_title = simple_string_canon(title, -1, converter, helper);
  ochusha_bbs_thread_set_title(thread, recovered_title);

  G_FREE(recovered_title);
  iconv_close(converter);

  return FALSE;
}


static void
ochusha_board_2ch_recover_threadlist(OchushaBulletinBoard *board,
				     OchushaConfig *config)
{
  GDir *dir;
  char *dat_dirname;
  const char *dat_filename;
  char path[PATH_MAX];

  g_return_if_fail(OCHUSHA_IS_BOARD_2CH(board));

  if (snprintf(path, PATH_MAX, "cache/%s%s%s/dat",
	       ochusha_bulletin_board_get_server(board),
	       ochusha_bulletin_board_get_base_path(board),
	       ochusha_bulletin_board_get_id(board)) >= PATH_MAX)
    return;

  dat_dirname = ochusha_config_find_directory(config, path, NULL);

  if (dat_dirname == NULL)
    return;

  dir = g_dir_open(dat_dirname, 0, NULL);
  G_FREE(dat_dirname);

  g_return_if_fail(dir != NULL);

  while (TRUE)
    {
      char *id;
      const char *dat_pos;
      OchushaBBSThread *thread;

      dat_filename = g_dir_read_name(dir);
      if (dat_filename == NULL)
	break;

      dat_pos = strstr(dat_filename, ".dat");
      if (dat_pos == NULL || dat_pos[4] != '\0')
	continue;

      id = G_STRNDUP(dat_filename, dat_pos - dat_filename);

      thread = ochusha_bulletin_board_lookup_bbs_thread_by_id(board, id);

      if (thread == NULL)
	{
	  if (snprintf(path, PATH_MAX, "cache/%s%s%s/dat/%s",
		       ochusha_bulletin_board_get_server(board),
		       ochusha_bulletin_board_get_base_path(board),
		       ochusha_bulletin_board_get_id(board),
		       dat_filename) < PATH_MAX)
	    {
	      int fd = ochusha_config_open_file(config, path, NULL, O_RDONLY);
	      if (fd >= 0)
		{
		  OchushaAsyncBuffer *dat_buffer
		    = ochusha_async_buffer_new_with_file_mmap(fd);
		  ochusha_async_buffer_fix(dat_buffer);
		  thread = ochusha_bulletin_board_bbs_thread_new(board, id,
								 NULL);
		  ochusha_bbs_thread_parse_responses(thread,
						     dat_buffer,
						     0, 1, TRUE,
						     recover_title,
						     NULL, NULL, NULL,
						     NULL);
		  OCHU_OBJECT_UNREF(G_OBJECT(dat_buffer));
		  thread->number_of_responses_read = 1;
		  board->thread_list = g_slist_append(board->thread_list,
						      thread);
		}
	    }
	}
      else if (thread->number_of_responses_read == 0)
	thread->number_of_responses_read = 1;

      G_FREE(id);
    };

  g_dir_close(dir);
}


static char *
ochusha_board_2ch_generate_base_path(OchushaBulletinBoard *board,
				     const char *url)
{
  return G_STRDUP("/");
}


static char *
ochusha_board_2ch_generate_board_id(OchushaBulletinBoard *board,
				    const char *url)
{
  char *id;
  char *abs_path = ochusha_utils_url_extract_http_absolute_path(url);
  if (abs_path != NULL && abs_path[0] != '/')
    {
      G_FREE(abs_path);
      abs_path = NULL;
    }
  g_return_val_if_fail(abs_path != NULL, NULL);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
      {
	char *tmp_pos = strchr(abs_path + 1, '/');
	if (tmp_pos != NULL)
	  {
	    *tmp_pos = '\0';
	  }
	id = G_STRDUP(abs_path + 1);
	break;
      }

    default:
      id = NULL;	/* ̤ݡ */
      break;
    }

  G_FREE(abs_path);

  return id;
}


static OchushaBBSThread *
ochusha_board_2ch_thread_new(OchushaBulletinBoard *board, const char *id,
			     const gchar *title)
{
  OchushaBBSThread *thread;
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board) && id != NULL, NULL);

  thread = ochusha_thread_2ch_new(OCHUSHA_BOARD_2CH(board), id, title);
  g_return_val_if_fail(thread != NULL, NULL);

  return thread;
}


static OchushaBBSThread *
ochusha_board_2ch_lookup_thread_by_url(OchushaBulletinBoard *board,
				       const char *url)
{
  OchushaBBSThread *thread = NULL;
  char *board_url = NULL;
  char *thread_id = NULL;
  gboolean is_kako_html = FALSE;

  if (ochusha_utils_2ch_check_url(url, &board_url, NULL, NULL, &thread_id,
				  NULL, NULL, &is_kako_html))
    {
      if (thread_id != NULL)
	{
#if 0	/* ǤƱ */
	  if (is_kako_html)
	    {
	      /* TODO: ФΥåˤ⤦֤ */
	      if (g_ascii_strcasecmp(board->base_url, board_url) == 0)
		thread
		  = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								   thread_id);
	    }
	  else
#endif
	    {
	      if (g_ascii_strcasecmp(board->base_url, board_url) == 0)
		thread
		  = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								   thread_id);
	    }
	}
    }

  if (board_url != NULL)
    G_FREE(board_url);

  if (thread_id != NULL)
    G_FREE(thread_id);

  return thread;
}


static OchushaBBSThread *
ochusha_board_2ch_lookup_kako_thread_by_url(OchushaBulletinBoard *board,
					    OchushaNetworkBroker *broker,
					    OchushaAsyncBuffer *buffer,
					    const char *url)
{
  OchushaBBSThread *thread = NULL;
  char *thread_id = NULL;
  gboolean is_kako_html = FALSE;

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board)
		       && OCHUSHA_IS_NETWORK_BROKER(broker)
		       && url != NULL, NULL);

  if (ochusha_utils_2ch_check_url(url, NULL, NULL, NULL, &thread_id,
				  NULL, NULL, &is_kako_html))
    {
      if (thread_id != NULL)
	{
	  OchushaAsyncBuffer *dat_buffer;
	  OchushaThread2ch *thread_2ch;

	  thread = ochusha_bulletin_board_bbs_thread_new(board, thread_id,
							 NULL);
	  thread_2ch = OCHUSHA_THREAD_2CH(thread);
	  ochusha_thread_2ch_set_kako_html_url(thread_2ch, url);
	  dat_buffer = ochusha_bbs_thread_get_responses_source(thread, broker,
					buffer,
					OCHUSHA_NETWORK_BROKER_CACHE_AS_IS);
	  if (dat_buffer != NULL)
	    {
	      ochusha_bbs_thread_parse_responses(thread,
						 dat_buffer,
						 0, -1, FALSE,
						 recover_title,
						 NULL, NULL, NULL, NULL);
	      ochusha_async_buffer_suspend(dat_buffer);
	      ochusha_async_buffer_terminate(dat_buffer);
	      OCHU_OBJECT_UNREF(G_OBJECT(dat_buffer));

	      if (thread->title != NULL)
		{
		  thread->number_of_responses_read = 1;
		  board->thread_list = g_slist_append(board->thread_list,
						      thread);
		}
	      else
		{
		  OCHU_OBJECT_UNREF(G_OBJECT(thread));
		  thread = NULL;
		}
	    }
	  G_FREE(thread_id);
	}
    }

  return thread;
}


static const char *
ochusha_board_2ch_get_response_character_encoding(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board), NULL);

  switch (((OchushaBulletinBoard *)board)->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      return "CP932";

    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
      return "EUC-JP";

    default:
      return NULL;
    }
}


static iconv_helper *
ochusha_board_2ch_get_response_iconv_helper(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board), NULL);

  switch (((OchushaBulletinBoard *)board)->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      return cp932_to_utf8_helper;

    default:
      return NULL;
    }
}


static gboolean
ochusha_board_2ch_is_new_thread_supported(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board), FALSE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      return TRUE;

    default:
      return FALSE;
    }
}


static gboolean
ochusha_board_2ch_create_new_thread(OchushaBulletinBoard *board,
				    OchushaNetworkBroker *broker,
				    const gchar *title_utf8,
				    const OchushaBBSResponse *response)
{
  OchushaBoard2ch *board_2ch;
  char *tmp_string = NULL;
  char *title = NULL;
  char *from = NULL;
  char *mail = NULL;
  char *message = NULL;
  const char *bbs;
  char *query = NULL;
  long time;	/* time_tˤ٤10ʿʸؤѴ
		 * ԤˡʤΤǡ褺longѴƤޤ
		 * long32bitʥƥˤ2038ǯ()¸ߤ롣
		 */
  iconv_t converter;
  OchushaUtils2chPostResult post_result;

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board), FALSE);
  g_return_val_if_fail(title_utf8 != NULL && *title_utf8 != '\0', FALSE);
  g_return_val_if_fail(response->name != NULL, FALSE);
  g_return_val_if_fail(response->mailto != NULL, FALSE);
  g_return_val_if_fail(response->content != NULL, FALSE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      break;

    default:
      return FALSE;	/* ̤ݡ */
    }

  converter = iconv_open(ochusha_bulletin_board_get_response_character_encoding(board), "UTF-8");
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  board_2ch = OCHUSHA_BOARD_2CH(board);

  tmp_string = convert_string(converter, NULL, title_utf8, -1);
  title = ochusha_utils_url_encode_string(tmp_string);
  if (title == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->name, -1);
  from = ochusha_utils_url_encode_string(tmp_string);
  if (from == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->mailto, -1);
  mail = ochusha_utils_url_encode_string(tmp_string);
  if (mail == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->content, -1);
  message = ochusha_utils_url_encode_string(tmp_string);
  if (message == NULL)
    goto error_exit;
  G_FREE(tmp_string);
  tmp_string = NULL;

  bbs = ochusha_bulletin_board_get_id(board);

  if (board_2ch->date != NULL)
    time = ochusha_utils_get_utc_time(board_2ch->date);
  else
    {
#if DEBUG_POST
      char message[4096];
      snprintf(message, 4096, "board_2ch(%s)->date == NULL\n",
	       board->name);
      ochusha_network_broker_output_log(broker, message);
#endif
#if DEBUG_POST_MOST
      fprintf(stderr, "board_2ch->date == NULL!\n");
#endif
      time = ochusha_utils_get_utc_time(board_2ch->last_modified);
    }

  if (time == -1)
    goto error_exit;

  /* XXX: ϡɥǥ󥰤㤦ΤϤäʤġġ*/
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      query = g_strdup_printf("submit=%%90%%56%%8B%%4B%%83%%58%%83%%8C%%83%%62%%83%%68%%8D%%EC%%90%%AC&subject=%s&FROM=%s&mail=%s&MESSAGE=%s&bbs=%s&time=%ld", title, from, mail, message, bbs, time);
      break;

    default:
      goto error_exit;
    }

  post_result = ochusha_utils_2ch_try_post(broker, board, query);
  if (post_result == OCHUSHA_UTILS_2CH_POST_NO_COOKIE)
    {
      sleep(10);
      post_result = ochusha_utils_2ch_try_post(broker, board, query);
    }

#if 0
  if (post_result == OCHUSHA_UTILS_2CH_POST_MOCHITSUKE)
    {
      /* XXX: äݤΤȤͭʤΤ֡ */
      sleep(30);
      post_result = ochusha_utils_2ch_try_post(broker, board, query);
    }
#endif
  if (post_result != OCHUSHA_UTILS_2CH_POST_SUCCESS)
    goto error_exit;

  /* ｪλ */
  if (tmp_string != NULL)
    G_FREE(tmp_string);
  if (title != NULL)
    G_FREE(title);
  if (from != NULL)
    G_FREE(from);
  if (mail != NULL)
    G_FREE(mail);
  if (message != NULL)
    G_FREE(message);
  if (query != NULL)
    G_FREE(query);
  iconv_close(converter);

  return TRUE;

 error_exit:
#if DEBUG_POST_MOST
  fprintf(stderr, "Error happen\n");
#endif

  if (tmp_string != NULL)
    G_FREE(tmp_string);
  if (title != NULL)
    G_FREE(title);
  if (from != NULL)
    G_FREE(from);
  if (mail != NULL)
    G_FREE(mail);
  if (message != NULL)
    G_FREE(message);
  if (query != NULL)
    G_FREE(query);
  iconv_close(converter);

  return FALSE;
}


OchushaBulletinBoard *
ochusha_board_2ch_new(const gchar *name, const char *base_url)
{
  g_assert(name != NULL && base_url != NULL);

  return OCHUSHA_BULLETIN_BOARD(g_object_new(OCHUSHA_TYPE_BOARD_2CH,
					     "name", name,
					     "base_url", base_url,
					     NULL));
}


const char *
ochusha_board_2ch_get_read_cgi_url(OchushaBoard2ch *board_2ch)
{
  OchushaBoard2chClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board_2ch), NULL);

  if (board_2ch->read_cgi_url != NULL)
    return board_2ch->read_cgi_url;

  klass = OCHUSHA_BOARD_2CH_GET_CLASS(board_2ch);
  if (klass->get_read_cgi_url != NULL)
    board_2ch->read_cgi_url = (*klass->get_read_cgi_url)(board_2ch);
  else
    {
      OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(board_2ch);
      char url[PATH_MAX];

      switch (board->bbs_type)
	{
	case OCHUSHA_BBS_TYPE_2CH:
	case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
	case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
	  if (snprintf(url, PATH_MAX, "http://%s/test/read.cgi",
		       ochusha_bulletin_board_get_server(board)) < PATH_MAX)
	    board_2ch->read_cgi_url = G_STRDUP(url);
	  break;

	default:
	  return NULL;
	}

    }

  return board_2ch->read_cgi_url;
}


const char *
ochusha_board_2ch_get_cookie(OchushaBoard2ch *board)
{
  return board->cookie;
}


void
ochusha_board_2ch_set_cookie(OchushaBoard2ch *board, const char *cookie)
{
  if (board->cookie != NULL)
    G_FREE(board->cookie);
  if (cookie != NULL)
    board->cookie = G_STRDUP(cookie);
  else
    board->cookie = NULL;
}


static OchushaAsyncBuffer *
ochusha_board_2ch_get_threadlist_source(OchushaBulletinBoard *board,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode)
{
  OchushaBoard2ch *board_2ch;
  char url[PATH_MAX];

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board), NULL);

  if (snprintf(url, PATH_MAX, "%s" OCHUSHA_SUBJECT_TXT, board->base_url)
      >= PATH_MAX)
    return NULL;

  board_2ch = OCHUSHA_BOARD_2CH(board);
  return ochusha_network_broker_read_from_url(broker, buffer, url,
					      board_2ch->last_modified, mode);
}


static void
unmark_alive(gpointer data, gpointer user_data)
{
  OchushaThread2ch *thread = OCHUSHA_THREAD_2CH(data);
  thread->alive = FALSE;
}


typedef struct _CollectDroppedThreadArgs
{
  GSList *thread_list;
  EachThreadCallback *each_thread_cb;
  gpointer callback_data;
} CollectDroppedThreadArgs;


static void
collect_dropped_thread(gpointer data, gpointer user_data)
{
  CollectDroppedThreadArgs *args = (CollectDroppedThreadArgs *)user_data;
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(data);

  if (!thread_2ch->alive)
    {
      /* threadDATäݤ*/
      OchushaBBSThread *thread = (OchushaBBSThread *)thread_2ch;
      if (thread->number_of_responses_read > 0)
	{
	  if (g_slist_find(args->thread_list, thread) == NULL)
	    {
	      thread->flags |= OCHUSHA_BBS_THREAD_DAT_DROPPED;
	      thread->number_of_responses_on_server = 0;
	      args->thread_list = g_slist_append(args->thread_list, thread);
	      if (args->each_thread_cb != NULL)
		if (!(*args->each_thread_cb)(thread, args->callback_data))
		  args->each_thread_cb = FALSE;
#if 0	/* debug */
	      fprintf(stderr, "Resurrect DAT dropped thread: %s\n",
		      ochusha_bbs_thread_get_url(thread));
#endif
	    }
	}
      else
	thread->board->dropped_list
	  = g_slist_append(thread->board->dropped_list, thread);
    }
  else
    OCHU_OBJECT_UNREF(G_OBJECT(data));	/* Ƥ륹thread_list
					 * Ѥ;פrefƤ롣
					 */
}


static void
undo_thread_ref(gpointer data, gpointer user_data)
{
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(data);
  if (thread_2ch->alive)
    OCHU_OBJECT_UNREF(G_OBJECT(data));	/* Ƥ륹thread_list
					 * Ѥ;פrefƤ롣
					 */
}


static gboolean
advance_parsing(gpointer data)
{
  ochusha_async_buffer_broadcast((OchushaAsyncBuffer *)data);
  OCHU_OBJECT_UNREF((GObject *)data);
  return FALSE;
}


/*
 * refresh_threadlist_by_subject_txt:
 *
 * Ϳ줿ХåեƤ2chĤsubject.txtȸʤƲϤͿ줿
 * OchushaBulletinBoard¤Τ򹹿롣
 * ĸĤ٤˻ꤵ줿ХåؿƤ֡
 *
 * ХåؿFALSE֤硢λǲϤλ롣
 *
 * ϤˤTRUE֤ԤFALSE֤
 */
static gboolean
refresh_threadlist_by_subject_txt(OchushaBulletinBoard *board,
				  OchushaAsyncBuffer *buffer,
				  EachThreadCallback *each_thread_cb,
				  gpointer callback_data)
{
  char default_buffer[PATH_MAX];
  char scan_buffer[5];
  gboolean result = FALSE;
  GSList *thread_list = NULL;
  GSList *old_thread_list = board->thread_list;
  iconv_t converter;
  iconv_helper *converter_helper;
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_data(G_OBJECT(buffer),
			"OchushaNetworkBroker::BufferStatus");

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board) && status != NULL, FALSE);

  converter = iconv_open("UTF-8//IGNORE",
			 ochusha_board_2ch_get_subject_txt_encoding(board));
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      converter_helper = cp932_to_utf8_helper;
      break;

    default:
      converter_helper = NULL;
    }

  if (!ochusha_async_buffer_active_ref(buffer))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      iconv_close(converter);
      return FALSE;
    }

  g_slist_foreach(old_thread_list, unmark_alive, NULL);

  ochusha_async_buffer_lock(buffer);
  {
    unsigned int offset = 0;
    gboolean buffer_fixed = FALSE;

    while (TRUE)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	unsigned int rest_of_data = buffer->length - offset;
	char *eol_pos = NULL;
	int interval = ANALYSIS_INTERVAL;

	while (rest_of_data > 0
	       && interval-- > 0
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    char *thread_id;
	    gchar *thread_title;
	    OchushaBBSThread *thread;
	    OchushaThread2ch *thread_2ch;
	    char *title_pos;
	    int n_responses_on_server = 0;
	    char *tmp_pos;
	    int title_len;
	    char *ext;

	    tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");

	    if (tmp_pos == NULL)
	      {
#if DEBUG_LIBOCHUSHA
		fprintf(stderr, "Unknown format found in subject.txt.\n");
#endif
		goto prepare_next_line;
	      }
	    if ((tmp_pos - cur_pos) < PATH_MAX)
	      {
		thread_id = memcpy(default_buffer, cur_pos, tmp_pos - cur_pos);
		thread_id[tmp_pos - cur_pos] = '\0';
	      }
	    else
	      thread_id = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

	    if ((ext = strstr(thread_id, ".dat")) != NULL)
	      *ext = '\0';
	    title_pos = tmp_pos + 2;	/* skip "<>" */

	    title_len = eol_pos - title_pos;

	    for (tmp_pos = eol_pos - 1; tmp_pos > title_pos; tmp_pos--)
	      if (*tmp_pos == '(')
		break;

	    if (*tmp_pos == '(')
	      {
		char *close_paren = memchr(tmp_pos + 1, ')',
					   eol_pos - tmp_pos - 1);
		if (close_paren != NULL)
		  {
		    int tmp_len = close_paren - tmp_pos - 1;
		    if (tmp_len > 0 && tmp_len <= 4)
		      {
			memcpy(scan_buffer, tmp_pos + 1, tmp_len);
			scan_buffer[tmp_len] = '\0';
			sscanf(scan_buffer, "%d", &n_responses_on_server);
			title_len -= (tmp_len + 2);
		      }
		  }
	      }

	    thread = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								    thread_id);
	    if (thread == NULL)
	      {
		thread_title = simple_string_canon(title_pos, title_len,
						   converter,
						   converter_helper);
		thread = ochusha_bulletin_board_bbs_thread_new(board,
							       thread_id,
							       thread_title);
		G_FREE(thread_title);
	      }

	    if (thread_id != default_buffer)
	      G_FREE(thread_id);

	    thread_2ch = OCHUSHA_THREAD_2CH(thread);

	    if (!thread_2ch->alive)
	      {
		thread_2ch->alive = TRUE;

		thread->flags &= ~OCHUSHA_BBS_THREAD_DAT_DROPPED;
		thread->number_of_responses_on_server = n_responses_on_server;

		OCHU_OBJECT_REF(G_OBJECT(thread));
		thread_list = g_slist_append(thread_list, thread);

		if (each_thread_cb != NULL
		    && !(*each_thread_cb)(thread, callback_data))
		  {
		    break;
		  }
	      }

	  prepare_next_line:
	    offset = (eol_pos + 1) - buffer_top;
	    buffer_top = (char *)buffer->buffer;
	    cur_pos = buffer_top + offset;
	    rest_of_data = buffer->length - offset;
	    eol_pos = NULL;
	  }

	if (!buffer_fixed && buffer->fixed)
	  {
	    buffer_fixed = TRUE;
	    eol_pos = NULL;
	    continue;
	  }

	if (buffer->fixed && eol_pos == NULL && interval > 0)
	  {
	    result = TRUE;
	    goto terminated;
	  }

	if (interval > 0)
	  {
	    if (!ochusha_async_buffer_wait(buffer))
	      {
#if DEBUG_ASYNC_BUFFER_MOST
		fprintf(stderr, "buffer has been terminated.\n");
#endif
		goto terminated;
	      }
	  }
	else
	  {
	    if (ochusha_async_buffer_is_busy(buffer))
	      {
		/*
		 * buffer˥륹åɤsuspend/terminate
		 * ȿ뤿˥å
		 */
		if (!ochusha_async_buffer_wait(buffer))
		  goto terminated;
	      }
	    else
	      {
		/*
		 * GUIΥ쥹ݥ󥹤βܻؤ¸
		 */
		OCHU_OBJECT_REF(G_OBJECT(buffer));
		g_idle_add_full(G_PRIORITY_HIGH_IDLE + 15,
				advance_parsing, buffer, NULL);
		if (!ochusha_async_buffer_wait(buffer))
		  goto terminated;
	      }
	  }
	interval = ANALYSIS_INTERVAL;

	if (status->state == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    goto terminated;
	  }
      }
  }
 terminated:
  ochusha_async_buffer_unlock(buffer);

  if (result)
    {
      OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(board);

      if (status->last_modified != NULL)
	{
	  if (board_2ch->last_modified != NULL)
	    G_FREE(board_2ch->last_modified);
	  board_2ch->last_modified = G_STRDUP(status->last_modified);
	}

      if (status->date != NULL)
	{
	  if (board_2ch->date != NULL)
	    G_FREE(board_2ch->date);
	  board_2ch->date = G_STRDUP(status->date);
	}
    }

  ochusha_async_buffer_active_unref(buffer);

  if (result)
    {
      CollectDroppedThreadArgs args =
	{
	  thread_list,
	  each_thread_cb,
	  callback_data
	};

      g_slist_foreach(old_thread_list, collect_dropped_thread, &args);
      board->thread_list = args.thread_list;
      g_slist_free(old_thread_list);
    }
  else
    {
      /* 顼ϸŤΤͥ */
      g_slist_foreach(old_thread_list, undo_thread_ref, NULL);
      g_slist_free(thread_list);
    }

  iconv_close(converter);

  return result;
}


/*
 * ochusha_board_2ch_refresh_threadlist:
 *
 * Ϳ줿ХåեƤ2chĤsubject.txtȸʤƲϤͿ줿
 * OchushaBulletinBoard¤Τ򹹿롣
 * ĸĤ٤˻ꤵ줿ХåؿƤ֡
 *
 * ХåؿFALSE֤硢λǲϤλ롣
 *
 * ϤˤTRUE֤ԤFALSE֤
 */
static gboolean
ochusha_board_2ch_refresh_threadlist(OchushaBulletinBoard *board,
				     OchushaAsyncBuffer *buffer,
				     EachThreadCallback *each_thread_cb,
				     gpointer callback_data)
{
  return refresh_threadlist_by_subject_txt(board, buffer,
					   each_thread_cb, callback_data);
}
