/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.input.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.CutoffUtilBeanInterface;
import jp.mosp.time.bean.SubstituteRegistBeanInterface;
import jp.mosp.time.bean.WorkOnHolidayRequestRegistBeanInterface;
import jp.mosp.time.bean.WorkOnHolidayRequestSearchBeanInterface;
import jp.mosp.time.comparator.settings.WorkOnHolidayRequestSubstituteDate1Comparator;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.CutoffDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestListDtoInterface;
import jp.mosp.time.input.vo.WorkOnHolidayRequestVo;

/**
 * 休日出勤申請情報の確認と編集を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SHOW}
 * </li><li>
 * {@link #CMD_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>
 * {@link #CMD_RE_SHOW}
 * </li><li>
 * {@link #CMD_DRAFT}
 * </li><li>
 * {@link #CMD_APPLI}
 * </li><li>
 * {@link #CMD_WITHDRAWN}
 * </li><li>
 * {@link #CMD_SORT}
 * </li><li>
 * {@link #CMD_PAGE}
 * </li><li>
 * {@link #CMD_INSERT_MODE}
 * </li><li>
 * {@link #CMD_EDIT_MODE}
 * </li><li>
 * {@link #CMD_BATCH_UPDATE}
 * </li><li>
 * {@link #CMD_TRANSFER}
 * </li><li>
 * {@link #CMD_SET_TRANSFER_DAY}
 * </li></ul>
 */
public class WorkOnHolidayRequestAction extends TimeAction {
	
	/**
	 * 表示コマンド。<br>
	 * <br>
	 * 現在ログインしているユーザの休日出勤申請画面を表示する。<br>
	 */
	public static final String	CMD_SHOW				= "TM1600";
	
	/**
	 * 選択表示コマンド。<br>
	 * <br>
	 * 休日出勤申請画面を表示する。<br>
	 */
	public static final String	CMD_SELECT_SHOW			= "TM1601";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 検索欄に入力した情報を元に休日出勤申請情報の検索を行う。<br>
	 */
	public static final String	CMD_SEARCH				= "TM1602";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * 新たな下書情報作成や申請を行った際に検索結果一覧にそれらが反映されるよう再表示を行う。<br>
	 */
	public static final String	CMD_RE_SHOW				= "TM1603";
	
	/**
	 * 下書きマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を休日出勤情報テーブルに登録し、下書情報として保存する。<br>
	 */
	public static final String	CMD_DRAFT				= "TM1604";
	
	/**
	 * 申請コマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を休日出勤情報テーブルに登録し、休日出勤申請を行う。以降、このレコードは上長が差戻をしない限り編集不可となる。<br>
	 * 休日出勤の申請可能時間を超過している、申請年月日で申請不可な日付が選択されている、<br>
	 * 申請時間が0時間0分のまま、休日出勤理由の項目が入力されていない、<br>
	 * といった状態で申請を行おうとした場合は場合はエラーメッセージにて通知し、申請は実行されない。<br>
	 */
	public static final String	CMD_APPLI				= "TM1605";
	
	/**
	 * 日付選択コマンド。<br>
	 * <br>
	 * 場合に応じて勤務日か休日の日付のみをプルダウンにそれぞれ表示させる。<br>
	 * 年または月のプルダウンを更新することでその年月で勤務日となっている日または休日となっている日を抽出して画面上に表示する。<br>
	 */
	public static final String	CMD_SET_DATE_SELECT		= "TM1606";
	
	/**
	 * 取下コマンド。<br>
	 * <br>
	 * 下書状態または差戻状態で登録されていたレコードの取下を行う。<br>
	 * 取下後、対象の休日出勤申請情報は未申請状態へ戻る。<br>
	 */
	public static final String	CMD_WITHDRAWN			= "TM1607";
	
	/**
	 * ソートコマンド。<br>
	 * <br>
	 * それぞれのレコードの値を比較して一覧表示欄の各情報毎に並び替えを行う。<br>
	 * これが実行される度に並び替えが昇順・降順と交互に切り替わる。<br>
	 */
	public static final String	CMD_SORT				= "TM1608";
	
	/**
	 * ページ繰りコマンド。<br>
	 * <br>
	 * 検索処理を行った際に検索結果が100件を超えた場合に分割されるページ間の遷移を行う。<br>
	 */
	public static final String	CMD_PAGE				= "TM1609";
	
	/**
	 * 出勤日決定コマンド。<br>
	 * <br>
	 * 入力された出勤日と振替申請のプルダウンの内容を基に処理を行う。<br>
	 * 振替休暇申請のプルダウンで"非申請"を選択した場合は勤務予定時刻、振替日1と振替日2のそれぞれの日付と休暇の長さのプルダウンが読取専用となる。<br>
	 * "申請"を選択、出勤日が法定休日である場合は振替日1の日付プルダウンのみ編集可能で残りの項目は読取専用となる。<br>
	 * "申請"を選択、出勤日が所定休日である場合は"非申請"を選択した時に読取専用だった全ての項目が編集可能となる。<br>
	 */
	public static final String	CMD_SET_ACTIVATION_DATE	= "TM1690";
	
	/**
	 * 新規登録モード切替コマンド。<br>
	 * <br>
	 * 申請テーブルの各入力欄に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_INSERT_MODE			= "TM1691";
	
	/**
	 * 編集モード切替コマンド。<br>
	 * <br>
	 * 選択したレコードの内容を申請テーブルの各入力欄にそれぞれ表示させる。申請テーブルヘッダに新規登録モード切替リンクを表示させる。
	 * に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_EDIT_MODE			= "TM1692";
	
	/**
	 * 一括更新コマンド。<br>
	 * <br>
	 * 一覧にて選択チェックボックスにチェックが入っている下書状態のレコードの申請処理を繰り返し行う。<br>
	 * ひとつもチェックが入っていない状態で一括更新ボタンをクリックした場合はエラーメッセージにて通知し、処理を中断する。<br>
	 */
	public static final String	CMD_BATCH_UPDATE		= "TM1695";
	
	/**
	 * 画面遷移コマンド。<br>
	 * <br>
	 * 現在表示している画面から、ワークフロー番号をMosP処理情報に設定し、画面遷移する。<br>
	 */
	public static final String	CMD_TRANSFER			= "TM1696";
	
	/**
	 * 。<br>
	 * <br>
	 * <br>
	 * <br>
	 */
	public static final String	CMD_SET_TRANSFER_DAY	= "TM1698";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public WorkOnHolidayRequestAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new WorkOnHolidayRequestVo();
	}
	
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SHOW)) {
			// 表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_SELECT_SHOW)) {
			// 選択表示
			prepareVo(false, false);
			select();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索
			prepareVo();
			search();
		} else if (mospParams.getCommand().equals(CMD_RE_SHOW)) {
			// 再表示
			prepareVo(true, false);
			search();
		} else if (mospParams.getCommand().equals(CMD_DRAFT)) {
			// 下書き
			prepareVo();
			draft();
		} else if (mospParams.getCommand().equals(CMD_APPLI)) {
			// 申請
			prepareVo();
			appli();
		} else if (mospParams.getCommand().equals(CMD_SET_DATE_SELECT)) {
			// 日付選択コマンド
			prepareVo();
			setDateSelect();
		} else if (mospParams.getCommand().equals(CMD_WITHDRAWN)) {
			// 取下
			prepareVo();
			withdrawn();
		} else if (mospParams.getCommand().equals(CMD_SORT)) {
			// ソート 
			prepareVo();
			sort();
		} else if (mospParams.getCommand().equals(CMD_PAGE)) {
			// ページ繰り
			prepareVo();
			page();
		} else if (mospParams.getCommand().equals(CMD_SET_ACTIVATION_DATE)) {
			// 出勤日決定コマンド
			prepareVo();
			setActivationDate();
			getWorkPlanFlag();
		} else if (mospParams.getCommand().equals(CMD_INSERT_MODE)) {
			// 新規モード切替
			prepareVo();
			insertMode();
		} else if (mospParams.getCommand().equals(CMD_EDIT_MODE)) {
			// 編集モード切替
			prepareVo();
			editMode();
		} else if (mospParams.getCommand().equals(CMD_BATCH_UPDATE)) {
			// 一括更新
			prepareVo();
			batchUpdate();
		} else if (mospParams.getCommand().equals(CMD_TRANSFER)) {
			// 遷移
			prepareVo(true, false);
			transfer();
		} else if (mospParams.getCommand().equals(CMD_SET_TRANSFER_DAY)) {
			prepareVo();
			transferDay();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void show() throws MospException {
		// 個人ID取得(ログインユーザ情報から)
		String personalId = mospParams.getUser().getPersonalId();
		// 対象日取得(システム日付)
		Date targetDate = getSystemDate();
		// 人事情報をVOに設定
		setEmployeeInfo(personalId, targetDate);
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		// 新規登録モード設定
		insertMode();
		
		getWorkPlanFlag();
	}
	
	/**
	 * 選択表示処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void select() throws MospException {
		// リクエストから個人ID及び対象日を取得
		String personalId = getTargetPersonalId();
		// 対象日取得(システム日付)
		Date targetDate = getSystemDate();
		// 個人ID確認
		if (personalId == null || personalId.isEmpty()) {
			// ログインユーザの個人IDを取得
			personalId = mospParams.getUser().getPersonalId();
		}
		// 人事情報をVOに設定
		setEmployeeInfo(personalId, targetDate);
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		// 新規登録モード設定
		insertMode();
		
		getWorkPlanFlag();
	}
	
	/**
	 * プルダウン設定
	 * @throws MospException 例外発生時
	 */
	private void setPulldown() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// システム日付取得
		Date date = getTargetDate();
		if (date == null) {
			date = getSystemDate();
		}
		// 検索条件
		vo.setAryPltSearchSubstituteRange(mospParams.getProperties().getCodeArray(TimeConst.CODE_SUBSTITUTE1_RANGE,
				true));
		vo.setAryPltSearchState(mospParams.getProperties().getCodeArray(TimeConst.CODE_APPROVAL_STATE, true));
		vo.setAryPltSearchRequestYear(getYearArray(DateUtility.getYear(date)));
		vo.setAryPltSearchRequestMonth(getMonthArray(false));
		vo.setAryPltEditSubstitute1Range(mospParams.getProperties().getCodeArray(TimeConst.CODE_SUBSTITUTE1_RANGE,
				false));
		// 勤務予定時刻
		vo.setAryPltEditStartHour(getHourArray());
		// 分は15分単位
		vo.setAryPltEditStartMinute(getMinuteArray(15));
		vo.setAryPltEditEndHour(getHourArray());
		// 分は15分単位
		vo.setAryPltEditEndMinute(getMinuteArray(15));
		
		// 出勤日/振替申請
		if (!setEditRequestPullDown(date)) {
			return;
		}
		if (vo.getPltEditRequestDay().equals("")) {
			// デフォルトの出勤日が次月になるので取得しなおす
			Date newDate = DateUtility.addMonth(date, 1);
			if (!setEditRequestPullDown(newDate)) {
				return;
			}
			if (vo.getPltEditRequestDay().equals("")) {
				// 再取得してもダメならエラーとする。
				addRequestDayNoGoingWorkDayErrorMessage();
				return;
			}
			date = newDate;
		}
		
		// 振替日
		setEditSubstitutePullDown(date);
		if (vo.getAryPltEditSubstitute1Day().length == 0) {
			// デフォルトの振替日が次月になるので取得しなおす
			Date newDate = DateUtility.addMonth(date, 1);
			setEditSubstitutePullDown(newDate);
			if (vo.getAryPltEditSubstitute1Day().length == 0) {
				// 再取得してもダメならエラーとする。
				addRequestDayNoTransferDayErrorMessage();
				return;
			}
		}
		
		vo.setPltEditSubstitute1Range("1");
	}
	
	/**
	 * @throws MospException 例外処理が発生した場合 
	 */
	protected void search() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// 検索クラス取得
		WorkOnHolidayRequestSearchBeanInterface search = timeReference().workOnHolidayRequestSearch();
		// VOの値を検索クラスへ設定
		search.setPersonalId(vo.getPersonalId());
		search.setSubstitute(vo.getPltSearchSubstitute());
		search.setSubstituteRange("");
		search.setWorkflowStatus(vo.getPltSearchState());
		int year = Integer.parseInt(vo.getPltSearchRequestYear());
		if (!vo.getPltSearchRequestMonth().isEmpty()) {
			int month = Integer.parseInt(vo.getPltSearchRequestMonth());
			// 締日ユーティリティー取得
			CutoffUtilBeanInterface cutoffUtil = timeReference().cutoffUtil();
			// 締日情報取得
			CutoffDtoInterface cutoffDto = cutoffUtil.getCutoffForPersonalId(vo.getPersonalId(), year, month);
			// 締期間の開始及び最終日
			Date firstDate = cutoffUtil.getCutoffFirstDate(cutoffDto.getCutoffCode(), year, month);
			// 対象年月及び締日から締期間最終日を取得
			Date lastDate = cutoffUtil.getCutoffLastDate(cutoffDto.getCutoffCode(), year, month);
			// 締期間を検索範囲に設定
			search.setRequestStartDate(firstDate);
			search.setRequestEndDate(lastDate);
		} else {
			search.setRequestStartDate(DateUtility.getDate(year, 1, 1));
			search.setRequestEndDate(DateUtility.getDate(year, 12, getLastDayofMonth(year, 12)));
		}
		// 検索条件をもとに検索クラスからマスタリストを取得
		List<WorkOnHolidayRequestListDtoInterface> list = search.getSearchList();
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(WorkOnHolidayRequestSubstituteDate1Comparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
		// 検索結果確認
		if (list.isEmpty()) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
		}
	}
	
	/**
	* 下書処理を行う。<br>
	 * @throws MospException 例外発生時
	*/
	protected void draft() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// 登録クラス取得
		WorkOnHolidayRequestRegistBeanInterface regist = time().workOnHolidayRequestRegist();
		SubstituteRegistBeanInterface substituteRegist = time().substituteRegist();
		// DTOの準備
		WorkOnHolidayRequestDtoInterface dto = timeReference().workOnHolidayRequest().findForKey(vo.getRecordId());
		if (dto == null) {
			dto = regist.getInitDto();
		}
		// DTOに値を設定
		setWorkOnHolidayDtoFields(dto);
		// 妥当性チェック
		regist.validate(dto);
		regist.checkDraft(dto);
		boolean substituteFlag = vo.getPltEditSubstitute().equals(
				String.valueOf(TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_ON));
		// 振替休日DTOの準備
		SubstituteDtoInterface substituteDto = null;
		if (substituteFlag) {
			// 振り替える場合
			substituteDto = substituteRegist.getInitDto();
			// 振替休日DTOに値を設定
			setSubstituteDtoFields1(substituteDto);
			substituteDto.setWorkflow(dto.getWorkflow());
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// 振替休日1の妥当性チェック
			substituteRegist.validate(substituteDto);
			substituteRegist.checkRegist(substituteDto, null);
		}
		// 登録クラスの取得。
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		// ワークフローの設定
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		if (workflowDto == null) {
			workflowDto = workflowRegist.getInitDto();
			workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_WORK_HOLIDAY);
		}
		workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
		// 登録後ワークフローの取得
		workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestDate(),
				PlatformConst.WORKFLOW_TYPE_TIME);
		if (workflowDto != null) {
			// ワークフローコメント登録
			platform().workflowCommentRegist().addComment(
					workflowDto,
					mospParams.getUser().getPersonalId(),
					mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
							new String[]{ mospParams.getName("WorkPaper") }));
			// ワークフロー番号セット
			dto.setWorkflow(workflowDto.getWorkflow());
			if (substituteFlag) {
				// 振り替える場合
				substituteDto.setWorkflow(workflowDto.getWorkflow());
			}
			// 登録
			regist.regist(dto);
			substituteRegist.delete(workflowDto.getWorkflow());
			if (substituteFlag) {
				// 振り替える場合
				substituteRegist.insert(substituteDto);
			}
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 下書成功メッセージ設定
		addDraftMessage();
		// 検索
		search();
		// 履歴編集対象を取得
		setEditUpdateMode(dto.getRequestDate());
		// 編集モード設定
		vo.setModeCardEdit(TimeConst.MODE_APPLICATION_DRAFT);
	}
	
	/**
	 * 申請処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void appli() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// 登録クラス取得
		WorkOnHolidayRequestRegistBeanInterface regist = time().workOnHolidayRequestRegist();
		SubstituteRegistBeanInterface substituteRegist = time().substituteRegist();
		// DTOの準備
		WorkOnHolidayRequestDtoInterface dto = timeReference().workOnHolidayRequest().findForKey(vo.getRecordId());
		if (dto == null) {
			dto = regist.getInitDto();
		}
		// DTOに値を設定
		setWorkOnHolidayDtoFields(dto);
		// 申請の相関チェック
		regist.checkAppli(dto);
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// 振出・休出フラグ
		boolean substituteFlag = vo.getPltEditSubstitute().equals(
				String.valueOf(TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_ON));
		// 振替休日DTOの準備
		SubstituteDtoInterface substituteDto = null;
		if (substituteFlag) {
			// 振り替える場合
			substituteDto = substituteRegist.getInitDto();
			// 振替休日DTOに値を設定
			setSubstituteDtoFields1(substituteDto);
			substituteDto.setWorkflow(dto.getWorkflow());
			// 振替休日の相関チェック
			substituteRegist.checkRegist(substituteDto, null);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// 振替日に勤怠申請情報(下書)を削除
			time().attendanceRegist().delete(substituteDto.getPersonalId(), substituteDto.getSubstituteDate());
		}
		// 登録クラスの取得
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		// ワークフローの設定
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		if (workflowDto == null) {
			workflowDto = workflowRegist.getInitDto();
			workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_WORK_HOLIDAY);
		}
		// 承認者個人ID
		workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
		// 登録後ワークフローの取得
		workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestDate(),
				PlatformConst.WORKFLOW_TYPE_TIME, null);
		if (workflowDto != null) {
			// ワークフロー番号セット
			dto.setWorkflow(workflowDto.getWorkflow());
			if (substituteFlag) {
				// 振り替える場合
				substituteDto.setWorkflow(workflowDto.getWorkflow());
			}
			// 登録
			regist.regist(dto);
			substituteRegist.delete(workflowDto.getWorkflow());
			if (substituteFlag) {
				// 振り替える場合
				substituteRegist.insert(substituteDto);
			}
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 申請成功メッセージ設定
		addAppliMessage();
		// 登録結果確認
		if (!mospParams.hasErrorMessage()) {
			// 登録が成功した場合、初期状態に戻す。
			insertMode();
			search();
		}
	}
	
	/**
	* 日付選択処理を行う。<br>
	* @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	*/
	protected void setDateSelect() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		Date date = DateUtility.getDate(vo.getPltEditRequestYear(), vo.getPltEditRequestMonth(), "1");
		if (!setEditRequestPullDown(date)) {
			return;
		}
		if (vo.getPltEditRequestDay().equals("")) {
			// デフォルトの出勤日が次月になるので取得しなおす
			Date newDate = DateUtility.addMonth(date, 1);
			if (!setEditRequestPullDown(newDate)) {
				return;
			}
			if (vo.getPltEditRequestDay().equals("")) {
				// 再取得してもダメならエラーとする。
				addRequestDayNoGoingWorkDayErrorMessage();
				return;
			}
			date = newDate;
		}
		setEditSubstitutePullDown(date);
		if (vo.getAryPltEditSubstitute1Day().length == 0) {
			// デフォルトの振替日が次月になるので取得しなおす
			Date newDate = DateUtility.addMonth(date, 1);
			setEditSubstitutePullDown(newDate);
			if (vo.getAryPltEditSubstitute1Day().length == 0) {
				// 再取得してもダメならエラーとする。
				addRequestDayNoTransferDayErrorMessage();
				return;
			}
		}
	}
	
	/**
	* 取下処理を行う。<br>
	* @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	*/
	protected void withdrawn() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// DTOの準備
		WorkOnHolidayRequestDtoInterface dto = timeReference().workOnHolidayRequest().findForKey(vo.getRecordId());
		// 存在確認
		checkSelectedDataExist(dto);
		// 取下の相関チェック
		time().workOnHolidayRequestRegist().checkWithdrawn(dto);
		// ワークフロー取得
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		// 存在確認
		checkSelectedDataExist(workflowDto);
		// ワークフロー登録
		workflowDto = platform().workflowRegist().withdrawn(workflowDto);
		if (workflowDto != null) {
			// ワークフローコメント登録
			platform().workflowCommentRegist().addComment(
					workflowDto,
					mospParams.getUser().getPersonalId(),
					mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
							new String[]{ mospParams.getName("TakeDown") }));
		}
		// 取消時の勤怠データ削除
		time().attendanceRegist().delete(dto.getPersonalId(), dto.getRequestDate());
		// 削除結果確認
		if (mospParams.hasErrorMessage()) {
			// 削除失敗メッセージ設定
			addDeleteHistoryFailedMessage();
			return;
		}
		// コミット
		commit();
		// 取下成功メッセージ設定
		addTakeDownMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * 一覧のソート処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void sort() throws MospException {
		setVoList(sortList(getTransferredSortKey()));
	}
	
	/**
	 * 一覧のページ処理を行う。
	 * @throws MospException 例外発生時
	 */
	protected void page() throws MospException {
		setVoList(pageList());
	}
	
	/**
	 * 有効日(編集)設定処理を行う。<br>
	 * 保持有効日モードを確認し、モード及びプルダウンの再設定を行う。<br>
	 * @throws MospException プルダウンの取得に失敗した場合
	 */
	protected void setActivationDate() throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// 現在の有効日モードを確認
		if (vo.getModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
			// 変更モードの場合
			// 退職済み確認
			if (reference().retirement().isRetired(vo.getPersonalId(), getEditRequestDate())) {
				addEmployeeRetiredMessage();
				return;
			}
			// 休職期間確認
			if (reference().suspension().isSuspended(vo.getPersonalId(), getEditRequestDate())) {
				addEmployeeSuspendedMessage();
				return;
			}
			// 出勤日が法定休日の場合、振替休暇申請範囲1は"全休"固定とする
//			boolean substituteFlag = false;
//			List<SubstituteDtoInterface> list = timeReference().substitute().getSubstituteList(
//					getPersonalId(vo.getLblEmployeeCode()), date);
//			for (SubstituteDtoInterface substituteDto : list) {
//				WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(
//						substituteDto.getWorkflow());
//				if (workflowDto == null) {
//					continue;
//				}
//				if (PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
			// 承認されている場合
//					if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(substituteDto.getSubstituteType())) {
			// 法定休日の場合
//						vo.setJsModeLegalHoliday("on");
//						substituteFlag = true;
//						break;
//					} else if (TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(substituteDto.getSubstituteType())) {
			// 所定休日の場合
//						vo.setJsModeLegalHoliday("off");
//						substituteFlag = true;
//						break;
//					} else {
//						continue;
//					}
//				}
//			}
//			if (!substituteFlag) {
//				ApplicationDtoInterface applicationDto = timeReference().application().findForPerson(
//						getPersonalId(vo.getLblEmployeeCode()), date);
//				timeReference().application().chkExistApplication(applicationDto, date);
//				if (mospParams.hasErrorMessage()) {
//					return;
//				}
//				ScheduleDateDtoInterface scheduleDto = timeReference().scheduleDate().getScheduleDateInfo(
//						applicationDto.getScheduleCode(), date, date);
//				timeReference().scheduleDate().chkExistScheduleDate(scheduleDto, date);
//				if (mospParams.hasErrorMessage()) {
//					return;
//				}
//				if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(scheduleDto.getWorkTypeCode())) {
			// 法定休日の場合
//					vo.setJsModeLegalHoliday("on");
//				} else if (TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(scheduleDto.getWorkTypeCode())) {
			// 所定休日の場合
//					vo.setJsModeLegalHoliday("off");
//				} else {
//					mospParams.addErrorMessage("");
//					return;
//				}
//			}
			if (setApprvalPullDown(vo.getPersonalId(), getEditRequestDate()) == false) {
				return;
			}
			WorkOnHolidayRequestRegistBeanInterface regist = time().workOnHolidayRequestRegist();
			WorkOnHolidayRequestDtoInterface dto = timeReference().workOnHolidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			dto.setPersonalId(vo.getPersonalId());
			dto.setRequestDate(getEditRequestDate());
			dto.setWorkOnHolidayType(getScheduleWorkTypeCode(vo.getPersonalId(), getEditRequestDate()));
			dto.setSubstitute(Integer.parseInt(vo.getPltEditSubstitute()));
			if (mospParams.hasErrorMessage()) {
				return;
			}
			regist.checkTemporaryClosingFinal(dto);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			regist.checkWorkOnHolidayOverlap(dto);
			regist.checkHolidayDate(dto);
//			regist.checkRequest(dto);
			vo.setPltEditSubstitute1Range("1");
//			vo.setPltEditSubstitute2Range("1");
			if (mospParams.hasErrorMessage()) {
				return;
			}
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
			transferDay();
		} else {
			// 変更モード以外
			String[] aryPltLblApproverSetting = new String[0];
			vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		}
	}
	
	/**
	 * 新規登録モードに設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void insertMode() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 編集モード設定
		vo.setModeCardEdit(TimeConst.MODE_APPLICATION_NEW);
		// 有効日(編集)モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		// 勤務予定の表示
		vo.setJsModeWorkPlanFlag(TimeConst.MODE_WORK_PLAN_APPLICATION_OFF);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(WorkOnHolidayRequestSubstituteDate1Comparator.class.getName());
		// プルダウン設定
		setPulldown();
		// 基本情報チェック
		if (null != getEditRequestDate()) {
			timeReference().workOnHolidayRequest().chkBasicInfo(vo.getPersonalId(), getEditRequestDate());
		}
	}
	
	/**
	 * 履歴編集モードで画面を表示する。<br>
	 * 履歴編集対象は、遷移汎用コード及び有効日で取得する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void editMode() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// リクエストから個人ID及び対象日を取得
		String personalId = getTargetPersonalId();
		// 対象日取得(システム日付)
		Date targetDate = getSystemDate();
		if (personalId != null) {
			// 人事情報をVOに設定
			setEmployeeInfo(personalId, targetDate);
			// ページ繰り設定
			setPageInfo(CMD_PAGE, getListLength());
			// 新規登録モード設定
			insertMode();
			
			getWorkPlanFlag();
		}
		// 遷移汎用コード及び有効日から履歴編集対象を取得し編集モードを設定
		setEditUpdateMode(getDate(getTransferredActivateDate()));
		// 有効日モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		setActivationDate();
	}
	
	/**
	 * 一括更新処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void batchUpdate() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// 一括更新処理
		// ワークフローの更新のみ
		time().workOnHolidayRequestRegist().update(getIdArray(vo.getCkbSelect()));
		// 一括更新結果確認
		if (mospParams.hasErrorMessage()) {
			// 更新失敗メッセージ設定
			addBatchUpdateFailedMessage();
			return;
		}
		// コミット
		commit();
		// 更新成功メッセージ設定
		addUpdateMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * ワークフロー番号をMosP処理情報に設定し、
	 * 連続実行コマンドを設定する。<br>
	 */
	protected void transfer() {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// MosP処理情報に対象ワークフローを設定
		setTargetWorkflow(vo.getAryWorkflow(getTransferredIndex()));
		// 承認履歴画面へ遷移(連続実行コマンド設定)
		mospParams.setNextCommand(ApprovalHistoryAction.CMD_APPROVAL_HISTORY_HOLIDAY_WORK_SELECT_SHOW);
	}
	
	/**
	 * 一括更新処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void transferDay() throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// 同日付に時差出勤が申請されているか確認する。
		getDifferenceRequest1(vo.getPersonalId(), getDate(vo.getPltEditSubstitute1Year(), vo
			.getPltEditSubstitute1Month(), vo.getPltEditSubstitute1Day()));
//		getDifferenceRequest2(personalId, getDate(vo.getPltEditSubstitute2Year(), vo.getPltEditSubstitute2Month(), vo
//			.getPltEditSubstitute2Day()));
		// 日付プルダウンの再取得
		// 振替日1の取得
		Date date = DateUtility.getDate(vo.getPltEditSubstitute1Year(), vo.getPltEditSubstitute1Month(), vo
			.getPltEditSubstitute1Day());
		getEditSubstitutePullDown(date, vo.getPersonalId(), 1);
		// 振替日2の取得
//		date = DateUtility.getDate(vo.getPltEditSubstitute2Year(), vo.getPltEditSubstitute2Month(), vo
//			.getPltEditSubstitute2Day());
//		getEditSubstitutePullDown(date, personalId, 2);
	}
	
	/**
	 * 履歴編集モードを設定する。<br>
	 * 申請日で編集対象情報を取得する。<br>
	 * @param requestDate 申請日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setEditUpdateMode(Date requestDate) throws MospException {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// 履歴編集対象取得
		WorkOnHolidayRequestDtoInterface dto = timeReference().workOnHolidayRequest().findForKeyOnWorkflow(
				vo.getPersonalId(), requestDate);
		List<SubstituteDtoInterface> list = timeReference().substitute().getSubstituteList(dto.getWorkflow());
		// 存在確認
		checkSelectedDataExist(dto);
		checkSelectedDataExist(list);
		// VOにセット
		setVoFields(dto);
		setVoFields(list);
		setDateSelect();
		// 出勤日のための再取得
		setVoFields(dto);
		setVoFields(list);
		getWorkPlanFlag();
	}
	
	/**
	 * 初期値を設定する。<br>
	 */
	public void setDefaultValues() {
		// VO準備
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// システム日付取得
		Date date = DateUtility.getSystemDate();
		// 検索項目設定
		vo.setRecordId(0);
		vo.setPltEditRequestYear("");
		vo.setPltEditRequestMonth("");
		vo.setPltEditRequestDay("");
		vo.setPltEditSubstitute("");
		vo.setPltEditStartHour("");
		vo.setPltEditStartMinute("");
		vo.setPltEditEndHour("");
		vo.setPltEditEndMinute("");
		vo.setPltEditSubstitute1Year("");
		vo.setPltEditSubstitute1Month("");
		vo.setPltEditSubstitute1Day("");
		vo.setTxtEditRequestReason("");
		vo.setPltEditStartHour("0");
		vo.setPltEditStartMinute("0");
		vo.setPltEditEndHour("0");
		vo.setPltEditEndMinute("0");
		vo.setJsModeLegalHoliday("off");
		vo.setPltSearchSubstitute("");
		vo.setPltSearchSubstituteRange("");
		vo.setPltSearchState("");
		vo.setPltSearchRequestYear(DateUtility.getStringYear(date));
		vo.setPltSearchRequestMonth("");
		// 承認者欄の初期化
		String[] aryPltLblApproverSetting = new String[0];
		vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
		vo.setPltApproverSetting1("");
		vo.setPltApproverSetting2("");
		vo.setPltApproverSetting3("");
		vo.setPltApproverSetting4("");
		vo.setPltApproverSetting5("");
		vo.setPltApproverSetting6("");
		vo.setPltApproverSetting7("");
		vo.setPltApproverSetting8("");
		vo.setPltApproverSetting9("");
		vo.setPltApproverSetting10("");
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param list 対象リスト
	 * @throws MospException 例外発生時
	 */
	protected void setVoList(List<? extends BaseDtoInterface> list) throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// データ配列初期化
		String[] aryCkbRecordId = new String[list.size()];
		String[] aryLblRequestDate = new String[list.size()];
		String[] aryLblRequestTime = new String[list.size()];
		String[] aryLblRequestReason = new String[list.size()];
		String[] aryLblSubstitute1Date = new String[list.size()];
//		String[] aryLblSubstitute2Date = new String[list.size()];
		String[] aryLblWorkflowStatus = new String[list.size()];
		String[] aryLblApproverName = new String[list.size()];
		String[] aryLblOnOff = new String[list.size()];
		long[] aryWorkflow = new long[list.size()];
		// データ作成
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			WorkOnHolidayRequestListDtoInterface dto = (WorkOnHolidayRequestListDtoInterface)list.get(i);
			// 配列に情報を設定
			aryCkbRecordId[i] = String.valueOf(dto.getTmdWorkOnHolidayRequestId());
			aryLblRequestDate[i] = getStringDateAndDay(dto.getRequestDate());
			aryLblRequestTime[i] = getTimeWaveFormat(dto.getStartTime(), dto.getEndTime(), dto.getRequestDate());
			aryLblRequestReason[i] = dto.getRequestReason();
			StringBuffer substitute1 = new StringBuffer();
			substitute1.append(getStringDateAndDay(dto.getSubstituteDate1()));
//			substitute1.append(" ");
//			if (dto.getSubstituteRange1() == 1) {
//				substitute1.append(mospParams.getName("AllTime"));
//			} else if (dto.getSubstituteRange1() == 2) {
//				substitute1.append(mospParams.getName("FrontTime"));
//			} else if (dto.getSubstituteRange1() == 3) {
//				substitute1.append(mospParams.getName("BackTime"));
//			}
			aryLblSubstitute1Date[i] = substitute1.toString();
//			StringBuffer substitute2 = new StringBuffer();
//			substitute2.append(getStringDateAndDay(dto.getSubstituteDate2()));
//			substitute2.append(" ");
//			if (dto.getSubstituteRange2() == 2) {
//				substitute2.append(mospParams.getName("FrontTime"));
//			} else if (dto.getSubstituteRange2() == 3) {
//				substitute2.append(mospParams.getName("BackTime"));
//			}
//			aryLblSubstitute2Date[i] = substitute2.toString();
			aryLblWorkflowStatus[i] = getStatusStageValueView(dto.getState(), dto.getStage());
			aryLblOnOff[i] = getButtonOnOff(dto.getState());
			aryLblApproverName[i] = dto.getApproverName();
			aryWorkflow[i] = dto.getWorkflow();
		}
		// データをVOに設定
		vo.setAryCkbWorkOnHolidayRequestListId(aryCkbRecordId);
		vo.setAryLblWorkDate(aryLblRequestDate);
		vo.setAryLblRequestTime(aryLblRequestTime);
		vo.setAryLblRequestReason(aryLblRequestReason);
		vo.setAryLblSubstituteDate1(aryLblSubstitute1Date);
//		vo.setAryLblSubstituteDate2(aryLblSubstitute2Date);
		vo.setAryLblState(aryLblWorkflowStatus);
		vo.setAryLblApprover(aryLblApproverName);
		vo.setAryLblOnOff(aryLblOnOff);
		vo.setAryWorkflow(aryWorkflow);
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setWorkOnHolidayDtoFields(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		String workTypeCode = getScheduleWorkTypeCode(vo.getPersonalId(), getEditRequestDate());
		// 結果確認
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// VOの値をDTOに設定
		dto.setTmdWorkOnHolidayRequestId(vo.getRecordId());
		dto.setPersonalId(vo.getPersonalId());
		dto.setRequestDate(getEditRequestDate());
		dto.setTimesWork(1);
		dto.setWorkOnHolidayType(workTypeCode);
		dto.setRequestReason(vo.getTxtEditRequestReason());
		dto.setSubstitute(Integer.parseInt(vo.getPltEditSubstitute()));
		if (dto.getSubstitute() != 1) {
			Date date;
			int year = Integer.parseInt(vo.getPltEditRequestYear());
			int month = Integer.parseInt(vo.getPltEditRequestMonth());
			int day = Integer.parseInt(vo.getPltEditRequestDay());
			int startHour = Integer.parseInt(vo.getPltEditStartHour());
			int startMinute = Integer.parseInt(vo.getPltEditStartMinute());
			int endHour = Integer.parseInt(vo.getPltEditEndHour());
			int endMinute = Integer.parseInt(vo.getPltEditEndMinute());
			dto.setStartTime(DateUtility.getDateTime(year, month, day, startHour, startMinute));
			// 終了時刻が開始時刻より前の時間だったら日付を一日進める
			if (endHour < startHour) {
				date = DateUtility.getDate(year, month, day);
				date = DateUtility.addDay(date, 1);
				year = DateUtility.getYear(date);
				month = DateUtility.getMonth(date);
				day = DateUtility.getDay(date);
			} else if (endHour == startHour) {
				if (endMinute < startMinute) {
					date = DateUtility.getDate(year, month, day);
					date = DateUtility.addDay(date, 1);
					year = DateUtility.getYear(date);
					month = DateUtility.getMonth(date);
					day = DateUtility.getDay(date);
				}
			}
			dto.setEndTime(DateUtility.getDateTime(year, month, day, endHour, endMinute));
		}
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setSubstituteDtoFields1(SubstituteDtoInterface dto) throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		String workTypeCode = getScheduleWorkTypeCode(vo.getPersonalId(), getEditRequestDate());
		// 結果確認
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// VOの値をDTOに設定
		dto.setPersonalId(vo.getPersonalId());
		dto.setSubstituteDate(DateUtility.getDate(vo.getPltEditSubstitute1Year(), vo.getPltEditSubstitute1Month(), vo
			.getPltEditSubstitute1Day()));
		dto.setSubstituteType(workTypeCode);
		dto.setSubstituteRange(1);
		dto.setWorkDate(getEditRequestDate());
		dto.setTimesWork(1);
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setSubstituteDtoFields2(SubstituteDtoInterface dto) throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		String workTypeCode = getScheduleWorkTypeCode(vo.getPersonalId(), getEditRequestDate());
		// 結果確認
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// VOの値をDTOに設定
		dto.setPersonalId(vo.getPersonalId());
		dto.setSubstituteDate(DateUtility.getDate(vo.getPltEditSubstitute2Year(), vo.getPltEditSubstitute2Month(), vo
			.getPltEditSubstitute2Day()));
		dto.setSubstituteType(workTypeCode);
		dto.setSubstituteRange(1);
		dto.setWorkDate(getEditRequestDate());
		dto.setTimesWork(1);
	}
	
	/**
	 * @return 文字列となっているリクエスト年月日をDate型へ変換して返す。
	 * @throws MospException 例外発生時
	 */
	protected Date getEditRequestDate() throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		return getDate(vo.getPltEditRequestYear(), vo.getPltEditRequestMonth(), vo.getPltEditRequestDay());
	}
	
	/**
	 * 出勤日/振替申請のプルダウンを設定する。<br>
	 * @param date 日付
	 * @return 該当する設定適用がない場合はfalseを返す
	 * @throws MospException 例外発生時
	 */
	protected boolean setEditRequestPullDown(Date date) throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		int year = DateUtility.getYear(date);
		int month = DateUtility.getMonth(date);
		int day = DateUtility.getDay(date);
		// 出勤日/振替申請
		vo.setAryPltEditRequestYear(getYearArray(year));
		vo.setAryPltEditRequestMonth(getMonthArray(false));
		vo.setAryPltEditRequestDay(getDayArray(TimeConst.DATE_MONTH_MAX_DAY));
		vo.setPltEditRequestYear(String.valueOf(year));
		vo.setPltEditRequestMonth(String.valueOf(month));
		vo.setPltEditRequestDay(String.valueOf(day));
		return true;
	}
	
	/**
	 * 振替日のプルダウンを設定する。<br>
	 * @param date 日付
	 * @throws MospException 例外発生時
	 */
	protected void setEditSubstitutePullDown(Date date) throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		int year = DateUtility.getYear(date);
		int month = DateUtility.getMonth(date);
		int lastDay = getLastDayofMonth(year, month);
		// 振替日
		vo.setAryPltEditSubstitute1Year(getYearArray(year));
		vo.setAryPltEditSubstitute1Month(getMonthArray(false));
		// 振替日の取得
		String[][] aryGetSubstituteDay = getDayArray(lastDay);
//		String[][] arySubstituteDay = new String[aryGetSubstituteDay.length - vo.getAryPltEditRequestDay().length][2];
		String[][] aryHoliday = vo.getAryPltEditRequestDay();
		int loopHoliday = 0;
		int loop = 0;
		for (String[] element : aryGetSubstituteDay) {
			if (loopHoliday < aryHoliday.length) {
				if (element[0].equals(aryHoliday[loopHoliday][0])) {
					loopHoliday++;
				} else {
//					arySubstituteDay[loop][0] = element[0];
//					arySubstituteDay[loop][1] = element[1];
					loop++;
				}
			} else {
//				arySubstituteDay[loop][0] = element[0];
//				arySubstituteDay[loop][1] = element[1];
				loop++;
			}
		}
//		vo.setAryPltEditSubstitute1Day(arySubstituteDay);
		vo.setAryPltEditSubstitute1Day(getDayArray(TimeConst.DATE_MONTH_MAX_DAY));
		vo.setPltEditSubstitute1Year(String.valueOf(year));
		vo.setPltEditSubstitute1Month(String.valueOf(month));
//		vo.setPltEditSubstitute1Day(getDefaultDay(arySubstituteDay, DateUtility.getDay(DateUtility.getSystemDate())));
		vo.setPltEditSubstitute1Day(String.valueOf(DateUtility.getDay(date)));
//		vo.setAryPltEditSubstitute2Year(getYearArray(year));
//		vo.setAryPltEditSubstitute2Month(getMonthArray());
//		vo.setAryPltEditSubstitute2Day(arySubstituteDay);
//		vo.setPltEditSubstitute2Year(year);
//		vo.setPltEditSubstitute2Month(String.valueOf(month));
//		vo.setPltEditSubstitute2Day(getDefaultDay(arySubstituteDay, DateUtility.getDay(DateUtility.getSystemDate())));
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setVoFields(WorkOnHolidayRequestDtoInterface dto) throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// DTOの値をVOに設定
		vo.setRecordId(dto.getTmdWorkOnHolidayRequestId());
		vo.setPltEditRequestYear(DateUtility.getStringYear(dto.getRequestDate()));
		vo.setPltEditRequestMonth(String.valueOf(DateUtility.getMonth(dto.getRequestDate())));
		vo.setPltEditRequestDay(String.valueOf(DateUtility.getDay(dto.getRequestDate())));
		vo.setTxtEditRequestReason(dto.getRequestReason());
		vo.setPltEditSubstitute(String.valueOf(dto.getSubstitute()));
		if (!DateUtility.getStringHour(dto.getStartTime()).isEmpty()) {
			vo.setPltEditStartHour(String.valueOf(DateUtility.getHour(dto.getStartTime())));
			vo.setPltEditStartMinute(String.valueOf(DateUtility.getMinute(dto.getStartTime())));
			vo.setPltEditEndHour(String.valueOf(DateUtility.getHour(dto.getEndTime())));
			vo.setPltEditEndMinute(String.valueOf(DateUtility.getMinute(dto.getEndTime())));
		}
		vo.setModeCardEdit(getApplicationMode(dto.getWorkflow()));
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param list 対象リスト
	 */
	protected void setVoFields(List<SubstituteDtoInterface> list) {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// DTOの値をVOに設定
		int i = 0;
		for (SubstituteDtoInterface dto : list) {
			Date substituteDate = dto.getSubstituteDate();
			int substituteRange = dto.getSubstituteRange();
			if (i == 0) {
				vo.setPltEditSubstitute1Year(DateUtility.getStringYear(substituteDate));
				vo.setPltEditSubstitute1Month(String.valueOf(DateUtility.getMonth(substituteDate)));
				vo.setPltEditSubstitute1Day(String.valueOf(DateUtility.getDay(substituteDate)));
				vo.setPltEditSubstitute1Range(String.valueOf(substituteRange));
			} else if (i == 1) {
//				vo.setPltEditSubstitute2Year(DateUtility.getStringYear(substituteDate));
//				vo.setPltEditSubstitute2Month(String.valueOf(DateUtility.getMonth(substituteDate)));
//				vo.setPltEditSubstitute2Day(String.valueOf(DateUtility.getDay(substituteDate)));
//				vo.setPltEditSubstitute2Range(String.valueOf(substituteRange));
			}
			i++;
		}
	}
	
	private String getScheduleWorkTypeCode(String personalId, Date targetDate) throws MospException {
		List<SubstituteDtoInterface> list = timeReference().substitute().getSubstituteList(personalId, targetDate);
		for (SubstituteDtoInterface substituteDto : list) {
			WorkflowDtoInterface workflowDto = reference().workflow()
				.getLatestWorkflowInfo(substituteDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
				// 承認完了の場合
				return substituteDto.getSubstituteType();
			}
		}
		ApplicationDtoInterface applicationDto = timeReference().application().findForPerson(personalId, targetDate);
		timeReference().application().chkExistApplication(applicationDto, targetDate);
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		ScheduleDtoInterface scheduleDto = timeReference().schedule().getScheduleInfo(applicationDto.getScheduleCode(),
				getEditRequestDate());
		timeReference().schedule().chkExistSchedule(scheduleDto, targetDate);
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		ScheduleDateDtoInterface scheduleDateDto = timeReference().scheduleDate().findForKey(
				scheduleDto.getScheduleCode(), scheduleDto.getActivateDate(), getEditRequestDate());
		timeReference().scheduleDate().chkExistScheduleDate(scheduleDateDto, targetDate);
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		return scheduleDateDto.getWorkTypeCode();
	}
	
	/**
	 * 振替日のプルダウンを設定する。<br>
	 * @param date 日付
	 * @param personalId 個人ID
	 * @param flg 振替日付判定フラグ
	 * @throws MospException 例外発生時
	 */
	protected void getEditSubstitutePullDown(Date date, String personalId, int flg) throws MospException {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// 日付取得
		int year = Integer.parseInt(vo.getPltEditSubstitute1Year());
		int month = Integer.parseInt(vo.getPltEditSubstitute1Month());
		int lastDay = getLastDayofMonth(year, month);
		
		ApplicationDtoInterface appDto = timeReference().application().findForPerson(personalId, date);
		String[][] holidayDay = new String[0][2];
		if (appDto != null) {
			holidayDay = timeReference().workOnHolidayInfo().getSelectArray(personalId,
					DateUtility.getDate(year, month, 1), DateUtility.getDate(year, month, lastDay));
		} else {
			// 該当する設定適用が存在しない
			timeReference().application().chkExistApplication(appDto, date);
			return;
		}
		// 振替日の取得
		String[][] aryGetSubstituteDay = getDayArray(lastDay);
		String[][] arySubstituteDay = new String[aryGetSubstituteDay.length - holidayDay.length][2];
		String[][] aryHoliday = holidayDay;
		int loopHoliday = 0;
		int loop = 0;
		for (String[] element : aryGetSubstituteDay) {
			if (loopHoliday < aryHoliday.length) {
				if (element[0].equals(aryHoliday[loopHoliday][0])) {
					loopHoliday++;
				} else {
					arySubstituteDay[loop][0] = element[0];
					arySubstituteDay[loop][1] = element[1];
					loop++;
				}
			} else {
				arySubstituteDay[loop][0] = element[0];
				arySubstituteDay[loop][1] = element[1];
				loop++;
			}
		}
//		vo.setAryPltEditSubstitute1Day(arySubstituteDay);
		vo.setAryPltEditSubstitute1Day(getDayArray(TimeConst.DATE_MONTH_MAX_DAY));
//		if (flg == 1) {
//			vo.setAryPltEditSubstitute1Day(arySubstituteDay);
//		} else {
//			vo.setAryPltEditSubstitute2Day(arySubstituteDay);
//		}
	}
	
	/**
	 * 
	 */
	protected void getWorkPlanFlag() {
		// VO取得
		WorkOnHolidayRequestVo vo = (WorkOnHolidayRequestVo)mospParams.getVo();
		// voの振替申請の状態チェック
		if (vo.getPltEditSubstitute().equals("1")) {
			vo.setJsModeWorkPlanFlag(TimeConst.MODE_WORK_PLAN_APPLICATION_ON);
		} else if (vo.getPltEditSubstitute().equals("2")) {
			vo.setJsModeWorkPlanFlag(TimeConst.MODE_WORK_PLAN_APPLICATION_OFF);
		} else {
			vo.setJsModeWorkPlanFlag("");
		}
	}
	
	/**
	 * 出勤日が存在しない場合のメッセージを設定する。<br>
	 */
	protected void addRequestDayNoGoingWorkDayErrorMessage() {
		mospParams.addErrorMessage(PlatformMessageConst.MSG_NOT_EXIST_AND_CHANGE_SOMETHING, mospParams
			.getName("Approver"), mospParams.getName("Application") + mospParams.getName("Day"));
	}
	
	/**
	 * 振替日が存在しない場合のメッセージを設定する。<br>
	 */
	protected void addRequestDayNoTransferDayErrorMessage() {
		mospParams.addErrorMessage(PlatformMessageConst.MSG_NOT_EXIST_AND_CHANGE_SOMETHING, mospParams
			.getName("Approver"), mospParams.getName("Application") + mospParams.getName("Day"));
	}
	
	/**
	 * 勤務日名称を取得する。
	 * @return 勤務日名称
	 */
	protected String getNameWorkDate() {
		return mospParams.getName("Work") + mospParams.getName("Day");
	}
	
}
