/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.human.RetirementReferenceBeanInterface;
import jp.mosp.platform.bean.human.SuspensionReferenceBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dao.workflow.WorkflowDaoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.SubstituteReferenceBeanInterface;
import jp.mosp.time.bean.SubstituteRegistBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dao.settings.AttendanceDaoInterface;
import jp.mosp.time.dao.settings.DifferenceRequestDaoInterface;
import jp.mosp.time.dao.settings.HolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.OvertimeRequestDaoInterface;
import jp.mosp.time.dao.settings.ScheduleDaoInterface;
import jp.mosp.time.dao.settings.ScheduleDateDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.SubstituteDaoInterface;
import jp.mosp.time.dao.settings.TimeSettingDaoInterface;
import jp.mosp.time.dao.settings.WorkOnHolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdSubstituteDto;

/**
 * 振替休日データ登録クラス。
 */
public class SubstituteRegistBean extends PlatformBean implements SubstituteRegistBeanInterface {
	
	/**
	 * 振替休日データDAOクラス。<br>
	 */
	protected SubstituteDaoInterface			dao;
	/**
	 * 振替休日データ参照インターフェース。<br>
	 */
	protected SubstituteReferenceBeanInterface	substituteReference;
	/**
	 * 休出申請データDAOクラス。<br>
	 */
	protected WorkOnHolidayRequestDaoInterface	workOnHolidayRequestDao;
	
	/**
	 * 休暇申請データDAOクラス。<br>
	 */
	protected HolidayRequestDaoInterface		holidayRequestDao;
	
	/**
	 * 代休申請データDAOクラス。<br>
	 */
	protected SubHolidayRequestDaoInterface		subHolidayRequestDao;
	
	/**
	 * 残業申請データDAOクラス。<br>
	 */
	protected OvertimeRequestDaoInterface		overtimeRequestDao;
	
	/**
	 * 時差出勤申請データDAOクラス。<br>
	 */
	protected DifferenceRequestDaoInterface		differenceRequestDao;
	
	/**
	 * 勤怠データDAOクラス。<br>
	 */
	protected AttendanceDaoInterface			attendanceDao;
	
	/**
	 * ワークフローDAOクラス。<br>
	 */
	protected WorkflowDaoInterface				workflowDao;
	
	/**
	 * 勤怠設定管理DAOクラス。<br>
	 */
	protected TimeSettingDaoInterface			timeSettingDao;
	
	/**
	 * カレンダマスタDAOクラス。<br>
	 */
	protected ScheduleDaoInterface				scheduleDao;
	
	/**
	 * カレンダ日マスタDAOクラス。<br>
	 */
	protected ScheduleDateDaoInterface			scheduleDateDao;
	
	/**
	 * 設定適用管理参照クラス。<br>
	 */
	protected ApplicationReferenceBeanInterface	application;
	
	/**
	 * 人事求職情報参照クラス。
	 */
	protected SuspensionReferenceBeanInterface	suspensionReference;
	
	/**
	 * 人事退職情報参照クラス。
	 */
	protected RetirementReferenceBeanInterface	retirementReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public SubstituteRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosP処理情報
	 * @param connection DBコネクション
	 */
	public SubstituteRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		// DAO準備
		dao = (SubstituteDaoInterface)createDao(SubstituteDaoInterface.class);
		substituteReference = (SubstituteReferenceBeanInterface)createBean(SubstituteReferenceBeanInterface.class);
		workOnHolidayRequestDao = (WorkOnHolidayRequestDaoInterface)createDao(WorkOnHolidayRequestDaoInterface.class);
		holidayRequestDao = (HolidayRequestDaoInterface)createDao(HolidayRequestDaoInterface.class);
		subHolidayRequestDao = (SubHolidayRequestDaoInterface)createDao(SubHolidayRequestDaoInterface.class);
		overtimeRequestDao = (OvertimeRequestDaoInterface)createDao(OvertimeRequestDaoInterface.class);
		differenceRequestDao = (DifferenceRequestDaoInterface)createDao(DifferenceRequestDaoInterface.class);
		attendanceDao = (AttendanceDaoInterface)createDao(AttendanceDaoInterface.class);
		workflowDao = (WorkflowDaoInterface)createDao(WorkflowDaoInterface.class);
		timeSettingDao = (TimeSettingDaoInterface)createDao(TimeSettingDaoInterface.class);
		scheduleDao = (ScheduleDaoInterface)createDao(ScheduleDaoInterface.class);
		scheduleDateDao = (ScheduleDateDaoInterface)createDao(ScheduleDateDaoInterface.class);
		application = (ApplicationReferenceBeanInterface)createBean(ApplicationReferenceBeanInterface.class);
		suspensionReference = (SuspensionReferenceBeanInterface)createBean(SuspensionReferenceBeanInterface.class);
		retirementReference = (RetirementReferenceBeanInterface)createBean(RetirementReferenceBeanInterface.class);
	}
	
	@Override
	public SubstituteDtoInterface getInitDto() {
		return new TmdSubstituteDto();
	}
	
	@Override
	public void insert(SubstituteDtoInterface dto) throws MospException {
		// DTO妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 新規登録情報の検証
		checkInsert(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdSubstituteId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void delete(long workflow) throws MospException {
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		List<SubstituteDtoInterface> list = dao.findForWorkflow(workflow);
		for (SubstituteDtoInterface dto : list) {
			// DTO妥当性確認
			validate(dto);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			// 確認
			checkDelete(dto);
			if (mospParams.hasErrorMessage()) {
				// エラーが存在したら履歴削除処理をしない
				continue;
			}
			// 論理削除
			logicalDelete(dao, dto.getTmdSubstituteId());
		}
		// ロック解除
		unLockTable();
	}
	
	/**
	 * 新規登録時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkInsert(SubstituteDtoInterface dto) throws MospException {
		// 対象レコードが重複していないかを確認
		checkDuplicateInsert(dao.findForKeyOnWorkflow(dto.getPersonalId(), dto.getSubstituteDate(), dto
			.getSubstituteRange(), dto.getWorkDate(), dto.getTimesWork()));
	}
	
	/**
	 * 削除時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkDelete(SubstituteDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdSubstituteId());
	}
	
	@Override
	public void validate(SubstituteDtoInterface dto) throws MospException {
		// 基本情報のチェック
		substituteReference.chkBasicInfo(dto.getPersonalId(), dto.getSubstituteDate());
	}
	
	@Override
	public void checkRegist(SubstituteDtoInterface dto1, SubstituteDtoInterface dto2) throws MospException {
		// 振替日1の休職チェック
		checkSuspension(dto1);
		// 振替日2の休職チェック
		checkSuspension(dto2);
		// 振替日1の退職チェック
		checkRetirement(dto1);
		// 振替日2の退職チェック
		checkRetirement(dto2);
		// 振替日1の妥当性チェック
		checkTargetTransferDate(dto1);
		// 振替日2の妥当性チェック
		checkTargetTransferDate(dto2);
		// 振替日1の休日チェック
		checkHolidayDate(dto1);
		// 振替日2の休日チェック
		checkHolidayDate(dto2);
		// 振替日1と振替日2の同じ日付チェック
		checkTransferDateSameDate(dto1, dto2);
		// 振替日1の範囲チェック
		checkTransferDateRange(dto1);
		// 振替日2の範囲チェック
		checkTransferDateRange(dto2);
		// 振替日1の他の申請チェック
		checkRequest(dto1);
		// 振替日2の他の申請チェック
		checkRequest(dto2);
		// 振替日1の勤怠の申請チェック
		checkAttendance(dto1);
		// 振替日2の勤怠の申請チェック
		checkAttendance(dto2);
		// 振替日1と振替日2の半休時の組み合わせチェック
		checkTransferDateCombination(dto1, dto2);
	}
	
	@Override
	public void checkTargetTransferDate(SubstituteDtoInterface dto) throws MospException {
		if (dto == null) {
			return;
		}
		String personalId = dto.getPersonalId();
		Date substituteDate = dto.getSubstituteDate();
		int substituteRange = dto.getSubstituteRange();
		boolean substituteFlag = false;
		List<SubstituteDtoInterface> substituteList = dao.findForList(personalId, substituteDate);
		for (SubstituteDtoInterface substituteDto : substituteList) {
			if (dto.getWorkflow() == substituteDto.getWorkflow()) {
				continue;
			}
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(substituteDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書でなく且つ取下でない場合
				substituteFlag = true;
				if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(substituteDto.getSubstituteType())
						|| TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(substituteDto.getSubstituteType())) {
					String[] aryRep = { DateUtility.getStringDate(dto.getSubstituteDate()),
						mospParams.getName("GoingWork") + mospParams.getName("Day"),
						mospParams.getName("Transfer") + mospParams.getName("Day") };
					mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_3, aryRep);
					return;
				}
				break;
			}
		}
		if (!substituteFlag) {
			ApplicationDtoInterface applicationDto = application.findForPerson(personalId, substituteDate);
			if (applicationDto == null) {
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Set")
						+ mospParams.getName("Apply"), null);
				return;
			}
			ScheduleDtoInterface scheduleDto = scheduleDao
				.findForInfo(applicationDto.getScheduleCode(), substituteDate);
			if (scheduleDto == null) {
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Calendar"), null);
				return;
			}
			ScheduleDateDtoInterface scheduleDateDto = scheduleDateDao.findForKey(scheduleDto.getScheduleCode(),
					scheduleDto.getActivateDate(), substituteDate);
			if (scheduleDateDto == null) {
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Calendar")
						+ mospParams.getName("Data"), null);
				return;
			}
			if (scheduleDateDto.getWorkTypeCode() == null || scheduleDateDto.getWorkTypeCode().isEmpty()
					|| TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())
					|| TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())) {
				// 法定休日、所定休日又は勤務形態が登録されていない場合
				String[] aryRep = { DateUtility.getStringDate(dto.getSubstituteDate()),
					mospParams.getName("GoingWork") + mospParams.getName("Day"),
					mospParams.getName("Transfer") + mospParams.getName("Day") };
				mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_3, aryRep);
				return;
			}
		}
		// 振替休日のチェック
		for (SubstituteDtoInterface substituteDto : substituteList) {
			if (dto.getWorkflow() == substituteDto.getWorkflow()) {
				continue;
			}
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(substituteDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					|| PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書又は取下の場合
				continue;
			}
			int range = substituteDto.getSubstituteRange();
			if (substituteRange == 1 || range == 1) {
				// 全休の場合
				mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_9, new String[]{
					DateUtility.getStringDate(dto.getSubstituteDate()),
					mospParams.getName("Transfer") + mospParams.getName("Day") });
				return;
			} else if (range == 2 || range == 3) {
				// 午前休又は午後休の場合
				if (range == substituteRange) {
					// 範囲が同じ場合
					String[] aryRep = { DateUtility.getStringDate(dto.getSubstituteDate()),
						mospParams.getName("GoingWork") + mospParams.getName("Day"),
						mospParams.getName("Transfer") + mospParams.getName("Day") };
					mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_3, aryRep);
					return;
				}
			}
		}
		// 休暇申請のチェック
		List<HolidayRequestDtoInterface> holidayRequestList = holidayRequestDao.findForList(personalId, substituteDate);
		for (HolidayRequestDtoInterface holidayRequestDto : holidayRequestList) {
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(holidayRequestDto.getWorkflow());
			if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					|| PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書又は取下の場合
				continue;
			}
			int holidayRange = holidayRequestDto.getHolidayRange();
			if (substituteRange == 1 || holidayRange == 1) {
				// 全休の場合
				mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_9, new String[]{
					DateUtility.getStringDate(dto.getSubstituteDate()),
					mospParams.getName("Transfer") + mospParams.getName("Day") });
				return;
			} else if (holidayRange == 2 || holidayRange == 3) {
				// 午前休又は午後休の場合
				if (holidayRange == substituteRange) {
					// 範囲が同じ場合
					String[] aryRep = { DateUtility.getStringDate(dto.getSubstituteDate()),
						mospParams.getName("GoingWork") + mospParams.getName("Day"),
						mospParams.getName("Transfer") + mospParams.getName("Day") };
					mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_3, aryRep);
					return;
				}
			} else if (holidayRange == 4) {
				// 時休の場合
				
			}
		}
		// 代休申請のチェック
		List<SubHolidayRequestDtoInterface> subHolidayRequestList = subHolidayRequestDao.findForList(personalId,
				substituteDate);
		for (SubHolidayRequestDtoInterface subHolidayRequestDto : subHolidayRequestList) {
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(subHolidayRequestDto.getWorkflow());
			if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					|| PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書又は取下の場合
				continue;
			}
			int holidayRange = subHolidayRequestDto.getHolidayRange();
			if (substituteRange == 1 || holidayRange == 1) {
				// 全休の場合
				mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_9, new String[]{
					DateUtility.getStringDate(dto.getSubstituteDate()),
					mospParams.getName("Transfer") + mospParams.getName("Day") });
				return;
			} else if (holidayRange == 2 || holidayRange == 3) {
				// 午前休又は午後休の場合
				if (holidayRange == substituteRange) {
					// 範囲が同じ場合
					String[] aryRep = { DateUtility.getStringDate(dto.getSubstituteDate()),
						mospParams.getName("GoingWork") + mospParams.getName("Day"),
						mospParams.getName("Transfer") + mospParams.getName("Day") };
					mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_3, aryRep);
					return;
				}
			}
		}
	}
	
	@Override
	public void checkTransferDateSameDate(SubstituteDtoInterface dto1, SubstituteDtoInterface dto2) {
		if (dto1 == null || dto2 == null) {
			return;
		}
		if (dto1.getSubstituteDate() == null) {
			// TODO TSUJI エラーメッセージの表示。表示内容については辻さんが考えてくれる
			mospParams.addErrorMessage("");
			return;
		}
		if (dto2.getSubstituteDate() == null) {
			// TODO TSUJI エラーメッセージの表示。表示内容については辻さんが考えてくれる
			mospParams.addErrorMessage("");
			return;
		}
		// 振替日1と振替日2に同じ日付を入力していると登録不可。
		if (dto1.getSubstituteDate().compareTo(dto2.getSubstituteDate()) == 0) {
			String errorMes1 = mospParams.getName("Transfer") + mospParams.getName("Day");
			String errorMes2 = mospParams.getName("Transfer") + mospParams.getName("Day")
					+ mospParams.getName("Transfer") + mospParams.getName("No1") + mospParams.getName("Or")
					+ mospParams.getName("Day") + mospParams.getName("No2");
			mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_4, errorMes1, errorMes2);
		}
	}
	
	@Override
	public void checkTransferDateRange(SubstituteDtoInterface dto) throws MospException {
		if (dto == null) {
			return;
		}
		// 年月日のエラーメッセージ
		String errorMes1 = DateUtility.getStringDate(dto.getSubstituteDate());
		Date workDate = dto.getWorkDate();
		// 設定適用マスタDTOの取得
		ApplicationDtoInterface applicationDto = application.findForPerson(dto.getPersonalId(), workDate);
		if (applicationDto == null) {
			String errorMes2 = mospParams.getName("Set") + mospParams.getName("Apply")
					+ mospParams.getName("Information");
			mospParams.addErrorMessage(TimeMessageConst.MSG_SETTING_APPLICATION_DEFECT, errorMes1, errorMes2);
			return;
		}
		// 勤怠設定管理DTOの取得
		TimeSettingDtoInterface timeSettingDto = timeSettingDao.findForInfo(applicationDto.getWorkSettingCode(),
				workDate);
		if (timeSettingDto == null) {
			String errorMes2 = mospParams.getName("WorkManage") + mospParams.getName("Set")
					+ mospParams.getName("Information");
			mospParams.addErrorMessage(TimeMessageConst.MSG_SETTING_APPLICATION_DEFECT, errorMes1, errorMes2);
			return;
		}
		Date beforeDate = DateUtility.addDay(DateUtility.addMonth(workDate, -timeSettingDto
			.getTransferAheadLimitMonth()), -timeSettingDto.getTransferAheadLimitDate());
		Date afterDate = DateUtility.addDay(
				DateUtility.addMonth(workDate, timeSettingDto.getTransferLaterLimitMonth()), timeSettingDto
					.getTransferLaterLimitDate());
		Date substituteDate = dto.getSubstituteDate();
		if (!substituteDate.after(beforeDate) || !substituteDate.before(afterDate)) {
			mospParams.addErrorMessage(TimeMessageConst.MSG_TRANSFER_DAY_EXCESS, errorMes1, null);
		}
	}
	
	@Override
	public void checkRequest(SubstituteDtoInterface dto) throws MospException {
		if (dto == null) {
			return;
		}
		String personalId = dto.getPersonalId();
		Date substituteDate = dto.getSubstituteDate();
		// 振替日の休出申請取得
		WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = workOnHolidayRequestDao.findForKeyOnWorkflow(
				personalId, substituteDate);
		if (workOnHolidayRequestDto != null) {
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(workOnHolidayRequestDto.getWorkflow());
			if (workflowDto != null) {
				if (workflowDto.getWorkflowStage() != 0
						&& !PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					String[] aryRep = { DateUtility.getStringDate(dto.getSubstituteDate()),
						mospParams.getName("Other") + mospParams.getName("Of"),
						mospParams.getName("Transfer") + mospParams.getName("Day") };
					mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_6, aryRep);
				}
			}
		}
		if (dto.getSubstituteRange() == 1) {
			// 全休の場合
			// 振替日の残業申請取得
			List<OvertimeRequestDtoInterface> list = overtimeRequestDao.findForList(personalId, substituteDate);
			for (OvertimeRequestDtoInterface requestDto : list) {
				// ワークフローの取得
				WorkflowDtoInterface workflowDto = workflowDao.findForKey(requestDto.getWorkflow());
				if (workflowDto == null) {
					continue;
				}
				if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 下書でなく且つ取下でない場合
					String[] aryRep = { DateUtility.getStringDate(dto.getSubstituteDate()),
						mospParams.getName("Other") + mospParams.getName("Of"),
						mospParams.getName("Transfer") + mospParams.getName("Day") };
					mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_6, aryRep);
					return;
				}
			}
			// 振替日の時差出勤申請取得
			DifferenceRequestDtoInterface differenceRequestDto = differenceRequestDao.findForKeyOnWorkflow(personalId,
					substituteDate);
			if (differenceRequestDto == null) {
				return;
			}
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(differenceRequestDto.getWorkflow());
			if (workflowDto == null) {
				return;
			}
			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書でなく且つ取下でない場合
				String[] aryRep = { DateUtility.getStringDate(dto.getSubstituteDate()),
					mospParams.getName("Other") + mospParams.getName("Of"),
					mospParams.getName("Transfer") + mospParams.getName("Day") };
				mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_6, aryRep);
			}
		}
	}
	
	@Override
	public void checkTransferDateCombination(SubstituteDtoInterface dto1, SubstituteDtoInterface dto2)
			throws MospException {
		if (dto1 == null || dto2 == null) {
			return;
		}
		Date workDate = dto1.getWorkDate();
		// 設定適用マスタDTOの取得
		ApplicationDtoInterface applicationDto = application.findForPerson(dto1.getPersonalId(), workDate);
		if (applicationDto == null) {
			// 該当する設定適用が存在しない
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Set")
					+ mospParams.getName("Apply"), null);
			return;
		}
		// 勤怠設定管理DTOの取得
		TimeSettingDtoInterface timeSettingDto = timeSettingDao.findForInfo(applicationDto.getWorkSettingCode(),
				workDate);
		if (timeSettingDto == null) {
			// 該当する勤怠設定が存在しない
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("WorkManage")
					+ mospParams.getName("Set"), null);
			return;
		}
		if (timeSettingDto.getTransferExchange() == 2) {
			// 前半休と後半休を入れ替えて取得することは不可とする場合
			if (dto1.getSubstituteRange() == dto2.getSubstituteRange()) {
				// 振替日1と振替日2の範囲が同じである場合
				mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_7, null);
			}
		}
	}
	
	@Override
	public void checkAttendance(SubstituteDtoInterface dto) throws MospException {
		if (dto == null) {
			return;
		}
		AttendanceDtoInterface attendanceDto = attendanceDao
			.findForKey(dto.getPersonalId(), dto.getSubstituteDate(), 1);
		if (attendanceDto == null) {
			return;
		}
		WorkflowDtoInterface workflowDto = workflowDao.findForKey(attendanceDto.getWorkflow());
		if (workflowDto == null) {
			return;
		}
		if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
				|| PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
			// 下書又は取下の場合
			return;
		}
		if (workflowDto.getWorkflowStage() == 0
				&& PlatformConst.CODE_STATUS_REVERT.equals(workflowDto.getWorkflowStatus())) {
			// 1次戻の場合
			return;
		}
		String[] aryRep = { getStringDate(dto.getSubstituteDate()), mospParams.getName("WorkManage"),
			mospParams.getName("Transfer") + mospParams.getName("Day") };
		mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_1, aryRep);
	}
	
	@Override
	public void checkHolidayDate(SubstituteDtoInterface dto) {
		if (dto == null) {
			return;
		}
		// 法定休日でなく且つ所定休日でない場合
		if (!TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(dto.getSubstituteType())
				&& !TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(dto.getSubstituteType())) {
			String[] aryRep = { DateUtility.getStringDate(dto.getSubstituteDate()), mospParams.getName("Holiday"),
				mospParams.getName("GoingWork") + mospParams.getName("Day") };
			mospParams.addErrorMessage(TimeMessageConst.MSG_REQUEST_CHECK_3, aryRep);
		}
	}
	
	@Override
	public void checkSuspension(SubstituteDtoInterface dto) throws MospException {
		if (dto == null) {
			return;
		}
		if (suspensionReference.isSuspended(dto.getPersonalId(), dto.getSubstituteDate())) {
			addEmployeeSuspendedMessage();
		}
	}
	
	@Override
	public void checkRetirement(SubstituteDtoInterface dto) throws MospException {
		if (dto == null) {
			return;
		}
		if (retirementReference.isRetired(dto.getPersonalId(), dto.getSubstituteDate())) {
			addEmployeeRetiredMessage();
		}
	}
	
}
