/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean;

import java.util.Date;
import java.util.List;
import java.util.Map;

import jp.mosp.framework.base.MospException;
import jp.mosp.time.dto.settings.PaidHolidayDataDtoInterface;

/**
 * 有給休暇情報参照インターフェース。
 */
public interface PaidHolidayInfoReferenceBeanInterface {
	
	/**
	 * 有給休暇データ集計リストを取得する。<br><br>
	 * @param personalId 個人ID
	 * @param targetDate 対象年月日
	 * @return 有給休暇情報
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<PaidHolidayDataDtoInterface> getPaidHolidayCalcInfo(String personalId, Date targetDate) throws MospException;
	
	/**
	 * 有給休暇情報を取得する。<br><br>
	 * @param personalId 個人ID
	 * @param targetDate 対象年月日
	 * @return 有給休暇情報
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Map<String, Object> getPaidHolidayInfo(String personalId, Date targetDate) throws MospException;
	
	/**
	 * 有給休暇申請可能数を取得する。<br><br>
	 * @param personalId 個人ID
	 * @param targetDate 対象年月日
	 * @return 有給休暇申請可能数
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Map<String, Object> getPaidHolidayPossibleRequest(String personalId, Date targetDate) throws MospException;
	
	/**
	 * 有給休暇次回付与予定日を取得する。<br><br>
	 * @param personalId 個人ID
	 * @return 有給休暇次回付与日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Date getNextGivingDate(String personalId) throws MospException;
	
	/**
	 * 有給休暇次回付与予定日数を取得する。<br><br>
	 * @param personalId 個人ID
	 * @return 有給休暇次回付与予定日数
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	int getNextGivingDay(String personalId) throws MospException;
	
	/**
	 * 有給休暇次回付与予定日を取得する。<br><br>
	 * @param personalId 個人ID
	 * @return 有給休暇次回付与日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Date getNextManualGivingDate(String personalId) throws MospException;
	
	/**
	 * 有給休暇次回付与予定日数を取得する。<br><br>
	 * @param personalId 個人ID
	 * @return 有給休暇次回付与予定日数
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String getNextManualGivingDaysAndHours(String personalId) throws MospException;
	
	/**
	 * 有給休暇データ次月リストを取得する。<br><br>
	 * @param personalId 個人ID
	 * @param cutoffDate 締日
	 * @param calculationYear 集計年
	 * @param calculationMonth 集計月
	 * @param list 有給休暇データリスト
	 * @return 有給休暇データ次月リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<PaidHolidayDataDtoInterface> getPaidHolidayNextMonthInfo(String personalId, Date cutoffDate,
			int calculationYear, int calculationMonth, List<PaidHolidayDataDtoInterface> list) throws MospException;
	
	/**
	 * 有給休暇期限切れ日数を取得する。<br><br>
	 * @param list 有給休暇データリスト
	 * @param cutoffDate 締日
	 * @return 有給休暇期限切れ日数
	 */
	double getExpirationDay(List<PaidHolidayDataDtoInterface> list, Date cutoffDate);
	
	/**
	 * 有給休暇新規データを取得する。<br><br>
	 * @param personalId 個人ID
	 * @param cutoffDate 締日
	 * @param calculationYear 集計年
	 * @param calculationMonth 集計月
	 * @return 有給休暇仮付与データ
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	PaidHolidayDataDtoInterface getNewPaidHolidayInfo(String personalId, Date cutoffDate, int calculationYear,
			int calculationMonth) throws MospException;
	
}
