/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.platform.bean.system.impl;

import java.sql.Connection;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.SeUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.portal.PasswordCheckBeanInterface;
import jp.mosp.platform.bean.system.UserPasswordRegistBeanInterface;
import jp.mosp.platform.dao.system.UserPasswordDaoInterface;
import jp.mosp.platform.dto.system.UserPasswordDtoInterface;
import jp.mosp.platform.dto.system.impl.PfaUserPasswordDto;

/**
 * ユーザパスワード情報登録クラス。
 */
public class UserPasswordRegistBean extends PlatformBean implements UserPasswordRegistBeanInterface {
	
	/**
	 * ユーザパスワード情報DAOクラス。<br>
	 */
	UserPasswordDaoInterface	dao;
	

	/**
	 * コンストラクタ。
	 */
	public UserPasswordRegistBean() {
	}
	
	/**
	 * コンストラクタ。
	 * @param mospParams MosPパラメータクラス。
	 * @param connection DBコネクション
	 */
	public UserPasswordRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		// DAO準備
		dao = (UserPasswordDaoInterface)createDao(UserPasswordDaoInterface.class);
	}
	
	@Override
	public UserPasswordDtoInterface getInitDto() {
		return new PfaUserPasswordDto();
	}
	
	@Override
	public void regist(UserPasswordDtoInterface dto) throws MospException {
		// 対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 現在のユーザパスワード情報取得
		UserPasswordDtoInterface currentDto = dao.findForInfo(dto.getUserId());
		// 存在確認
		if (currentDto != null) {
			// 論理削除
			delete(currentDto);
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setPfaUserPasswordId(findForMaxId(dao) + 1);
		// 登録
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void delete(UserPasswordDtoInterface dto) throws MospException {
		// 論理削除
		logicalDelete(dao, dto.getPfaUserPasswordId());
	}
	
	@Override
	public void initPassword(List<String> userIdList) throws MospException {
		// システム日付取得
		Date systemDate = getSystemDate();
		// パスワード確認クラス取得
		PasswordCheckBeanInterface passwordCheck = (PasswordCheckBeanInterface)createBean(PasswordCheckBeanInterface.class);
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 更新処理
		for (String userId : userIdList) {
			// 初期パスワード取得
			String initialPassword = passwordCheck.getInitialPassword(userId);
			// パスワード作成
			String password = SeUtility.encrypt(SeUtility.encrypt(initialPassword));
			// 対象ユーザにおけるシステム日付の情報を取得
			UserPasswordDtoInterface dto = dao.findForInfo(userId);
			// 存在確認(存在しなければ新規登録、存在すれば更新)
			if (dto != null) {
				// 論理削除
				delete(dto);
			}
			// DTO作成
			dto = getInitDto();
			dto.setUserId(userId);
			dto.setChangeDate(systemDate);
			dto.setPassword(password);
			// レコード識別ID最大値をインクリメントしてDTOに設定
			dto.setPfaUserPasswordId(findForMaxId(dao) + 1);
			// 登録
			dao.insert(dto);
		}
		// ロック解除
		unLockTable();
	}
	
}
