/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.wizards.project;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;

import jp.sf.maskat.runtime.MaskatRuntimePlugin;
import jp.sf.maskat.runtime.PropertyInitializer;
import jp.sf.maskat.ui.ISharedImages;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.Messages;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IContributor;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.ide.IDEWorkbenchMessages;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.ui.internal.ide.IIDEHelpContextIds;
import org.eclipse.ui.internal.ide.dialogs.ProjectContentsLocationArea;
import org.eclipse.ui.internal.ide.dialogs.ProjectContentsLocationArea.IErrorMessageReporter;
import org.eclipse.ui.internal.ide.filesystem.FileSystemSupportRegistry;

/**
 * Standard main page for a wizard that is creates a project resource.
 * <p>
 * This page may be used by clients as-is; it may be also be subclassed to suit.
 * </p>
 * <p>
 * Example usage:
 * <pre>
 * mainPage = new WizardNewProjectCreationPage("basicNewProjectPage");
 * mainPage.setTitle("Project");
 * mainPage.setDescription("Create a new project resource.");
 * </pre>
 * </p>
 */
public class NewMaskatProjectCreationPage extends WizardPage {
	
	/** The wizard page ID */
	public static final String PAGE_ID = MaskatUIPlugin.PLUGIN_ID
			+ ".newMaskatProjectCreationPage";
	
       // initial value stores
    private String initialProjectFieldValue;


    protected CheckboxTableViewer viewer;
	
    // widgets
    protected Text projectNameField;

    protected Label projectLabel;
    
    private Listener nameModifyListener = new Listener() {
        public void handleEvent(Event e) {
        	setLocationForSelection();
            boolean valid = validatePage();
            setPageComplete(valid);
                
        }
    };

    protected ProjectContentsLocationArea locationArea;

    // constants
	protected static final int SIZING_TEXT_FIELD_WIDTH = 250;
    
    protected Text maskatFWPath;
    
    protected Button shouldCopy;

    protected IPath frameworkPath;

    protected boolean shouldCopyFlag;
		
	public NewMaskatProjectCreationPage(IPath path, boolean copy) {
		this(PAGE_ID, path, copy);
		setTitle(Messages.getString("wizard.project.title")); //$NON-NLS-1$
		setDescription(Messages.getString("wizard.project.description")); //$NON-NLS-1$
	}
	
    /**
     * Creates a new project creation wizard page.
     *
     * @param pageName the name of this page
     */
    public NewMaskatProjectCreationPage(String pageName,
    		IPath path, boolean copy) {
        super(pageName);
        
        this.frameworkPath = path;
        this.shouldCopyFlag = copy;
        setPageComplete(false);
    }
    
    /** (non-Javadoc)
     * Method declared on IDialogPage.
     */
    public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NULL);
        composite.setFont(parent.getFont());

        initializeDialogUnits(parent);

        PlatformUI.getWorkbench().getHelpSystem().setHelp(composite,
                IIDEHelpContextIds.NEW_PROJECT_WIZARD_PAGE);

        composite.setLayout(new GridLayout());
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        createProjectNameGroup(composite);

        Group projectGroup = new Group(composite, SWT.NONE);
        projectGroup.setLayout(new GridLayout());
        projectGroup.setLayoutData(new GridData(GridData.FILL_BOTH));
        projectGroup.setText(Messages.getString("wizard.project.group.project")); //$NON-NLS-1$
        
        locationArea = createProjectLocationArea(getErrorReporter(), projectGroup);
        if(initialProjectFieldValue != null) {
			locationArea.updateProjectName(initialProjectFieldValue);
		}
        Group fwGroup = new Group(composite, SWT.NONE);
        GridLayout layout = new GridLayout(3, false);
        layout.marginLeft = 5;
        fwGroup.setLayout(layout);
        fwGroup.setLayoutData(new GridData(GridData.FILL_BOTH));
        
        fwGroup.setText(Messages.getString("wizard.project.group.framework")); //$NON-NLS-1$
		Label dirLabel = new Label(fwGroup, SWT.NULL);
		dirLabel.setText(Messages.getString("wizard.project.label.path")); //$NON-NLS-1$
		
		maskatFWPath = new Text(fwGroup, SWT.BORDER);
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;
		maskatFWPath.setLayoutData(gridData);
		maskatFWPath.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setPageComplete(validatePage());
			}
		});
		shouldCopy = new Button(fwGroup, SWT.CHECK | SWT.RIGHT);
		GridData copyData = new GridData();
		copyData.horizontalSpan = 5;
		shouldCopy.setLayoutData(copyData);
		shouldCopy.setText(Messages.getString("wizard.project.label.shuldCopy")); //$NON-NLS-1$
		viewer = CheckboxTableViewer.newCheckList(fwGroup, SWT.FULL_SELECTION | SWT.BORDER);
		shouldCopy.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				viewer.getControl().setEnabled(shouldCopy.getSelection());
			}
		});		
		Table table = viewer.getTable();
		GridData pluginGridData = new GridData(GridData.FILL_BOTH);
		pluginGridData.horizontalSpan = 3;
		table.setLayoutData(pluginGridData);		
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

	    TableColumn idColumn = new TableColumn(table,SWT.CENTER);
	    idColumn.setText(Messages.getString(
	    		"wizard.project.plugin.pluginId")); //$NON-NLS-1$
	    idColumn.setWidth(120);	
	    TableColumn versionColumn = new TableColumn(table,SWT.LEFT );
	    versionColumn.setText(Messages.getString(
	    		"wizard.project.plugin.version")); //$NON-NLS-1$
	    versionColumn.setWidth(80);
	    TableColumn discColumn = new TableColumn(table,SWT.LEFT);
	    discColumn.setText(Messages.getString(
	    		"wizard.project.plugin.description")); //$NON-NLS-1$
	    discColumn.setWidth(300);	    
	    
		viewer.setContentProvider(new ArrayContentProvider());
		viewer.setLabelProvider(new PluginResouceLabelProvider());
		viewer.setInput(createTableObjects());
		viewer.setAllChecked(true);
		
		viewer.addCheckStateListener(new ICheckStateListener() {
			public void checkStateChanged(CheckStateChangedEvent event) {
				PluginResourceData data = (PluginResourceData) event.getElement();
				if (data.isRequired()) {
					CheckboxTableViewer viewer = (CheckboxTableViewer) event.getSource();
					viewer.setChecked(data, true);
				}
			}
		});

		// Scale the button based on the rest of the dialog
		setButtonLayoutData(locationArea.getBrowseButton());
		initPage();
        setPageComplete(validatePage());
        // Show description on opening
        setErrorMessage(null);
        setMessage(null);
        setControl(composite);
    }

	protected ProjectContentsLocationArea createProjectLocationArea(
			IErrorMessageReporter reporter,	Composite composite) {
		return new ProjectContentsLocationArea(reporter, composite);
	}
	
    private void initPage() {
    	maskatFWPath.setText(frameworkPath.toString());
    	shouldCopy.setSelection(shouldCopyFlag);
    }
 
	private class PluginResouceLabelProvider extends LabelProvider
		implements ITableLabelProvider {
	
		private Image pluginImage =
				MaskatUIPlugin.getImage(ISharedImages.IMG_WIZBAN_PLUGIN);
		
		private Image requiredPluginImage =
				MaskatUIPlugin.getImage(ISharedImages.IMG_WIZBAN_REQUIRED_PLUGIN);
		
		public Image getColumnImage(Object element, int columnIndex) {
			if (columnIndex == 0) {
				return ((PluginResourceData) element).isRequired() ?
						requiredPluginImage : pluginImage;
			}
			return null;
		}
	
		public String getColumnText(Object element, int columnIndex) {
			PluginResourceData data = (PluginResourceData) element;
			switch (columnIndex) {
			case 0:
				return data.getPluginId();
			case 1:
				return data.getVersion();				
			case 2:
				return data.getDescription();
			}
			return null;			
		}
	}
	private Object createTableObjects() {
		ArrayList list = new ArrayList();
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(
				MaskatRuntimePlugin.PLUGIN_ID, "resources"); //$NON-NLS-1$
		IExtension[] extensions = point.getExtensions();
		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements = extensions[i].getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				IConfigurationElement element = elements[j];
				String path = element.getAttribute("path"); //$NON-NLS-1$
				PropertyInitializer initializer = null;
				if (element.getAttribute("propertyInitializer") != null) { //$NON-NLS-1$
					try {
						initializer = (PropertyInitializer)
							element.createExecutableExtension("propertyInitializer"); //$NON-NLS-1$
					} catch (CoreException e) {
						MaskatUIPlugin.log(new Status(IStatus.ERROR, MaskatUIPlugin.PLUGIN_ID, 0,
								e.toString(), e));
					}
				}
				PluginResourceData data = new PluginResourceData(
					element.getAttribute("pluginId"), //$NON-NLS-1$
					element.getAttribute("version"), //$NON-NLS-1$
					path,
					element.getAttribute("description"), //$NON-NLS-1$
					new Boolean(element.getAttribute("required")).booleanValue(),
					extensions[i].getNamespaceIdentifier(),
					initializer);
				
				try {
					if (path != null) {
						IContributor contributor = element.getContributor();
						data.setUrl(FileLocator.toFileURL(
							Platform.getBundle(contributor.getName()).getEntry(path)));
					}
					list.add(data);
				} catch (IOException e) {
					MaskatUIPlugin.log(new Status(IStatus.ERROR, MaskatUIPlugin.PLUGIN_ID, 0,
							e.toString(), e));
				}
			}
		}
        Comparator comparator = new Comparator() {
            public int compare(Object obj1, Object obj2) {
            	int b1 = ((PluginResourceData) obj1).isRequired() ? 1 : 0;
            	int b2 = ((PluginResourceData) obj2).isRequired() ? 1 : 0;
                return b2 - b1;
            }
        };
		Collections.sort(list, comparator);
		
        return (PluginResourceData[]) list.toArray(
			new PluginResourceData[0]);
	}


	
    /**
	 * Get an error reporter for the receiver.
	 * @return IErrorMessageReporter
	 */
	private IErrorMessageReporter getErrorReporter() {
		return new IErrorMessageReporter(){
			/* (non-Javadoc)
			 * @see org.eclipse.ui.internal.ide.dialogs.ProjectContentsLocationArea.IErrorMessageReporter#reportError(java.lang.String)
			 */
			public void reportError(String errorMessage) {
				setErrorMessage(errorMessage);
				boolean valid = errorMessage == null;
				if(valid) {
					valid = validatePage();
				}
				
				setPageComplete(valid);
			}

			public void reportError(String errorMessage, boolean infoOnly) {
				reportError(errorMessage);
			}
		};
	}

    /**
     * Creates the project name specification controls.
     *
     * @param parent the parent composite
     */
    private final void createProjectNameGroup(Composite parent) {
        // project specification group
        Composite projectGroup = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        projectGroup.setLayout(layout);
        projectGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        // new project label
        projectLabel = new Label(projectGroup, SWT.NONE);
        projectLabel.setText(IDEWorkbenchMessages.WizardNewProjectCreationPage_nameLabel);
        projectLabel.setFont(parent.getFont());

        // new project name entry field
        projectNameField = new Text(projectGroup, SWT.BORDER);
        GridData data = new GridData(GridData.FILL_HORIZONTAL);
        data.widthHint = SIZING_TEXT_FIELD_WIDTH;
        projectNameField.setLayoutData(data);
        projectNameField.setFont(parent.getFont());

        // Set the initial value first before listener
        // to avoid handling an event during the creation.
        if (initialProjectFieldValue != null) {
			projectNameField.setText(initialProjectFieldValue);
		}
        projectNameField.addListener(SWT.Modify, nameModifyListener);
    }


    /**
     * Returns the current project location path as entered by 
     * the user, or its anticipated initial value.
     * Note that if the default has been returned the path
     * in a project description used to create a project
     * should not be set.
     *
     * @return the project location path or its anticipated initial value.
     */
    public IPath getLocationPath() {
        return new Path(locationArea.getProjectLocation());
    }
    
    /**
    /**
     * Returns the current project location URI as entered by 
     * the user, or <code>null</code> if a valid project location
     * has not been entered.
     *
     * @return the project location URI, or <code>null</code>
     * @since 3.2
     */
    public URI getLocationURI() {
    	return locationArea.getProjectLocationURI();
    }

    /**
     * Creates a project resource handle for the current project name field value.
     * <p>
     * This method does not create the project resource; this is the responsibility
     * of <code>IProject::create</code> invoked by the new project resource wizard.
     * </p>
     *
     * @return the new project resource handle
     */
    public IProject getProjectHandle() {
        return ResourcesPlugin.getWorkspace().getRoot().getProject(
                getProjectName());
    }

    /**
     * Returns the current project name as entered by the user, or its anticipated
     * initial value.
     *
     * @return the project name, its anticipated initial value, or <code>null</code>
     *   if no project name is known
     */
    public String getProjectName() {
        if (projectNameField == null) {
			return initialProjectFieldValue;
		}

        return getProjectNameFieldValue();
    }

    /**
     * Returns the value of the project name field
     * with leading and trailing spaces removed.
     * 
     * @return the project name in the field
     */
    private String getProjectNameFieldValue() {
        if (projectNameField == null) {
			return ""; //$NON-NLS-1$
		}

        return projectNameField.getText().trim();
    }

    /**
     * Sets the initial project name that this page will use when
     * created. The name is ignored if the createControl(Composite)
     * method has already been called. Leading and trailing spaces
     * in the name are ignored.
     * Providing the name of an existing project will not necessarily 
     * cause the wizard to warn the user.  Callers of this method 
     * should first check if the project name passed already exists 
     * in the workspace.
     * 
     * @param name initial project name for this page
     * 
     * @see IWorkspace#validateName(String, int)
     * 
     */
    public void setInitialProjectName(String name) {
        if (name == null) {
			initialProjectFieldValue = null;
		} else {
            initialProjectFieldValue = name.trim();
            if(locationArea != null) {
				locationArea.updateProjectName(name.trim());
			}
        }
    }

    /**
     * Set the location to the default location if we are set to useDefaults.
     */
    void setLocationForSelection() {
    	locationArea.updateProjectName(getProjectNameFieldValue());
    }

  
    /**
     * Returns whether this page's controls currently all contain valid 
     * values.
     *
     * @return <code>true</code> if all controls are valid, and
     *   <code>false</code> if at least one is invalid
     */
    protected boolean validatePage() {
        IWorkspace workspace = IDEWorkbenchPlugin.getPluginWorkspace();

        String projectFieldContents = getProjectNameFieldValue();
        if (projectFieldContents.equals("")) { //$NON-NLS-1$
            setErrorMessage(null);
            setMessage(IDEWorkbenchMessages.WizardNewProjectCreationPage_projectNameEmpty);
            return false;
        }

        IStatus nameStatus = workspace.validateName(projectFieldContents,
                IResource.PROJECT);
        if (!nameStatus.isOK()) {
            setErrorMessage(nameStatus.getMessage());
            return false;
        }

        IProject handle = getProjectHandle();
        if (handle.exists()) {
            setErrorMessage(IDEWorkbenchMessages.WizardNewProjectCreationPage_projectExistsMessage);
            return false;
        }
        
        /*
         * If not using the default value validate the location.
         */
        if (!locationArea.isDefault()) {
            IStatus locationStatus = workspace.validateProjectLocationURI(handle,
                    locationArea.getProjectLocationURI());
            if (!locationStatus.isOK()) {
                setErrorMessage(locationStatus.getMessage());
                return false;
            }
        }
        String path = maskatFWPath.getText();
		if (path == null || path.length() == 0) {
			setErrorMessage(Messages.getString(
					"wizard.project.msg.error.notfwpath")); //$NON-NLS-1$);
			maskatFWPath.setFocus();
			return false;
		}
		if (path.indexOf("/") == -1) {
			path = "/" + path;
		}
		URI fwUrl = FileSystemSupportRegistry.getInstance()
			.getDefaultConfiguration().getContributor().getURI(path);
		
        IStatus fwLocationStatus = workspace.validateProjectLocationURI(handle, fwUrl);
        if (!fwLocationStatus.isOK()) {
            setErrorMessage(fwLocationStatus.getMessage());
			maskatFWPath.setFocus();
            return false;
        }
        setErrorMessage(null);
        setMessage(null);
        return true;
    }

    /*
     * see @DialogPage.setVisible(boolean)
     */
    public void setVisible(boolean visible) {
        super.setVisible(visible);
        if (visible) {
			projectNameField.setFocus();
		}
    }

    /**
     * Returns the useDefaults.
     * @return boolean
     */
    public boolean useDefaults() {
        return locationArea.isDefault();
    }

	public boolean shouldCopyMaskatFramework() {
		return shouldCopy.getSelection();
	}

	public IPath getMaskatFrameworkPath() {
		return new Path(maskatFWPath.getText());
	}
	
	public PluginResourceData[] getSelectedPlugins() {
		Object[] select = viewer.getCheckedElements();
		PluginResourceData[] result = new PluginResourceData[select.length];
		for (int i = 0; i < select.length; i++) {
			result[i] = (PluginResourceData) select[i];
		}
		return result;
	}
}

