/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.layout.outline;

import java.util.List;

import jp.sf.maskat.core.layout.AbstractContainer;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.Messages;
import jp.sf.maskat.ui.editors.layout.LayoutGraphicalEditor;
import jp.sf.maskat.ui.editors.layout.editparts.ComponentEditPart;
import jp.sf.maskat.ui.editors.layout.editparts.IdDirectEditable;
import jp.sf.maskat.ui.editors.layout.editparts.LayoutElementEditPart;
import jp.sf.maskat.ui.editors.layout.tools.IdDirectEditmanger;
import jp.sf.maskat.ui.editors.layout.tools.TextCellEditorLocator;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.draw2d.geometry.Rectangle;
//import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.ContextMenuProvider;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.gef.ui.actions.GEFActionConstants;
import org.eclipse.gef.ui.actions.UpdateAction;
import org.eclipse.gef.ui.parts.ContentOutlinePage;
import org.eclipse.gef.ui.parts.TreeViewer;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.TreeItem;
//import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.part.IPageSite;


public class LayoutOutLinePage extends ContentOutlinePage implements
	ISelectionChangedListener, IAdaptable {
	
	public static final String EDITOR_ID = MaskatUIPlugin.PLUGIN_ID
			+ ".outlinePage"; //$NON-NLS-1$	

	private LayoutGraphicalEditor editor;
	
	private EditNameAction directEdit;
	
	private ISelection selection = null;
	
	public LayoutOutLinePage(LayoutGraphicalEditor editor) {
		super(new TreeViewer());
		this.editor = editor;
	}

	public void init(IPageSite pageSite) {
		super.init(pageSite);
		ActionRegistry registry = (ActionRegistry) editor.getAdapter(ActionRegistry.class);
		IActionBars bars = pageSite.getActionBars();
		String delId = ActionFactory.DELETE.getId();
		String undoId = ActionFactory.UNDO.getId();
		String redoId = ActionFactory.REDO.getId();
		bars.setGlobalActionHandler(delId, registry.getAction(delId));
		bars.setGlobalActionHandler(undoId, registry.getAction(undoId));
		bars.setGlobalActionHandler(redoId, registry.getAction(redoId));
		bars.updateActionBars();
		directEdit = new EditNameAction();
	}

	private class EditNameAction extends Action {
		
		public EditNameAction() {
			super.setText(Messages.getString(
					"layout.cmd.rename.msg.title")); //$NON-NLS-1$
		}
		
		public boolean canEdit() {
			LayoutElementEditPart editPart = getLayoutEditPart();
			if (editPart != null && editPart.getParent() != null) {
				IdDirectEditable editable = (IdDirectEditable)
						editPart.getAdapter(IdDirectEditable.class);
				try {
					if (editable != null && !"".equals(editable.getId())) {
						return true;
					}
				} catch (IllegalArgumentException e) {
					return false;
				}
			}
			return false;
		}

		public boolean canDelete() {
			LayoutElementEditPart editPart = getLayoutEditPart();
			return editPart != null ? editPart.canDelete() : false;
		}
		
		private LayoutElementEditPart getLayoutEditPart() {
			if (selection instanceof StructuredSelection) {
				StructuredSelection sel = (StructuredSelection) selection;
				if (sel != null && !sel.isEmpty() && sel.getFirstElement() instanceof EditPart) {
					OutLineTreeEditPart editPart = (OutLineTreeEditPart) sel.getFirstElement();
					if (editPart.getWidget() instanceof TreeItem) {
						EditPart part = editPart.getLayoutEditPart();
						if (part instanceof LayoutElementEditPart) {
							return (LayoutElementEditPart) part;
						}
					}
				}
			}
			return null;
		}
		
		public void run() {
			if (canEdit()) {
				StructuredSelection sel = (StructuredSelection) selection;
				OutLineTreeEditPart editPart = (OutLineTreeEditPart) sel.getFirstElement();				
				org.eclipse.swt.graphics.Rectangle rec = ((TreeItem) editPart.getWidget()).getBounds();
				Rectangle area = new Rectangle(rec.x, rec.y, rec.width, rec.height);
				GraphicalEditPart layoutEditpart = (GraphicalEditPart) editPart.getLayoutEditPart();
				if (layoutEditpart != null 	&& layoutEditpart.getParent() != null && layoutEditpart.getAdapter(IdDirectEditable.class) != null) {
					new IdDirectEditmanger(layoutEditpart, (Composite) getViewer().getControl(), 
							new TextCellEditorLocator(area)).show();
				}			
			}
		}
	}
	
	public void createControl(Composite parent) {
		super.createControl(parent);
		EditPartViewer viewer = getViewer();
		GraphicalViewer gviewer = (GraphicalViewer) editor.getAdapter(GraphicalViewer.class);				
		viewer.setEditDomain(gviewer.getEditDomain());
		
		EditPart editPart = gviewer.getRootEditPart();
		viewer.setEditPartFactory(new OutLineTreeEditPartFactory(gviewer));
		
		ActionRegistry registry = (ActionRegistry) editor.getAdapter(ActionRegistry.class);
		ContextMenuProvider provider = new OutlineContextMenuProvider(viewer, registry);
		viewer.setContextMenu(provider);
		viewer.setContents(getRootModel(editPart));
		viewer.addSelectionChangedListener(this);
	}

	public void update() {
		GraphicalViewer gviewer = (GraphicalViewer) editor.getAdapter(GraphicalViewer.class);			
		getViewer().setContents(getRootModel(gviewer.getRootEditPart()));
	}
	
	private class OutlineContextMenuProvider extends ContextMenuProvider {
		private ActionRegistry actionRegistry;

		public OutlineContextMenuProvider(EditPartViewer viewer, ActionRegistry registry) {
			super(viewer);
			actionRegistry = registry;
		}

		public void buildContextMenu(IMenuManager menu) {
			updateDeleteActions(actionRegistry);
			GEFActionConstants.addStandardActionGroups(menu);
			IAction undoAction = actionRegistry.getAction(ActionFactory.UNDO.getId());
			menu.appendToGroup(GEFActionConstants.GROUP_EDIT, undoAction);
			IAction redoAction = actionRegistry.getAction(ActionFactory.REDO.getId());
			menu.appendToGroup(GEFActionConstants.GROUP_EDIT, redoAction);
			if (directEdit.canDelete()) {
				menu.appendToGroup(GEFActionConstants.GROUP_EDIT, new Separator());
				IAction delAction = actionRegistry.getAction(ActionFactory.DELETE.getId());
				menu.appendToGroup(GEFActionConstants.GROUP_EDIT, delAction);
			}
			if (directEdit.canEdit()) {
				menu.add(directEdit);
			}
		}
	}
	
	private void updateDeleteActions(ActionRegistry registry) {
		IAction delAction = registry.getAction(ActionFactory.DELETE.getId());
		if (delAction instanceof UpdateAction) {
			((UpdateAction) delAction).update();
		}
		IAction undoAction = registry.getAction(ActionFactory.UNDO.getId());
		if (undoAction instanceof UpdateAction) {
			((UpdateAction) undoAction).update();
		}
		IAction redoAction = registry.getAction(ActionFactory.REDO.getId());
		if (redoAction instanceof UpdateAction) {
			((UpdateAction) redoAction).update();
		}
	}
	
	private Object getRootModel(EditPart parentElement) {
		List partList = parentElement.getChildren();
		if (partList.size() > 0) {
			Object model = ((EditPart) partList.get(0)).getModel();
			AbstractContainer root = new AbstractContainer() {};
			root.addChild(model);
			return root;
		}
		return null;
	}
	
	public EditPartViewer getViewer() {
		return super.getViewer();
	}

	private EditPart beforeSelectionEditPart = null;
	private long beforeSelectionTime = 0;
	
	public void selectionChanged(SelectionChangedEvent event) {
		selection = event.getSelection();
		if (selection instanceof StructuredSelection) {
			StructuredSelection sel = (StructuredSelection) selection;
			if (sel != null && !sel.isEmpty() && sel.getFirstElement() instanceof EditPart) {
				OutLineTreeEditPart editPart = (OutLineTreeEditPart) sel.getFirstElement();
				if (beforeSelectionEditPart == editPart && sel.size() == 1 &&
						System.currentTimeMillis() - beforeSelectionTime <= 200) {
					if (directEdit.canEdit()) {
						directEdit.run();
					}
				} else {
					GraphicalEditPart g = (GraphicalEditPart) editPart.getLayoutEditPart();
					if (g instanceof ComponentEditPart) {
						ComponentEditPart part = (ComponentEditPart) g;
						part.refreshSelectedWidget(null);
					}
				}
				beforeSelectionEditPart = editPart;
				beforeSelectionTime = System.currentTimeMillis();
			}
		}
	}

	public Object getAdapter(Class adapter) {
		return editor.getAdapter(adapter);
	}
}
