/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.layout;

import java.beans.IntrospectionException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import jp.sf.maskat.core.betwixt.MaskatBeanIO;
import jp.sf.maskat.core.event.EventDef;
import jp.sf.maskat.core.layout.Layout;
import jp.sf.maskat.core.layout.LayoutDef;

import org.apache.commons.io.IOUtils;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.xml.sax.SAXException;

/**
 * マスカットエディタで利用するレイアウト定義、イベント定義情報を持つクラスです
 * 
 * このクラスはレイアウト、イベント定義XMLとJavaBean間の変換を行います。
 * 
 * ・レイアウト定義XML　<-> jp.sf.maskat.core.layout.LayoutDef
 * ・イベント定義XML <-> jp.sf.maskat.core.event.EventDef
 *
 */
public class MaskatResources {

	/**
	 * レイアウト情報
	 */
    private LayoutDef layoutDef = null;

    /**
     * イベント情報
     */
    private EventDef eventDef;

    /**
     * レイアウト定義XML
     */
    private IFile layoutXMLFile;

    /**
     * イベント定義XML
     */
    private IFile eventXMLFile;

    /**
     * コンストラクタです
     * 
     * @param layoutXMLFile　レイアウト定義XML
     */
    MaskatResources(IFile layoutXMLFile) {
        this.layoutXMLFile = layoutXMLFile;

        /*
         * イベント定義XMLはレイアウト定義XMLと同フォルダにあり
         * レイアウト定義XML名の後ろに _e が付与されます。
         * 
         * レイアウト定義XML layout.xml の場合、イベント定義XMLは
         * layout_e.xmlとなります。
         */
        IContainer folder = layoutXMLFile.getParent();
        String eventXMLFileName = layoutXMLFile.getLocation()
                .removeFileExtension().lastSegment()
                + "_e.xml";
        this.eventXMLFile = folder.getFile(new Path(eventXMLFileName));
    }

    /**
     * レイアウト定義XMLのファイルリソースを取得します
     * 
     * @return　レイアウト定義XMLのファイルリソース
     */
    public IFile getLayoutXMLFile() {
        return layoutXMLFile;
    }

    /**
     * イベント定義XMLのファイルリソースを取得します
     * 
     * @return イベント定義XMLのファイルリソース
     */
    public IFile getEventXMLFile() {
        return eventXMLFile;
    }

    /**
     * レイアウトを取得します
     * 
     * @return レイアウト
     */
    Layout getLayout() {
    	/*
    	 *  マスカットIDE v2.0からLayoutDefには１つのLayoutしか定義できない
    	 *  仕様となりました。そのため配列の 0番目を返却しています。
    	 */
        Layout[] layouts = layoutDef.getLayouts();
        return layouts.length > 0 ? layouts[0] : null;
    }

    /**
     * レイアウト、イベント定義XMLの読み込みを行います
     * 
     * @throws CoreException ランタイムエラーが発生した場合
     * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
     */
    public void load() throws CoreException, IntrospectionException,
                      IOException, SAXException {
        loadLayout();
        loadEvent();
        connect();
    }

    /**
     * レイアウトにイベントを登録します
     */
    void connect() {
        layoutDef.setEventDef(eventDef);
    }

    /**
     * レイアウト定義XMLの読み込みを行います
     * 
     * @throws CoreException ランタイムエラーが発生した場合
     * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
     */
    void loadLayout() throws CoreException, IntrospectionException,
                     IOException, SAXException {
        InputStream in = null;
        try {
            if (layoutXMLFile.exists()) {
                in = layoutXMLFile.getContents();
                layoutDef = MaskatBeanIO.readLayoutDef(in);
            }
        } finally {
            IOUtils.closeQuietly(in);
        }
    }

    /**
     * イベント定義XMLの読み込みを行います
     * イベント定義XMLが存在しない場合は空のイベント定義を作成します
     * 
     * @throws CoreException ランタイムエラーが発生した場合
     * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
     */
    void loadEvent() throws CoreException, SAXException, IOException,
                    IntrospectionException {
        if (!eventXMLFile.exists()) {
        	eventDef = new EventDef();
        	return;
        }
        InputStream in = null;
        try {
            eventXMLFile.refreshLocal(IResource.DEPTH_ONE, null);
            if (eventXMLFile.exists()) {
                in = eventXMLFile.getContents();
                eventDef = MaskatBeanIO.readEventDef(in);
            }
        } finally {
            IOUtils.closeQuietly(in);
        }
    }

    /**
     * レイアウト、イベント定義XMLの保存を行います
     * 
     * @throws CoreException ランタイムエラーが発生した場合
     * @throws IntrospectionException JavaBeanからの変換でエラーが発生した場合
     * @throws IOException ファイル書き込みでエラーが発生した場合
     * @throws SAXException XML パーサでエラーが発生した場合
     */
    public void save() throws IntrospectionException, IOException,
                      SAXException, CoreException {
        writeLayout();
        writeEvent();
    }

    /**
     * レイアウト定義XMLの保存を行います
     * 
     * @throws CoreException ランタイムエラーが発生した場合
     * @throws IntrospectionException JavaBeanからの変換でエラーが発生した場合
     * @throws IOException ファイル書き込みでエラーが発生した場合
     * @throws SAXException XML パーサでエラーが発生した場合
     */
    private void writeLayout() throws IntrospectionException, IOException,
                              SAXException, CoreException {

    	String output = MaskatBeanIO.writeLayoutDef(layoutDef);
        byte[] bytes = output.getBytes("UTF-8");
        layoutXMLFile.setContents(new ByteArrayInputStream(bytes), false,
                false, null);
    }

    /**
     * イベント定義XMLの保存を行います
     * 
     * @throws CoreException ランタイムエラーが発生した場合
     * @throws IntrospectionException JavaBeanからの変換でエラーが発生した場合
     * @throws IOException ファイル書き込みでエラーが発生した場合
     * @throws SAXException XML パーサでエラーが発生した場合
     */
    private void writeEvent() throws IntrospectionException, IOException,
                             SAXException, CoreException {
        if (eventDef == null) {
            return;
        }
        String result = MaskatBeanIO.writeEventDef(eventDef);

        if (!eventXMLFile.exists()) {
        	eventXMLFile.create(new ByteArrayInputStream(new byte[0]), false, null);
        }
        eventXMLFile.setContents(new ByteArrayInputStream(result.getBytes("UTF-8")),
        		false, false, null);        

        /*
         * イベント参照からイベントの置き換えを行います
         * TODO: なぜ保存後にeventRefを置き換えているのか不明
         *        IDEではイベント参照を定義できないので読み込み時に
         *        行うのでは？
         */
        eventDef.prepareEvent();
    }

    /**
     * EventDefを取得します
     * @return EventDef
     */
    public EventDef getEventDef() {
        return eventDef;
    }

    /**
     * LayoutDefを取得します
     * @return LayoutDef
     */
    public LayoutDef getLayoutDef() {
        return layoutDef;
    }
}
