/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.layout;

import java.io.IOException;
import java.util.EventObject;
import java.util.Iterator;
import java.util.List;

import jp.sf.maskat.ui.MaskatNature;
import jp.sf.maskat.ui.MaskatPerspectiveFactory;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.Messages;
import jp.sf.maskat.ui.editors.LayoutEditorMenuManager;
import jp.sf.maskat.ui.editors.layout.actions.CopyComponentAction;
import jp.sf.maskat.ui.editors.layout.actions.CutComponentAction;
import jp.sf.maskat.ui.editors.layout.actions.DeleteComponentAction;
import jp.sf.maskat.ui.editors.layout.actions.PasteComponentAction;
import jp.sf.maskat.ui.editors.layout.editparts.ChainedEditPartFactory;
import jp.sf.maskat.ui.editors.layout.outline.LayoutOutLinePage;
import jp.sf.maskat.ui.editors.layout.tools.AdvancedSelectionTool;
import jp.sf.maskat.ui.views.properties.tabbed.TabbedEventPropertySheetPage;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.draw2d.PositionConstants;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.gef.DefaultEditDomain;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.SnapToGeometry;
import org.eclipse.gef.SnapToGrid;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.gef.ui.actions.AlignmentAction;
import org.eclipse.gef.ui.parts.GraphicalEditorWithFlyoutPalette;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertySheetPageContributor;

/**
 * グラフィカルレイアウトエディタクラスです
 * 
 * グラフィカルユーザインタフェースによるレイアウト定義を行います
 * 主な機能は以下のとおりです。
 * ・コンポーネントの配置
 * ・コンポーネントの移動、サイズ変更
 * ・コンポーネントの削除
 * ・コンポーネントのラベル直接編集
 */
public class LayoutGraphicalEditor
		extends GraphicalEditorWithFlyoutPalette
		implements ITabbedPropertySheetPageContributor,
					IResourceChangeListener {
	/**
	 * レイアウトエディタID
	 */
	public static final String EDITOR_ID = MaskatUIPlugin.PLUGIN_ID
			+ ".layoutGraphicalEditor"; //$NON-NLS-1$

	/**
	 * レイアウト、イベントのリソースデータ
	 */
	protected MaskatResources data;

	/**
	 * アウトラインページ
	 */
	private LayoutOutLinePage outlinePage = null;

	/**
	 * パレットカスタマイザ
	 */
	private PluggablePaletteCustomizer customizer;

	/**
	 * プロジェクト
	 */
	private IProject project;

	/**
	 * セーブ処理フラグ
	 * doSaveメソッドによる保存処理中はtrue
	 */
	private boolean saving = false;

	/**
	 * デフォルトコンストラクタです
	 *
	 */
	public LayoutGraphicalEditor() {
		super();
		setEditDomain(new DefaultEditDomain(this));
	}

	/**
     * {@inheritDoc}
     */
    public void createPartControl(Composite parent) {
		super.createPartControl(parent);
		getEditDomain().setDefaultTool(new AdvancedSelectionTool());
    }

	/**
     * {@inheritDoc}
     */
	protected PaletteRoot getPaletteRoot() {
		synchronized (LayoutGraphicalEditor.class) {
			if (customizer == null) {
				customizer = new PluggablePaletteCustomizer();
			}
		}
		return customizer.getPaletteRoot();
	}

    /**
     * {@inheritDoc}
     */
	protected void initializeGraphicalViewer() {
		GraphicalViewer viewer = getGraphicalViewer();
		MaskatNature nature = MaskatNature.getNature(project);

		/*
		 * このプロジェクトにマスカットネイチャが存在した場合、各種設定
		 * をプリファレンスストアーに保存することができます。
		 */
		if (nature != null) {
			boolean isGrid = nature.getGridSelection();
			boolean isToGrid = nature.getSnapToGridSelection();
			boolean isToObj = nature.getSnapToGeometrySelection();
			int gridXSize = nature.getGridSize();
			int gridYSize = gridXSize;

			viewer.setProperty(SnapToGrid.PROPERTY_GRID_SPACING, new Dimension(
					gridXSize, gridYSize));

			viewer.setProperty(SnapToGeometry.PROPERTY_SNAP_ENABLED, Boolean
					.valueOf(isToObj));

			viewer.setProperty(SnapToGrid.PROPERTY_GRID_ENABLED, Boolean
					.valueOf(isToGrid));

			viewer.setProperty(SnapToGrid.PROPERTY_GRID_VISIBLE, Boolean
					.valueOf(isGrid));

			genContextMenu(isGrid, isToGrid, isToObj);

		} else {
			/*
			 * マスカットネイチャが存在しないプロジェクトの場合、「グリッドの表示」、
			 * 「グリッドに合わせる」、「オブジェクトに合わせる」が非選択状態となります。
			 */
			genContextMenu(false, false, false);
		}
		if (viewer.getEditDomain().getPaletteViewer() != null) {
			viewer.getEditDomain().getPaletteViewer().setCustomizer(customizer);
		} else {
			try {
				IWorkbenchPage page = getSite().getPage();
				page.showView(MaskatPerspectiveFactory.ID_PALETTE);
			} catch (PartInitException e) {
				MaskatUIPlugin.log(e.getStatus());
			}
		}
		viewer.setContents(data.getLayout());
	}

	/**
	 * コンテキストメニューを作成します
	 * 
	 * 引数の値に合わせてメニューにあるトグルボタン、チェックボタンに
	 * 初期値を設定します。
	 * 
	 * @param isGrid 「グリッドを表示する」の初期値
	 * @param isToGrid 「グリッドに合わせる」の初期値
	 * @param isToObj 「オブジェクトに合わせる」の初期値
	 */
	protected void genContextMenu(boolean isGrid, boolean isToGrid,
			boolean isToObj) {
		final GraphicalViewer viewer = getGraphicalViewer();
		MenuManager mm = viewer.getContextMenu();
		if (mm == null) {
			mm = new LayoutEditorMenuManager(this);
			viewer.setContextMenu(mm);
		}
		mm.removeAll();

		IWorkbenchWindow window = getSite().getWorkbenchWindow();
		mm.add(ActionFactory.UNDO.create(window));
		mm.add(ActionFactory.REDO.create(window));
		mm.add(new Separator());
		mm.add(ActionFactory.CUT.create(window));
		mm.add(ActionFactory.COPY.create(window));
		mm.add(ActionFactory.PASTE.create(window));
		mm.add(new Separator());
		mm.add(ActionFactory.DELETE.create(window));
		mm.add(new Separator());

		/*
		 * 整列メニューの作成
		 * エディタのactionRegistryから項目を取得する
		 */
		MenuManager smm = new MenuManager(Messages
				.getString("layout.contextmenu.alignmenu")); //$NON-NLS-1$
		ActionRegistry registry = getActionRegistry();

		for (Iterator ite = registry.getActions(); ite.hasNext();) {
			Object action = ite.next();
			if (action instanceof AlignmentAction) {
				smm.add((IAction) action);
			}
		}
		mm.add(smm);

		/*
		 * 「グリッドに合わせる」メニューの作成
		 * 「グリッドに合わせる」と「オブジェクトに合わせる」は排他となり
		 * ます。選択するとメニュー項目にチェックマークが付きます。
		 */
		final boolean isSnapGrid = isToGrid;
		final IAction snapToGridAction = new Action(Messages
				.getString("layout.contextmenu.snaptogrid"), //$NON-NLS-1$
				IAction.AS_RADIO_BUTTON) {
			private boolean selected = isSnapGrid;

			public void run() {
				boolean checked = isChecked();
				if (selected && isChecked()) {
					checked = false;
					setChecked(checked);
				}
				selected = checked;
				viewer.setProperty(SnapToGrid.PROPERTY_GRID_ENABLED, Boolean
						.valueOf(checked));
				storeGridSelection("snapToGrid", isChecked()); //$NON-NLS-1$
			}
		};
		/*
		 * 「オブジェクトに合わせる」メニューの作成
		 * 「グリッドに合わせる」と「オブジェクトに合わせる」は排他となり
		 * ます。選択するとメニュー項目にチェックマークが付きます。
		 */
		final boolean isSnapObj = isToObj;
		final IAction snapToGeometryAction = new Action(Messages
				.getString("layout.contextmenu.snaptogeometry"), //$NON-NLS-1$
				IAction.AS_RADIO_BUTTON) {
			private boolean selected = isSnapObj;

			public void run() {
				boolean checked = isChecked();
				if (selected && isChecked()) {
					checked = false;
					setChecked(checked);
				}
				selected = checked;
				viewer.setProperty(SnapToGeometry.PROPERTY_SNAP_ENABLED,
						Boolean.valueOf(isChecked()));
				storeGridSelection("snapToObj", isChecked()); //$NON-NLS-1$
			}
		};
		/*
		 * 「グリッドを表示する」メニューの作成
		 * 選択するとメニュー項目にチェックマークが付きます。
		 */
		IAction viewGridAction = new Action(Messages
				.getString("layout.contextmenu.grid"), //$NON-NLS-1$
				IAction.AS_CHECK_BOX) {
			public void run() {
				snapToGridAction.setEnabled(isChecked());
				if (!isChecked()) {
					viewer.setProperty(SnapToGrid.PROPERTY_GRID_ENABLED,
							Boolean.FALSE);
				} else if (snapToGridAction.isChecked()) {
					viewer.setProperty(SnapToGrid.PROPERTY_GRID_ENABLED,
							Boolean.TRUE);
				}
				viewer.setProperty(SnapToGrid.PROPERTY_GRID_VISIBLE, Boolean
						.valueOf(isChecked()));
				storeGridSelection("gridView", isChecked()); //$NON-NLS-1$
			}
		};
		/*
		 * 「グリッドサイズのUP」メニューの作成
		 * 選択ごとにグリッドサイズが 2 大きくなります
		 */
		IAction upGridAction = new Action(
				Messages.getString("layout.contextmenu.gridsize.up"), IAction.AS_PUSH_BUTTON) { //$NON-NLS-1$
			public void run() {
				Dimension d = (Dimension) viewer
						.getProperty(SnapToGrid.PROPERTY_GRID_SPACING);
				Dimension n = new Dimension(d.width + 2, d.height + 2);
				viewer.setProperty(SnapToGrid.PROPERTY_GRID_SPACING, n);
				storeGridSize(n.width);
			}
		};
		/*
		 * 「グリッドサイズのDown」メニューの作成
		 * 選択ごとにグリッドサイズが 2 小さくなります
		 */
		IAction downGridAction = new Action(
				Messages.getString("layout.contextmenu.gridsize.down"), IAction.AS_PUSH_BUTTON) { //$NON-NLS-1$
			public void run() {
				Dimension d = (Dimension) viewer
						.getProperty(SnapToGrid.PROPERTY_GRID_SPACING);
				Dimension n = new Dimension(d.width > 0 ? d.width - 2 : 0,
						d.height > 0 ? d.height - 2 : 0);
				viewer.setProperty(SnapToGrid.PROPERTY_GRID_SPACING, n);
				storeGridSize(n.width);
			}
		};
		/*
		 * グリッドサイズを「標準値に戻す」メニューの作成
		 */
		IAction baseGridAction = new Action(
				Messages.getString("layout.contextmenu.gridsize.standard"), IAction.AS_PUSH_BUTTON) { //$NON-NLS-1$
			public void run() {
				int size = Integer.parseInt(MaskatNature.GRID_SIZE_DEFAULT);
				viewer.setProperty(SnapToGrid.PROPERTY_GRID_SPACING,
						new Dimension(size, size));
				storeGridSize(Integer.parseInt(MaskatNature.GRID_SIZE_DEFAULT));
			}
		};
		snapToGridAction.setChecked(isToGrid);
		snapToGridAction.setEnabled(isGrid);
		snapToGeometryAction.setChecked(isToObj);
		viewGridAction.setChecked(isGrid);

		MenuManager gridMenu = new MenuManager(Messages
				.getString("layout.contextmenu.gridmenu")); //$NON-NLS-1$
		gridMenu.add(viewGridAction);
		gridMenu.add(snapToGridAction);
		gridMenu.add(snapToGeometryAction);
		gridMenu.add(new Separator());

		MenuManager sizeMenu = new MenuManager(Messages
				.getString("layout.contextmenu.gridsize")); //$NON-NLS-1$
		sizeMenu.add(upGridAction);
		sizeMenu.add(downGridAction);
		sizeMenu.add(new Separator());
		sizeMenu.add(baseGridAction);
		gridMenu.add(sizeMenu);

		mm.add(gridMenu);
		mm.add(new Separator());

		IAction showPropertiesViewAction = new Action(Messages
				.getString("layout.contextmenu.showproperties")) {
			public void run() {
				try {
					getSite().getPage().showView(IPageLayout.ID_PROP_SHEET);
				} catch (PartInitException e) {
					MaskatUIPlugin.log(e.getStatus());
				}
			}
		};
		mm.add(showPropertiesViewAction);
		mm.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
		getSite().registerContextMenu("#layoutGraphicalEditor", mm,
				getSite().getSelectionProvider());
	}

	/**
	 * メニューで選択されたグリッド情報をプレファレンスストアに保存します
	 * 
	 * @param propertyName プロパティ名 （"gridView", "snapToGrid", "snapToObj"）
	 * @param value 値
	 */
	private void storeGridSelection(String propertyName, boolean value) {
		MaskatNature nature = MaskatNature.getNature(project);
		if ("gridView".equals(propertyName)) { //$NON-NLS-1$
			nature.setGridSelection(value);
		} else if ("snapToGrid".equals(propertyName)) { //$NON-NLS-1$
			if (value) {
				nature.setSnapToGeometrySelection(false);
			}
			nature.setSnapToGridSelection(value);
		} else if ("snapToObj".equals(propertyName)) { //$NON-NLS-1$
			if (value) {
				nature.setSnapToGridSelection(false);
			}
			nature.setSnapToGeometrySelection(value);
		}
		try {
			nature.getPreferenceStore().save();
		} catch (IOException e) {
			IStatus status = new Status(IStatus.ERROR,
					MaskatUIPlugin.PLUGIN_ID, 0,
					e.getMessage() == null ? "" : e.getMessage(), e); //$NON-NLS-1$
			MaskatUIPlugin.log(status);
		}
	}

	/**
	 * グリッドサイズをプレファレンスストアに保存します
	 * 
	 * @param size グリッドサイズ
	 */
	private void storeGridSize(int size) {
		MaskatNature nature = MaskatNature.getNature(project);
		nature.setGridSize(size);
		try {
			nature.getPreferenceStore().save();
		} catch (IOException e) {
			IStatus status = new Status(IStatus.ERROR,
					MaskatUIPlugin.PLUGIN_ID, 0,
					e.getMessage() == null ? "" : e.getMessage(), e); //$NON-NLS-1$
			MaskatUIPlugin.log(status);
		}
	}

    /**
     * {@inheritDoc}
     */
	public void doSave(IProgressMonitor monitor) {
		/*
		 * 保存処理中はsavingフラグを立ててresourceChangedリスナ
		 * の処理を行わないようにします。
		 */
		try {
			saving = true;
			data.save();
			getCommandStack().markSaveLocation();
			firePropertyChange(IEditorPart.PROP_DIRTY);
		} catch (CoreException e) {
			ErrorDialog.openError(getSite().getShell(), Messages.getString(
					"layout.editor.save.error.title"), //$NON-NLS-1$
					Messages.getString("layout.editor.save.msg.error"), //$NON-NLS-1$
							e.getStatus());
			MaskatUIPlugin.log(e.getStatus());
		} catch (Exception e) {
			IStatus status = new Status(IStatus.ERROR,
					MaskatUIPlugin.PLUGIN_ID, 0, e.getMessage() == null ? ""
							: e.getMessage(), e);
			MessageDialog.openError(getSite().getShell(), Messages.getString(
					"layout.editor.save.error.title"), //$NON-NLS-1$
					Messages.getString("layout.editor.save.msg.error")); //$NON-NLS-1$
			MaskatUIPlugin.log(status);
		} finally {
			saving = false;
		}
	}

    /**
     * {@inheritDoc}
     */
	public void doSaveAs() {
	}

    /**
     * {@inheritDoc}
     */
	public boolean isDirty() {
		return getCommandStack().isDirty();
	}

    /**
     * {@inheritDoc}
     */
	public boolean isSaveAsAllowed() {
		return false;
	}

    /**
     * {@inheritDoc}
     */
	public void commandStackChanged(EventObject event) {
		super.commandStackChanged(event);
		firePropertyChange(IEditorPart.PROP_DIRTY);
	}

    /**
     * {@inheritDoc}
     */
	protected void configureGraphicalViewer() {
		super.configureGraphicalViewer();

		GraphicalViewer viewer = getGraphicalViewer();
		viewer.setEditPartFactory(new ChainedEditPartFactory());
	}

    /**
     * {@inheritDoc}
     */
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException {
		// Workaround for bug #226, #299:
		// All editors in a multi-page editor should have key binding service
		// to switch shortcut keys.
		site.getKeyBindingService();

		super.init(site, input);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
	}

    /**
     * {@inheritDoc}
     */
	protected void setInput(IEditorInput input) {
		super.setInput(input);

		IFile layoutXmlFile = (IFile) input.getAdapter(IFile.class);
		if (layoutXmlFile != null) {
			setPartName(layoutXmlFile.getName());
			project = layoutXmlFile.getProject();
		}

		this.data = new MaskatResources(layoutXmlFile);
		try {
			data.load();
			updatePaletteEntry();
			
		} catch (CoreException e) {
			MaskatUIPlugin.log(new Status(IStatus.ERROR,
					MaskatUIPlugin.PLUGIN_ID, 0, Messages.getString(
							"layout.editor.load.msg.error"), e)); //$NON-NLS-1$
		} catch (Exception e) {
			IStatus status = new Status(IStatus.ERROR,
					MaskatUIPlugin.PLUGIN_ID, 0,
							e.getMessage() == null ? "" : e.getMessage(), e);
			MaskatUIPlugin.log(status);
		}
	}

	private void updatePaletteEntry() {
		customizer.setProject(project);
		MaskatNature nature = MaskatNature.getNature(project);
		List plugins = nature.getInstallWidgetPlugins();
		String paletteEntries = nature.getPaletteEntries();
		customizer.updatePaletteEntry(plugins, paletteEntries);		
	}
	
    /**
     * {@inheritDoc}
     */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if (this.equals(getSite().getPage().getActiveEditor())
				|| this.equals(part)) {
			updateActions(getSelectionActions());
		}
	}

    /**
     * {@inheritDoc}
     */
	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		super.dispose();
	}

    /**
     * {@inheritDoc}
     */
	public Object getAdapter(Class type) {
		if (type == IPropertySheetPage.class) {
			return new TabbedEventPropertySheetPage(this);
		} else if (IContentOutlinePage.class.equals(type)) {
			outlinePage = new LayoutOutLinePage(this);
			getSelectionSynchronizer().addViewer(outlinePage.getViewer());
			return outlinePage;
		}
		return super.getAdapter(type);
	}

    /**
     * {@inheritDoc}
     */
	public String getContributorId() {
		return EDITOR_ID;
	}

    /**
     * {@inheritDoc}
     */
	protected void createActions() {
		super.createActions();
		ActionRegistry registry = getActionRegistry();

		/*
		 * 水平方向の整列アクション
		 */
		IAction action = new AlignmentAction((IWorkbenchPart) this,
				PositionConstants.LEFT);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this,
				PositionConstants.CENTER);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this,
				PositionConstants.RIGHT);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		/*
		 * 垂直方向の整列アクション
		 */
		action = new AlignmentAction((IWorkbenchPart) this,
				PositionConstants.TOP);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this,
				PositionConstants.MIDDLE);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this,
				PositionConstants.BOTTOM);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		/*
		 * コピー、カット、ペーストアクション
		 */
		action = new CopyComponentAction(this);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new CutComponentAction(this);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new PasteComponentAction(this);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		/*
		 * 削除アクション
		 */
		action = new DeleteComponentAction(this);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());
	}

    /**
     * {@inheritDoc}
     */
	protected void firePropertyChange(int property) {
		super.firePropertyChange(property);
	}

    /**
     * {@inheritDoc}
     */
	public void resourceChanged(IResourceChangeEvent event) {
		IResourceDelta delta = event.getDelta();
		if (delta != null) {
			delta = delta.findMember(data.getLayoutXMLFile().getFullPath());
		}
		if (delta != null && !saving) {
			switch (delta.getKind()) {
			case IResourceDelta.CHANGED:
				syncExecute(new Runnable() {
					public void run() {
						setInput(getEditorInput());
						getGraphicalViewer().setContents(data.getLayout());						
						if (outlinePage != null) {
							outlinePage.update();
						}
						getCommandStack().flush();
					}
				});
				break;
			case IResourceDelta.REMOVED:
				getEditorSite().getPage().closeEditor(this, false);
				break;
			}
		}
	}

	private void syncExecute(Runnable runnable) {
		Display display = Display.getCurrent();
		if (display != null && !display.isDisposed()) {
			display.syncExec(runnable);
		}
	}
}
