/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui;

import java.text.MessageFormat;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * プラグインで利用するメッセージを取得するクラスです
 * 
 * リソースバンドルから指定されたメッセージキーに割り当てられた
 * メッセージを取得します。
 * 
 * 読み込まれるプロパティファイル
 * ・jp.sf.maskat.ui/message_ja.properties (日本語)
 * ・jp.sf.maskat.ui/message.properties (英語)
 * 
 * 使用例）String message = Messages.getString(メッセージキー);
 *  
 */
public class Messages {
	
	/**
	 * バンドル名
	 */
	private static final String BUNDLE_NAME = MaskatUIPlugin.PLUGIN_ID
			+ ".messages"; //$NON-NLS-1$

	/**
	 * リソースバンドル
	 */
	private static final ResourceBundle RESOURCE_BUNDLE = ResourceBundle
			.getBundle(BUNDLE_NAME);

	/**
	 * 指定されたキーに割り当てられたメッセージを取得します
	 * 
	 * @param key メッセージキー
	 * @return メッセージキーに割り当てられたメッセージ
	 */
	public static String getString(String key) {
		try {
			return RESOURCE_BUNDLE.getString(key);
		} catch (MissingResourceException e) {
			return '!' + key + '!';
		}
	}

	/**
	 * 指定されたキーに割り振られたメッセージパターンを指定された引数で
	 * フォーマットして返却されます
	 * 
	 * 例） keyで割り振られたメッセージ: "メッセージキーは{0}を定義しています。"
	 *      arguments: Object[] {"messageId"};
	 *      
	 *      以下のメッセージが返却されます。
	 *      メッセージキーはmessageIdを定義しています。
	 * 
	 * @param key メッセージキー
	 * @param arguments 置き換えたいオブジェクト配列
	 * @return メッセージ
	 */
	public static String format(String key, Object[] arguments) {
		try {
			String pattern = RESOURCE_BUNDLE.getString(key);
			if (arguments == null || arguments.length == 0) {
				return pattern;
			} else {
				return MessageFormat.format(pattern, arguments);
			}
		} catch (MissingResourceException e) {
			return '!' + key + '!';
		}
	}

	/**
	 * メッセージパターンを指定された引数でフォーマットして返却されます
	 * 
	 * @param key メッセージキー
	 * @param argument 置き換えたいオブジェクト配列
	 * @return メッセージ
	 */
	public static String format(String key, boolean argument) {
		return format(key, Boolean.valueOf(argument));
	}

	/**
	 * メッセージパターンを指定された引数でフォーマットして返却されます
	 * 
	 * @param key メッセージキー
	 * @param argument 置き換えたい値
	 * @return メッセージ
	 */
	public static String format(String key, short argument) {
		return format(key, new Short(argument));
	}

	/**
	 * メッセージパターンを指定された引数でフォーマットして返却されます
	 * 
	 * @param key メッセージキー
	 * @param argument 置き換えたい値
	 * @return メッセージ
	 */
	public static String format(String key, int argument) {
		return format(key, new Integer(argument));
	}

	/**
	 * メッセージパターンを指定された引数でフォーマットして返却されます
	 * 
	 * @param key メッセージキー
	 * @param argument 置き換えたい値
	 * @return メッセージ
	 */
	public static String format(String key, long argument) {
		return format(key, new Long(argument));
	}

	/**
	 * メッセージパターンを指定された引数でフォーマットして返却されます
	 * 
	 * @param key メッセージキー
	 * @param argument 置き換えたい値
	 * @return メッセージ
	 */
	public static String format(String key, float argument) {
		return format(key, new Float(argument));
	}

	/**
	 * メッセージパターンを指定された引数でフォーマットして返却されます
	 * 
	 * @param key メッセージキー
	 * @param argument 置き換えたい値
	 * @return メッセージ
	 */
	public static String format(String key, double argument) {
		return format(key, new Double(argument));
	}

	/**
	 * メッセージパターンを指定された引数でフォーマットして返却されます
	 * 
	 * @param key メッセージキー
	 * @param argument 置き換えたい値
	 * @return メッセージ
	 */
	public static String format(String key, Object argument) {
		return format(key, new Object[] { argument });
	}

	/**
	 * コンストラクタです（シングルトン）
	 */
	private Messages() {
		// NOP
	}
}
