/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui;

import java.util.ArrayList;
import java.util.List;

import jp.sf.maskat.core.event.Event;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 *  マスカットプロジェクトのネイチャークラスです
 *  
 *  EventEditorContributorインターフェースであるネイチャが存在する場合
 *  外部ハンドラ実装の呼び出しが行えるようにプロジェクトに登録します。
 *  
 *  プロジェクトスコープのプリファレンスストアに以下の情報を格納しています
 *  ・フレームワークパス　(FRAMEWORK_PATH)
 *  ・グリッド表示の有無 （GRID_SELECTION）
 *  ・グリッドに合わせる （SNAPTOGRID_SELECTION）
 *  ・オブジェクトに合わせる （SNAPTOGEOMETRY_SELECTION）
 *  ・グリッドサイズ （GRID_SIZE）
 */
public class MaskatNature implements IProjectNature {

	/**
	 * マスカットネイチャーID
	 **/
	public static final String NATURE_ID = MaskatUIPlugin.PLUGIN_ID
			+ ".maskatNature"; //$NON-NLS-1$

	/**
	 * プレファレンスストアで利用する「フレームワークパス」のキー
	 */
	public static final String FRAMEWORK_PATH = "frameworkPath"; //$NON-NLS-1$

	/**
	 * 「フレームワークパス」のデフォルト値
	 */
	public static final String FRAMEWORK_PATH_DEFAULT = "./maskat"; //$NON-NLS-1$

	/**
	 * プレファレンスストアで利用する「グリッド表示の有無」のキー
	 */
	public static final String GRID_SELECTION = "gridSelection"; //$NON-NLS-1$

	/**
	 * 「グリッド表示の有無」のデフォルト値
	 */
	public static final String GRID_SELECTION_DEFAULT = "false"; //$NON-NLS-1$

	/**
	 * プレファレンスストアで利用する「グリッドに合わせる」のキー
	 */
	public static final String SNAPTOGRID_SELECTION = "snapToGridSelection"; //$NON-NLS-1$

	/**
	 * 「グリッドに合わせる」のデフォルト値
	 */
	public static final String SNAPTOGRID_SELECTION_DEFAULT = "false"; //$NON-NLS-1$

	/**
	 * プレファレンスストアで利用する「オブジェクトに合わせる」のキー
	 */
	public static final String SNAPTOGEOMETRY_SELECTION = "snapToGeometrySelection"; //$NON-NLS-1$

	/**
	 * 「オブジェクトに合わせる」のデフォルト値
	 */
	public static final String SNAPTOGEOMETRY_SELECTION_DEFAULT = "false"; //$NON-NLS-1$

	/**
	 * プレファレンスストアで利用する「グリッドサイズ」のキー
	 */
	public static final String GRID_SIZE = "gridSize"; //$NON-NLS-1$

	/**
	 * 「グリッドサイズ」のデフォルト値
	 */
	public static final String GRID_SIZE_DEFAULT = "15"; //$NON-NLS-1$

	/**
	 *  インストールされた部品プラグインリストのキー
	 */
	public static final String INSTALL_WIDGET_PLUGINS = "installWidgetPlugins";

	/**
	 * パレット表示状態格納キー 
	 */
	public static final String PALETTE_ENTRIES = "paleteEntries"; //$NON-NLS-1$
	
	/**
	 * このネイチャのプロジェクト
	 */
	private IProject project;

	/**
	 * プレファレンスストア (プロジェクトスコープ）
	 */
	private IPersistentPreferenceStore store;

	/**
	 * イベントエディタコントリビュータ
	 */
	private EventEditorContributor contributor = null;
	
	/**
	 * 指定されたプロジェクトにマスカットネイチャーを追加します
	 * 
	 * EventEditorContributorインターフェースであるNatureが存在する場合
	 * そのネイチャーIDをプロジェクトに登録します。
	 * 
	 * @param project プロジェクト
	 * @param monitor 進捗モニタ
	 * @throws CoreException 実行時エラーが発生した場合
	 */
	public static void addNature(IProject project, IProgressMonitor monitor)
			throws CoreException {
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}
		if (monitor.isCanceled() || project == null) {
			return;
		}
		IExtensionPoint point = Platform.getExtensionRegistry()
				.getExtensionPoint("org.eclipse.core.resources.natures"); //$NON-NLS-1$
		IExtension[] extensions = point.getExtensions();
		for (int i = 0; i < extensions.length; i++) {
			String[] ids = getEventEditorContributors(extensions[i]);
			for (int j = 0; j < ids.length; j++) {
				if (!project.hasNature(ids[j])) {
					IProjectDescription description = project.getDescription();
					String[] oldIds = description.getNatureIds();
					String[] newIds = new String[oldIds.length + 1];
					System.arraycopy(oldIds, 0, newIds, 0, oldIds.length);
					newIds[oldIds.length] = ids[j];
					description.setNatureIds(newIds);
					project.setDescription(description, monitor);
				}
			}
		}
		if (!project.hasNature(NATURE_ID)) {
			IProjectDescription description = project.getDescription();
			String[] oldIds = description.getNatureIds();
			String[] newIds = new String[oldIds.length + 1];
			System.arraycopy(oldIds, 0, newIds, 0, oldIds.length);
			newIds[oldIds.length] = NATURE_ID;
			description.setNatureIds(newIds);
			project.setDescription(description, monitor);
			project.refreshLocal(IResource.DEPTH_INFINITE, monitor);
		}
	}

	/**
	 * EventEditorContributorインターフェースを持っているネイチャーIDの
	 * 配列を取得します。
	 * 
	 * @param extension ネイチャのIExtension
	 * @return ネイチャーIDの配列
	 * @throws CoreException 実行時エラーが発生した場合
	 */
	private static String[] getEventEditorContributors(IExtension extension)
			throws CoreException {

		ArrayList netureIds = new ArrayList();
		String natureId = extension.getUniqueIdentifier();
		IConfigurationElement[] elements = extension
				.getConfigurationElements();
		for (int i = 0; i < elements.length; i++) {
			IConfigurationElement element = elements[i];
			if ("runtime".equals(element.getName())) { //$NON-NLS-1$
				IConfigurationElement[] children = element.getChildren();
				for (int j = 0; j < children.length; j++) {
					if ("run".equals(children[j].getName())) {
						Object object = children[j]
								.createExecutableExtension("class"); //$NON-NLS-1$
						if (object instanceof EventEditorContributor) {
							netureIds.add(natureId);
							break;
						}
					}
				}
			}
		}
		return (String[]) netureIds.toArray(new String[0]);
	}

	/**
	 * マスカットネイチャーを削除します
	 * 
	 * @param project プロジェクト
	 * @param monitor 進捗モニタ
	 * @throws CoreException 実行時エラーが発生した場合
	 */
	public static void removeNature(IProject project, IProgressMonitor monitor)
			throws CoreException {
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}
		if (monitor.isCanceled() || project == null) {
			return;
		}

		if (project.hasNature(NATURE_ID)) {
			IProjectDescription description = project.getDescription();
			String[] oldIds = description.getNatureIds();
			String[] newIds = new String[oldIds.length - 1];
			for (int i = 0; i < oldIds.length; i++) {
				if (NATURE_ID.equals(oldIds[i])) {
					System.arraycopy(oldIds, 0, newIds, 0, i);
					System.arraycopy(oldIds, i + 1, newIds, i, newIds.length
							- i);
					break;
				}
			}
			description.setNatureIds(newIds);
			project.setDescription(description, monitor);
			project.refreshLocal(IResource.DEPTH_INFINITE, monitor);
		}
	}

	/**
	 * マスカットネイチャーを取得します
	 * 
	 * @param project プロジェクト
	 * @return マスカットネイチャー
	 */
	public static MaskatNature getNature(IProject project) {
		try {
			if (project != null && project.hasNature(NATURE_ID)) {
				return (MaskatNature) project.getNature(NATURE_ID);
			}
		} catch (CoreException e) {
			MaskatUIPlugin.log(e.getStatus());
		}
		return null;
	}

    /**
     * {@inheritDoc}
     */
	public IProject getProject() {
		return project;
	}

    /**
     * {@inheritDoc}
     */
	public void setProject(IProject project) {
		if (project == null) {
			return;
		}
		this.project = project;

		IScopeContext scope = new ProjectScope(project);
		store = new ScopedPreferenceStore(scope, MaskatUIPlugin.PLUGIN_ID);
		store.setDefault(FRAMEWORK_PATH, FRAMEWORK_PATH_DEFAULT);
		store.setDefault(GRID_SELECTION, GRID_SELECTION_DEFAULT);
		store.setDefault(SNAPTOGRID_SELECTION, SNAPTOGRID_SELECTION_DEFAULT);
		store.setDefault(SNAPTOGEOMETRY_SELECTION,
				SNAPTOGEOMETRY_SELECTION_DEFAULT);
		store.setDefault(GRID_SIZE, GRID_SIZE_DEFAULT);
		store.setDefault(INSTALL_WIDGET_PLUGINS, "");
		store.setDefault(PALETTE_ENTRIES, "");
		
		contributor = null;
		try {
			IProjectDescription description = this.project.getDescription();
			String[] natureIds = description.getNatureIds();
			for (int i = 0; i < natureIds.length; i++) {
				if (!(NATURE_ID.equals(natureIds[i]))) {
					IProjectNature nature = this.project
							.getNature(natureIds[i]);
					if (nature instanceof EventEditorContributor) {
						contributor = (EventEditorContributor) nature;
						break;
					}
				}
			}
		} catch (CoreException e) {
			MaskatUIPlugin.log(e.getStatus());
		}
	}
	
	/**
	 * プレファレンスストアを取得します
	 * 
	 * @return プレファレンスストア
	 */
	public IPersistentPreferenceStore getPreferenceStore() {
		return store;
	}

    /**
     * {@inheritDoc}
     */
	public void configure() throws CoreException {
		// NOP
	}

    /**
     * {@inheritDoc}
     */
	public void deconfigure() throws CoreException {
		// NOP
	}

	/**
	 * プレファレンスストアから「フレームワークパス」を取得します
	 * 
	 * @return フレームワークパス
	 */
	public IPath getMaskatFrameworkPath() {
		return new Path(store.getString(FRAMEWORK_PATH));
	}

	/**
	 * プレファレンスストアへ「フレームワークパス」を設定します
	 * 
	 * @param path フレームワークパス
	 */
	public void setMaskatFrameworkPath(IPath path) {
		store.setValue(FRAMEWORK_PATH, path.toString());
	}

	/**
	 * プレファレンスストアから「グリッド表示の有無」を取得します
	 * 
	 * @return 「グリッド表示の有無」の設定値
	 */
	public boolean getGridSelection() {
		return new Boolean(store.getString(GRID_SELECTION)).booleanValue();
	}

	/**
	 * プレファレンスストアへ「グリッド表示の有無」を設定します
	 * 
	 * @param selection 「グリッド表示の有無」の設定値
	 * 
	 */
	public void setGridSelection(boolean selection) {
		store.setValue(GRID_SELECTION, String.valueOf(selection));
	}

	/**
	 * プレファレンスストアから「グリッドに合わせる」を取得します
	 * 
	 * @return 「グリッドに合わせる」の設定値
	 */
	public boolean getSnapToGridSelection() {
		return new Boolean(store.getString(SNAPTOGRID_SELECTION))
				.booleanValue();
	}

	/**
	 * プレファレンスストアへ「グリッドに合わせる」を設定します
	 * 
	 * @param selection 「グリッドに合わせる」の設定値
	 */
	public void setSnapToGridSelection(boolean selection) {
		store.setValue(SNAPTOGRID_SELECTION, String.valueOf(selection));
	}

	/**
	 * プレファレンスストアから「オブジェクトに合わせる」を取得します
	 * 
	 * @return 「オブジェクトに合わせる」の設定値
	 */
	public boolean getSnapToGeometrySelection() {
		return new Boolean(store.getString(SNAPTOGEOMETRY_SELECTION))
				.booleanValue();
	}

	/**
	 * プレファレンスストアへ「オブジェクトに合わせる」を設定します
	 * 
	 * @param selection 「オブジェクトに合わせる」の設定値
	 */
	public void setSnapToGeometrySelection(boolean selection) {
		store.setValue(SNAPTOGEOMETRY_SELECTION, String.valueOf(selection));
	}

	/**
	 * プレファレンスストアからグリッドサイズを取得します
	 * 
	 * @return グリッドサイズ
	 */
	public int getGridSize() {
		String v = store.getString(GRID_SIZE);
		if (v == null || v.length() < 1) {
			v = GRID_SIZE_DEFAULT;
		}
		return new Integer(v).intValue();
	}

	/**
	 * プレファレンスストアへグリッドサイズを設定します
	 * 
	 * @param size グリッドサイズ
	 */
	public void setGridSize(int size) {
		store.setValue(GRID_SIZE, String.valueOf(size));
	}

	/**
	 * イベントエディタコントリビュータを取得します
	 * 
	 * @return イベントエディタコントリビュータ
	 */
	public EventEditorContributor getEventEditorContributor() {
		return this.contributor;
	}

	/**
	 * イベントエディタコントリビュータを設定します
	 * 
	 * @param contributor イベントエディタコントリビュータ
	 */
	public void setEventEditorContributor(EventEditorContributor contributor) {
		this.contributor = contributor;
	}

	/**
	 * 外部イベントハンドラ実装を呼び出します
	 * 
	 * @param event イベント
	 * @throws CoreException 実行時エラーが発生した場合
	 */
	public void openEventHandlerImpl(Event event) throws CoreException {
		if (this.contributor != null) {
			this.contributor.openEventHandlerImpl(event);
		}
	}

	/**
	 * プレファレンスストアからインストールされた部品プラグインリストを取得します
	 * 
	 * @return インストールされた部品プラグインの配列
	 */
	public List getInstallWidgetPlugins() {
		String plugins = store.getString(INSTALL_WIDGET_PLUGINS);
		return plugins == null ? new ArrayList() :
				java.util.Arrays.asList(plugins.split(","));
	}

	/**
	 * プレファレンスストアへインストールされた部品プラグインリストを設定します
	 * 
	 * @param インストールされた部品プラグインリスト
	 */
	public void setInstallWidgetPlugins(String param) {
		store.setValue(INSTALL_WIDGET_PLUGINS, param);
	}

	/**
	 * プレファレンスストアからパレット設定情報を取得します
	 * 
	 * @return パレット設定情報
	 */
	public String getPaletteEntries() {
		return store.getString(PALETTE_ENTRIES);
	}

	/**
	 * プレファレンスストアへパレット設定情報を設定します
	 * 
	 * @param パレット設定情報
	 */
	public void setPaletteEntries(String entries) {
		store.setValue(PALETTE_ENTRIES, entries);
	}
}
