/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui;

import java.net.URI;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ContainerSelectionDialog;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.ui.internal.ide.filesystem.FileSystemSupportRegistry;

/**
 * マスカットプロパティ画面に表示されるマスカットフレームワーク部分の
 * 描画および制御を行うクラスです。
 */
public class MaskatFrameworkDialogArea extends Composite {

	/**
	 * マスカットフレームワークのパス
	 */
	private Text maskatFWPath;

	/**
	 * 現設定の有効性 （有効な値が定義されている場合、true)
	 */
	private boolean valid;

	/**
	 * 現設定のエラーメッセージ（無効な値が定義されている場合）
	 */
	private String errorMessage;

	/**
	 * カレントプロジェクト
	 */
	private IProject selectProject;

	/**
	 * コンストラクタです。
	 * 
	 * @param project カレントプロジェクト
	 * @param parent 親コンポジット
	 * @param style 表示スタイル
	 */
	public MaskatFrameworkDialogArea(IProject project, Composite parent,
			int style) {
		super(parent, style);
		selectProject = project;
		createControl();
	}

	/**
	 * マスカットフレームワーク部分の画面作成
	 */
	private void createControl() {
		setLayout(new GridLayout());
		setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		Group fwGroup = new Group(this, SWT.NONE);
		GridLayout layout = new GridLayout(4, false);
		layout.marginLeft = 5;
		fwGroup.setLayout(layout);
		fwGroup.setLayoutData(new GridData(GridData.FILL_BOTH));
		fwGroup.setText(Messages.getString("property.title")); //$NON-NLS-1$
		Label dirLabel = new Label(fwGroup, SWT.NULL);
		dirLabel.setText(Messages.getString(
				"property.label.framework")); //$NON-NLS-1$		

		maskatFWPath = new Text(fwGroup, SWT.BORDER);
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;
		maskatFWPath.setLayoutData(gridData);
		maskatFWPath.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setValid(validatePage());
			}
		});
		Button button = new Button(fwGroup, SWT.PUSH);
		button.setText(Messages.getString("property.browse")); //$NON-NLS-1$
		button.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleBrowse();
			}
		});
	}

	/**
	 * フレームワークパスの「参照」ボタンが押されたとき呼び出され
	 * フォルダ選択ダイアログを表示します
	 */
	private void handleBrowse() {
		ContainerSelectionDialog dialog = new ContainerSelectionDialog(
				getShell(), ResourcesPlugin.getWorkspace().getRoot(), true,
				Messages.getString("property.selectdialog.title")); //$NON-NLS-1$
		if (dialog.open() == Window.OK) {
			Object[] result = dialog.getResult();
			if (result.length == 1) {
				maskatFWPath.setText(((Path) result[0]).toString());
			}
		}
	}

	/**
	 * 現設定の有効性を取得します
	 * 
	 * @return 設定された値が有効な場合 true
	 *          無効な場合 false
	 */
	public boolean isValid() {
		return valid;
	}

	/**
	 * 現設定の有効性を設定します
	 * 
	 * @param valid 有効な場合 true, 無効な場合 false
	 */
	public void setValid(boolean valid) {
		this.valid = valid;
	}

	/**
	 * 現設定でのエラーメッセージを取得します
	 * 
	 * @return エラーメッセージ
	 */
	public String getErrorMessage() {
		return errorMessage;
	}

	/**
	 * 現設定でのエラーメッセージを設定します
	 * 
	 * @param newMessage エラーメッセージ
	 */
	public void setErrorMessage(String newMessage) {
		this.errorMessage = newMessage;
	}

	/**
	 * 現設定が正しい値なのか判定します
	 * 
	 * @return 有効な場合 true, 無効な場合 false
	 */
	private boolean validatePage() {
		String path = maskatFWPath.getText();
		if (path == null || path.length() == 0) {
			setErrorMessage(Messages.getString(
					"property.msg.error.notpath")); //$NON-NLS-1$
			maskatFWPath.setFocus();
			return false;
		}
		IWorkspace workspace = IDEWorkbenchPlugin.getPluginWorkspace();
		URI fwUrl = FileSystemSupportRegistry.getInstance()
				.getDefaultConfiguration().getContributor().getURI(path);
		IStatus fwLocationStatus = workspace.validateProjectLocationURI(
				selectProject, fwUrl);
		if (!fwLocationStatus.isOK()) {
			setErrorMessage(fwLocationStatus.getMessage());
			maskatFWPath.setFocus();
			return false;
		}
		setErrorMessage(null);
		return true;
	}

	/**
	 * マスカットフレームワークパスを設定します
	 * 
	 * @param path マスカットフレームワークパス
	 */
	public void setMaskatFWPath(String path) {
		maskatFWPath.setText(path);
	}

	/**
	 * マスカットフレームワークパスを設定します
	 * 
	 * @param path マスカットフレームワークパス
	 */
	public void setMaskatFWPath(IPath path) {
		maskatFWPath.setText(path.toString());
	}

	/**
	 * マスカットフレームワークパスを取得します
	 * 
	 * @return マスカットフレームワークパス
	 */
	public String getMaskatFWPath() {
		return maskatFWPath.getText();
	}
}
