/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.event;

import java.util.Set;

/**
 * イベント発生時の処理を定義するクラスです
 * 
 * EventDef、Componentの子要素として登録することができます。
 * EventDefの子要素： レイアウトに対するイベント処理
 * Componentの子要素： コンポーネントのイベント処理
 *
 */
public class Event extends AbstractEventElement {

	/**
	 * 非同期通信のデフォルト値
	 */
	public static final boolean ASYNC_DEFAULT = true;

	/**
	 * イベントタイプのデフォルト値
	 */
	public static final String TYPE_DEFAULT = "remote";

	/**
	 * 通信タイムアウト時間のデフォルト値
	 */
	public static final int TIMEOUT_DEFAULT = -1;

	/**
	 * イベントID
	 * onclick, onblorなどのイベント名
	 */
	private String id;

	/**
	 * 参照用イベント定義名
	 * EventRef要素への参照
	 */
	private String ref;

	/**
	 * イベント発生直後に呼び出される関数名
	 */
	private String start;

	/**
	 * HTTP要求送信直前に呼び出される関数名
	 */
	private String before;

	/**
	 * 応答メッセージ受信直後に呼び出される関数名
	 */
	private String after;

	/**
	 * イベント処理終了直前に呼び出される関数
	 */
	private String finish;

	/**
	 * 処理の種別
	 * remote（リモート処理） : サーバ通信を伴う処理
	 * local（ローカル処理） : サーバ通信を伴わない処理
	 */
	private String type = TYPE_DEFAULT;

	/**
	 * 非同期通信
	 * true : 有効
	 * false : 無効 (同期通信)
	 */
	private boolean async = ASYNC_DEFAULT;

	/**
	 * 通信リクエスト先のURL
	 * URLはコンテナHTMLからの相対パスを指定する
	 */
	private String remoteUrl;

	/**
	 * 通信タイムアウト時間[ms]
	 */
	private int timeout = TIMEOUT_DEFAULT;

	/**
	 * タイムアウト発生時に呼び出される関数名
	 */
	private String onTimeoutError;

	/**
	 * レイアウト定義XMLに定義したcomfirmDialogの
	 * コンポーネント名を定義する
	 */
	private String confirmDialog;

	/**
	 * レイアウト定義XMLに定義したendDialogの
	 * コンポーネント名を定義する
	 */
	private String endDialog;

	/**
	 * 独自マーシャル処理を行う関数名
	 */
	private String marshal;

	/**
	 * 独自アンマーシャル処理を行う関数名
	 */
	private String unmarshal;

	/**
	 * イベント無効フラグ
	 */
	private boolean enable = true;

	/**
	 * デフォルトコンストラクタです
	 */
	public Event() {
	}

	/**
	 * コンストラクタ
	 * 
	 * @param id イベントID
	 */
	public Event(String id) {
		this.id = id;
		this.enable = false;
	}

	/**
	 * 応答メッセージ受信直後に呼び出される関数名を取得します
	 * 
	 * @return 応答メッセージ受信直後に呼び出される関数名
	 */
	public String getAfter() {
		return after;
	}

	/**
	 * 応答メッセージ受信直後に呼び出される関数名を設定します
	 * 
	 * @param after 応答メッセージ受信直後に呼び出される関数名
	 */
	public void setAfter(String after) {
		this.after = after;
	}

	/**
	 * 非同期通信の有無を判定します
	 * 
	 * @return 非同期通信の場合、trueが返却されます。
	 */
	public boolean isAsync() {
		return async;
	}

	/**
	 * 非同期通信の有無を設定します
	 * 
	 * @param async 非同期通信の場合、trueを設定します
	 */
	public void setAsync(boolean async) {
		this.async = async;
	}

	/**
	 * HTTP要求送信直前に呼び出される関数名を取得します
	 * 
	 * @return HTTP要求送信直前に呼び出される関数名
	 */
	public String getBefore() {
		return before;
	}

	/**
	 * HTTP要求送信直前に呼び出される関数名を設定します
	 * 
	 * @param before HTTP要求送信直前に呼び出される関数名
	 */
	public void setBefore(String before) {
		this.before = before;
	}

	/**
	 * イベント処理終了直前に呼び出される関数を取得します
	 * 
	 * @return イベント処理終了直前に呼び出される関数
	 */
	public String getFinish() {
		return finish;
	}

	/**
	 * イベント処理終了直前に呼び出される関数を設定します
	 * 
	 * @param finish イベント処理終了直前に呼び出される関数
	 */
	public void setFinish(String finish) {
		this.finish = finish;
	}

	/**
	 * イベント発生直後に呼び出される関数名を取得します
	 * 
	 * @return イベント発生直後に呼び出される関数名
	 */
	public String getStart() {
		return start;
	}

	/**
	 * イベント発生直後に呼び出される関数名を設定します
	 * 
	 * @param start イベント発生直後に呼び出される関数名
	 */
	public void setStart(String start) {
		this.start = start;
	}

	/**
	 * 処理の種別を取得します
	 * @return リモート通信の場合には"remote"、
	 *          ローカル通信の場合には"local"が返却されます。
	 */
	public String getType() {
		return type;
	}

	/**
	 * 処理の種類を設定します
	 * 
	 * @param type リモート通信の場合には"remote"、
	 *              ローカル通信の場合には"local"を設定します。
	 */
	public void setType(String type) {
		this.type = type;
	}

	/**
	 * イベントIDを取得します
	 * 
	 * @return イベントID
	 */
	public String getId() {
		return id;
	}

	/**
	 * イベントIDを設定します
	 * 
	 * @param id 設定したいイベントID
	 */
	public void setId(String id) {
		this.id = id;
	}

	/**
	 * EventDefへの参照を取得します
	 * 
	 * @return EventDefへの参照
	 */
	public String getRef() {
		return ref;
	}

	/**
	 * EventDefへの参照を設定します
	 * 
	 * @param ref 設定したいEventDefへの参照
	 */
	public void setRef(String ref) {
		this.ref = ref;
	}

	/**
	 * 通信リクエスト先のURLを取得します
	 * 
	 * @return 通信リクエスト先のURL
	 */
	public String getRemoteUrl() {
		return remoteUrl;
	}

	/**
	 * 通信リクエスト先のURLを設定します
	 * 
	 * @param remoteUrl 通信リクエスト先のURL
	 */
	public void setRemoteUrl(String remoteUrl) {
		this.remoteUrl = remoteUrl;
	}

	/**
	 * タイムアウト発生時に呼び出される関数名を取得します
	 * 
	 * @return タイムアウト発生時に呼び出される関数名
	 */
	public String getOnTimeoutError() {
		return onTimeoutError;
	}

	/**
	 * タイムアウト発生時に呼び出される関数名を設定します
	 * 
	 * @param onTimeoutError タイムアウト発生時に呼び出される関数名
	 */
	public void setOnTimeoutError(String onTimeoutError) {
		this.onTimeoutError = onTimeoutError;
	}

	/**
	 * 通信タイムアウト時間を取得します
	 * 
	 * @return タイムアウト時間 (ms)
	 */
	public int getTimeout() {
		return timeout;
	}

	/**
	 * 通信タイムアウト時間を設定します
	 * 
	 * @param timeout タイムアウト時間 (ms)
	 */
	public void setTimeout(int timeout) {
		this.timeout = timeout;
	}

	/**
	 * レイアウト定義XMLに定義したcomfirmDialogのコンポーネント名を
	 * 取得します
	 * 
	 * @return comfirmDialogのコンポーネント名
	 */
	public String getConfirmDialog() {
		return confirmDialog;
	}

	/**
	 * レイアウト定義XMLに定義したcomfirmDialogのコンポーネント名を
	 * 設定します
	 * 
	 * @param confirmDialog comfirmDialogのコンポーネント名
	 */
	public void setConfirmDialog(String confirmDialog) {
		this.confirmDialog = confirmDialog;
	}

	/**
	 * レイアウト定義XMLに定義したendDialogのコンポーネント名を
	 * 取得します
	 * 
	 * @return endDialogのコンポーネント名
	 */
	public String getEndDialog() {
		return endDialog;
	}

	/**
	 * レイアウト定義XMLに定義したendDialogのコンポーネント名を
	 * 設定します
	 * 
	 * @param endDialog endDialogのコンポーネント名
	 */
	public void setEndDialog(String endDialog) {
		this.endDialog = endDialog;
	}

	/**
	 * 独自マーシャル処理を行う関数名を取得します
	 * 
	 * @return 独自マーシャル処理を行う関数名
	 */
	public String getMarshal() {
		return marshal;
	}

	/**
	 * 独自マーシャル処理を行う関数名を設定します
	 * 
	 * @param marshal 独自マーシャル処理を行う関数名
	 */
	public void setMarshal(String marshal) {
		this.marshal = marshal;
	}

	/**
	 * 独自アンマーシャル処理を行う関数名を取得します
	 * 
	 * @return 独自アンマーシャル処理を行う関数名
	 */
	public String getUnmarshal() {
		return unmarshal;
	}

	/**
	 * 独自アンマーシャル処理を行う関数名を設定します
	 * 
	 * @param unmarshal 独自アンマーシャル処理を行う関数名
	 */
	public void setUnmarshal(String unmarshal) {
		this.unmarshal = unmarshal;
	}

	/**
	 * このイベントが無効か判定します
	 * 
	 * @return このイベントが無効な場合、trueが返却されます
	 */
	public boolean isEnable() {
		return enable;
	}

	/**
	 * このイベントの有効無効を設定します
	 * 
	 * @param enable このイベントを無効にする場合には true、
	 *                有効にする場合には false を設定します。
	 */
	public void setEnable(boolean enable) {
		this.enable = enable;
	}

	/**
	 * このイベントに登録されている Param 要素を取得します
	 * （Param要素は要求メッセージ情報を持つ要素です）
	 * 
	 * @return Param 要素
	 */
	public Param getParam() {
		Param param = (Param) getChildByTypeIdx(Param.class, 0);
		if (param == null) {
			param = new Param(false);
			addChild(param);
		}
		return param;
	}

	/**
	 * このイベントに登録されている Result 要素を取得します
	 * （Result要素は応答メッセージ情報を持つ要素です）
	 * 
	 * @return Result 要素
	 */
	public Result getResult() {
		Result result = (Result) getChildByTypeIdx(Result.class, 0);
		if (result == null) {
			result = new Result(false);
			addChild(result);
		}
		return result;
	}

	/**
	 * このイベントに登録されている以下のオブジェクトを取得する
	 * ・Source要素の参照オブジェクト名（obj）
	 * ・Target要素の出力オブジェクト名（out）
	 * 
	 * @param resultSet この引数に対してaddメソッドでオブジェクトが
	 *                   追加される
	 */
	public void getRelatedObjNames(Set resultSet) {
		Param param = (Param) getChildByTypeIdx(Param.class, 0);
		if (param != null) {
			Source[] sources = param.getSources();
			for (int i = 0; i < sources.length; i++) {
				if (sources[i].getObj() != null) {
					resultSet.add(sources[i].getObj());
				}
			}
		}

		Result result = (Result) getChildByTypeIdx(Result.class, 0);
		if (result != null) {
			Target[] targets = result.getTargets();
			for (int i = 0; i < targets.length; i++) {
				if (targets[i].getOut() != null) {
					resultSet.add(targets[i].getOut());
				}
			}
		}
	}

    /**
     * {@inheritDoc}
     */
	public String toString() {
		return "";
	}
}
