/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.betwixt;

import jp.sf.maskat.core.MaskatElement;
import jp.sf.maskat.core.event.Source;
import jp.sf.maskat.core.event.Target;

import org.apache.commons.betwixt.AttributeDescriptor;
import org.apache.commons.betwixt.ElementDescriptor;
import org.apache.commons.betwixt.strategy.ValueSuppressionStrategy;

/**
 * メッセージスキーマLの要素に定義されている属性の出力制御を行うクラスです。
 * <p>
 * 必須属性、属性の存在有無などから出力の有無を制御しています。
 */
public class MessageSchemaValueSuppressionStrategy extends
		ValueSuppressionStrategy {
	
	/**
	 * sourceノードフラグ
	 * 以下の条件に該当するとき無効な属性と識別します
	 * ・node属性  -  「node属性が存在しなくChildNode属性が存在する」
	 * ・type属性  -  「値がxsd:stringでありbindが存在する」
	 * ・childNode属性 - 「nodeとbindが存在し、childNodeが存在しない」
	 * ・maxOccurs属性 - 「bindとchildNodeが存在しなくnodeが存在する」
	 */
	private static final String FLAG_SOURCE = "flagSource";

	/**
	 * sourceノードフラグ (CHILDNODE_NOT_NULL)
	 * 以下の条件に該当するとき無効なエレメントと識別します
	 * ・bindが存在しない場合
	 * ・childNodeが存在しなくnodeが存在する場合
	 * ・childNodeが存在しnodeが存在しない場合
	 */
	private static final String FLAG_SOURCE_NODE_CHILDNODE_NOT_NULL =
			"flagSourceNodeAndChildNodeAreNotNull";

	/**
	 * sourceノードフラグ (NOT_NULL_CHILDNODE_NULL)
	 * 以下の条件に該当するとき無効なエレメントと識別します
	 * ・bindが存在しない場合
	 * ・childNodeが存在しnodeが存在しない場合
	 * ・childNodeとnodeが存在する場合
	 */
	private static final String FLAG_SOURCE_NODE_NOT_NULL_CHILDNODE_NULL =
			"flagSourceNodeIsNotNullChildNodeIsNull";

	/**
	 * sourceノードフラグ (NULL_CHILDNODE_NOT_NULL)
	 * 以下の条件に該当するとき無効なエレメントと識別します
	 * ・bindが存在しない場合
	 * ・childNodeが存在しなくnodeが存在する場合
	 * ・childNodeとnodeが存在する場合
	 */
	private static final String FLAG_SOURCE_NODE_NULL_CHILDNODE_NOT_NULL =
			"flagSourceNodeIsNullChildNodeIsNotNull";

	/**
	 * targetノードフラグ
	 * 以下の条件に該当するとき無効な属性と識別します
	 * ・in属性  -  「in属性が存在しなくinkey属性が存在する」
	 * ・type属性  -  「値がxsd:stringでありbindが存在する」
	 * ・inkey属性 - 「inが存在し、bindが存在しない」
	 * ・maxOccurs属性 - 「bindとinkeyが存在しなくinが存在する」
	 */
	private static final String FLAG_TARGET = "flagTarget";

	/**
	 * targetノードフラグ (IN_KEY_NOT_NULL)
	 * 以下の条件に該当するとき無効なエレメントと識別します
	 * ・bindが存在しない場合
	 * ・inが存在しない場合
	 */
	private static final String FLAG_TARGET_IN_INKEY_NOT_NULL =
			"flagTargetInIsNotNullInkeyIsNotNull";

	/**
	 * targetノードフラグ (IN_NULL_INKEY_NOT_NULL)
	 * 以下の条件に該当するとき無効なエレメントと識別します
	 * ・bindが存在しない場合
	 * ・inが存在する場合
	 */
	private static final String FLAG_TARGET_IN_NULL_INKEY_NOT_NULL =
			"flagTargetInIsNullInkeyIsNotNull";

	/**
	 * カレントエレメント
	 */
	private MaskatElement currentElement = null;

	/**
	 * カレントエレメントディスクリプタ
	 */
	private ElementDescriptor currentDescriptor = null;

	/**
	 * デフォルトコンストラクターです
	 *
	 */
	public MessageSchemaValueSuppressionStrategy() {
		super();
	}

	/**
     * {@inheritDoc}
     */
	public boolean suppressAttribute(AttributeDescriptor attributeDescriptor,
			String value) {
		String attributeName = attributeDescriptor.getQualifiedName();
		String propertyName = attributeDescriptor.getPropertyName();

		/*
		 * フラグ属性はそのエレメント、属性の条件が定義されており
		 * 属性自体は特に処理を行いません。
		 */
		if (isFlagAttribute(attributeName)) {
			return true;
		}

		if (currentElement != null) {
			if (currentElement instanceof Source
					&& usedFor(currentDescriptor, FLAG_SOURCE)) {
				Source source = (Source) currentElement;
				if ("name".equals(attributeName) && "node".equals(propertyName)) {
					return source.getNode() == null
							&& source.getChildNode() != null;
				}
				if ("type".equals(attributeName) && "xsd:string".equals(value)) {
					return source.getBinds().length > 0;
				}
				if ("name".equals(attributeName)
						&& "childNode".equals(propertyName)) {
					if (source.getNode() != null) {
						return source.getChildNode() == null
								|| source.getBinds().length > 0;
					}
				}
				if ("maxOccurs".equals(attributeName)) {
					return !(source.getBinds().length > 0
							&& source.getChildNode() != null && source
							.getNode() == null);
				}
			}
			if (currentElement instanceof Target
					&& usedFor(currentDescriptor, FLAG_TARGET)) {
				Target target = (Target) currentElement;
				if ("name".equals(attributeName) && "in".equals(propertyName)) {
					return target.getIn() == null && target.getInkey() != null;
				}
				if ("type".equals(attributeName) && "xsd:string".equals(value)) {
					return target.getBinds().length > 0;
				}
				if ("name".equals(attributeName)
						&& "inkey".equals(propertyName)) {
					return target.getIn() != null
							|| !(target.getBinds().length > 0);
				}
				if ("maxOccurs".equals(attributeName)) {
					return !(target.getBinds().length > 0
							&& target.getInkey() != null && target.getIn() == null);
				}
			}
		}
		return false;
	}

	/**
     * {@inheritDoc}
     */
	public boolean suppressElement(ElementDescriptor element,
			String namespaceUri, String localName, String qualifiedName,
			Object value) {

		if (value instanceof MaskatElement) {
			currentElement = (MaskatElement) value;
		}
		currentDescriptor = element;

		if (value instanceof Source) {
			Source source = (Source) value;
			if (source.getBinds().length <= 0) {
				return usedFor(element, FLAG_SOURCE_NODE_CHILDNODE_NOT_NULL)
						|| usedFor(element,
								FLAG_SOURCE_NODE_NOT_NULL_CHILDNODE_NULL)
						|| usedFor(element,
								FLAG_SOURCE_NODE_NULL_CHILDNODE_NOT_NULL);
			}

			if (source.getChildNode() == null && source.getNode() != null) {
				return usedFor(element, FLAG_SOURCE_NODE_CHILDNODE_NOT_NULL)
						|| usedFor(element,
								FLAG_SOURCE_NODE_NULL_CHILDNODE_NOT_NULL);
			}

			if (source.getChildNode() != null && source.getNode() == null) {
				return usedFor(element, FLAG_SOURCE_NODE_CHILDNODE_NOT_NULL)
						|| usedFor(element,
								FLAG_SOURCE_NODE_NOT_NULL_CHILDNODE_NULL);
			}

			if (source.getChildNode() != null && source.getNode() != null) {
				return usedFor(element,
						FLAG_SOURCE_NODE_NULL_CHILDNODE_NOT_NULL)
						|| usedFor(element,
								FLAG_SOURCE_NODE_NOT_NULL_CHILDNODE_NULL);
			}

		}
		if (value instanceof Target) {
			Target target = (Target) value;
			if (target.getBinds().length <= 0) {
				return usedFor(element, FLAG_TARGET_IN_INKEY_NOT_NULL)
						|| usedFor(element, FLAG_TARGET_IN_NULL_INKEY_NOT_NULL);
			}

			if (target.getIn() != null) {
				return usedFor(element, FLAG_TARGET_IN_NULL_INKEY_NOT_NULL);
			}

			if (target.getIn() == null) {
				return usedFor(element, FLAG_TARGET_IN_INKEY_NOT_NULL);
			}
		}
		return false;
	}

	/**
	 * フラグ属性かどうかを判定します。
	 * 
	 * @param flagAttribute　属性名
	 * @return フラグ属性の場合、trueを返します。
	 */
	private boolean isFlagAttribute(String flagAttribute) {
		return FLAG_SOURCE.equals(flagAttribute)
				|| FLAG_SOURCE_NODE_CHILDNODE_NOT_NULL.equals(flagAttribute)
				|| FLAG_SOURCE_NODE_NOT_NULL_CHILDNODE_NULL
						.equals(flagAttribute)
				|| FLAG_SOURCE_NODE_NULL_CHILDNODE_NOT_NULL
						.equals(flagAttribute)
				|| FLAG_TARGET.equals(flagAttribute)
				|| FLAG_TARGET_IN_INKEY_NOT_NULL.equals(flagAttribute)
				|| FLAG_TARGET_IN_NULL_INKEY_NOT_NULL.equals(flagAttribute);
	}

	/**
	 * 指定された属性が実際に定義されているか判定します。
	 * 
	 * @param element ElementDescriptor
	 * @param flagAttribute 属性名
	 * @return 実際に定義されている場合、trueを返します
	 */
	private boolean usedFor(ElementDescriptor element, String flagAttribute) {
		return element.getAttributeDescriptors().length > 0
				&& element.getAttributeDescriptor(flagAttribute) != null;
	}
}
