/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.betwixt;

import java.beans.IntrospectionException;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;

import jp.sf.maskat.core.event.EventElement;
import jp.sf.maskat.core.event.Param;
import jp.sf.maskat.core.event.Result;

import org.apache.commons.betwixt.XMLIntrospector;
import org.apache.commons.betwixt.io.BeanWriter;
import org.apache.commons.io.IOUtils;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * イベント定義に定義したリモートイベントの送受信スキーマを生成する
 * クラスです。
 * <p>
 * 定義されたリモートイベントのスキーマを送信、受信ごとに生成します
 * </p>
 */
public class MessageSchemaGenerator {

	/**
	 * スキーマ定義XML用 betwixt設定ファイル名
	 */
	public static final String MESSAGE_XSD_CONFIG = "xsdConfig.betwixt";

	/**
	 * デフォルトコンストラクターです。
	 */
	public MessageSchemaGenerator() {
		super();
	}

	/**
	 * Paramオブジェクトから送信メッセージスキーマを生成します。
	 * 
	 * @param param スキーマ生成を行いたいParamオブジェクト
	 * @return 送信メッセージスキーマのInputStreamを返します
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 */
	public static InputStream generateRequestXMLSchema(Param param)
			throws IntrospectionException,
					IOException,
					SAXException {
		return generateMessageXMLSchema(param);
	}

	/**
	 * Resultオブジェクトから受信メッセージスキーマを生成します。
	 * @param result スキーマ生成を行いたいResultオブジェクト
	 * @return 受信メッセージスキーマのInputStreamを返します
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 */
	public static InputStream generateResponseXMLSchema(Result result)
					throws IntrospectionException,
							IOException,
							SAXException {
		return generateMessageXMLSchema(result);
	}

	/**
	 * EventElementオブジェクトからスキーマを生成します。
	 * @param element スキーマ生成を行いたいオブジェクト
	 * @return 生成したスキーマのInputStreamを返します
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 */
	private static InputStream generateMessageXMLSchema(EventElement element)
					throws IntrospectionException,
							IOException,
							SAXException {
		StringWriter outputWriter = new StringWriter();

		outputWriter.write("<?xml version='1.0' encoding='UTF-8' ?>");
		String lineSeparator = System.getProperty("line.separator");
		outputWriter.write(lineSeparator);

		BeanWriter beanWriter = new BeanWriter(outputWriter);
		beanWriter.getXMLIntrospector().getConfiguration()
				.setAttributesForPrimitives(false);
		beanWriter.getBindingConfiguration().setMapIDs(false);
		beanWriter.getBindingConfiguration().setValueSuppressionStrategy(
				new MessageSchemaValueSuppressionStrategy());
		beanWriter.enablePrettyPrint();
		beanWriter.setInitialIndentLevel(0);
		beanWriter.setWriteEmptyElements(false);

		XMLIntrospector intro = beanWriter.getXMLIntrospector();
		InputStream in = null;
		try {
			in = MaskatBeanIO.class.getResourceAsStream(MESSAGE_XSD_CONFIG);
			intro.register(new InputSource(in));
		} finally {
			IOUtils.closeQuietly(in);
		}
		beanWriter.write(element);
		return IOUtils.toInputStream(outputWriter.toString());
	}
}
