/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.betwixt;

import java.beans.IntrospectionException;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.sf.maskat.core.layout.DynaComponent;
import jp.sf.maskat.core.layout.DynaComponentClass;
import jp.sf.maskat.core.layout.DynaProperty;
import jp.sf.maskat.core.layout.UnknownComponent;
import jp.sf.maskat.core.layout.UnknownComponentClass;

import org.apache.commons.beanutils.DynaClass;
import org.apache.commons.betwixt.AttributeDescriptor;
import org.apache.commons.betwixt.BeanProperty;
import org.apache.commons.betwixt.Descriptor;
import org.apache.commons.betwixt.ElementDescriptor;
import org.apache.commons.betwixt.Options;
import org.apache.commons.betwixt.TextDescriptor;
import org.apache.commons.betwixt.XMLBeanInfo;
import org.apache.commons.betwixt.XMLIntrospector;
import org.apache.commons.betwixt.digester.MultiMappingBeanInfoDigester;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Beanを解析しXMLBeanInfoを生成するクラスです。
 * 
 * XMLIntrospectorクラスを拡張しcdata-section, textに対応する
 * 処理を行っています。
 *
 */
public class MaskatXMLIntrospector extends XMLIntrospector {

	/**
	 * Digester used to parse the multi-mapping XML descriptor files
	 */
    private MultiMappingBeanInfoDigester multiMappingdigester;
    
    private HashMap beanMap = new HashMap();

	/**
	 * デフォルトコンストラクタです
	 *
	 */
    public MaskatXMLIntrospector() {
		super();
	}

	/**
     * {@inheritDoc}
     */
    public XMLBeanInfo introspect(DynaClass dynaClass) {
    	XMLBeanInfo xmlInfo = super.introspect(dynaClass);
    	ElementDescriptor descriptor = xmlInfo.getElementDescriptor();
    	
    	if (dynaClass instanceof DynaComponentClass) {
       		DynaComponentClass dynaComponent = (DynaComponentClass) dynaClass;    		
    		String qualifiedName = dynaComponent.getQualifiedName();
       		descriptor.setQualifiedName(qualifiedName);
       		DynaProperty prop = (DynaProperty) dynaComponent.getDynaProperty("context");
       		
       		if (prop != null) {
				TextDescriptor desc = getComponentTextDescriptor(DynaComponent.class);
				if (desc != null) {
					descriptor.addContentDescriptor(desc);
       			}
       			if (prop.isCdataSection()) {
       				Options options = getComponentOption(DynaComponent.class);
       				if (options != null) {
       					descriptor.setOptions(options);
       				}
       			}
       		}

    	} else if (	dynaClass instanceof UnknownComponentClass) {
    		String qualifiedName = ((UnknownComponentClass) dynaClass).getQualifiedName();
       			descriptor.setQualifiedName(qualifiedName);
       			
       		DynaProperty prop = (DynaProperty) ((UnknownComponentClass) dynaClass).getDynaProperty("context");
       		if (prop != null) {
       			if (prop instanceof jp.sf.maskat.core.layout.DynaProperty) {
       				if (!((jp.sf.maskat.core.layout.DynaProperty) prop).isEnabled()) {
       					return xmlInfo;
       				}
       			}
				TextDescriptor desc = getComponentTextDescriptor(UnknownComponent.class);
				if (desc != null) {
					descriptor.addContentDescriptor(desc);
       			}
       			if (prop.isCdataSection()) {
       				Options options = getComponentOption(UnknownComponent.class);
       				if (options != null) {
       					descriptor.setOptions(options);
       				}
       			}
       		}
       		
    	}
    	return xmlInfo;
    }

	/**
	 * DynaComponetのElementDescriptorを取得します。
	 * 
	 * @param clazz 取得したいクラス
	 * @return ElementDescriptorが存在する場合にはElementDescriptor
	 *          存在しない場合には null
	 */
    private ElementDescriptor getComponentDescriptor(Class clazz) {
    	XMLBeanInfo dynaInfo = (XMLBeanInfo) beanMap.get(clazz);
    	if (dynaInfo != null) {
    		return dynaInfo.getElementDescriptor();
    	}
    	return null;
    }


	/**
	 * DynaComponentのElementDescriptorからTextDescriptorを取得します。
	 * 
	 * @param clazz 取得したいクラス
	 * @return TextDescriptorが存在する場合にはTextDescriptor
	 *          存在しない場合には null
	 */
    private TextDescriptor getComponentTextDescriptor(Class clazz ) {
   		ElementDescriptor element = getComponentDescriptor(clazz);
   		if (element != null) {
   			Descriptor[] descriptors = element.getContentDescriptors();
   			for (int i = 0; i < descriptors.length; i++) {
    			if (descriptors[i] instanceof TextDescriptor) {
    				return (TextDescriptor) descriptors[i];
    			}
    		}
    	}
    	return null;
    }

	/**
	 * DynaComponentのElementDescriptorからOptionを取得します。
	 * 
	 * <p>
	 * cdata-sectionのエンコードを行うOptionを取得します。
	 * org.apache.commons.betwixt.mixed-content-encoding
	 * </p>
	 * 
	 * @param clazz 取得したいクラス
	 * @return Optionが存在する場合にはOption
	 *          存在しない場合には null
	 */
    private Options getComponentOption(Class clazz) {
   		ElementDescriptor element = getComponentDescriptor(clazz);
   		if (element != null) {
   			return element.getOptions();
    	}
    	return null;
    }

    /**
     * {@inheritDoc}
     */
    protected void addProperty(
    		BeanProperty beanProperty, 
    		List elements, 
    		List attributes,
    		List contents) {
    	Descriptor nodeDescriptor = createXMLDescriptor(beanProperty);
    	if (nodeDescriptor == null) {
    		return;
    	}
    	if (nodeDescriptor instanceof ElementDescriptor) {
    		elements.add(nodeDescriptor);
    	} else if (nodeDescriptor instanceof AttributeDescriptor) {
    		if (!"context".equals(beanProperty.getPropertyName())) {
    			attributes.add(nodeDescriptor);
    		}
    	} else {
    		contents.add(nodeDescriptor);
    	}                                 
    }

    /**
     * <p>Registers the class mappings specified in the multi-class document
     * given by the <code>InputSource</code>.
     * </p>
     * <p>
     * <strong>Note:</strong> that this method will override any existing mapping
     * for the speficied classes.
     * </p>
     * @since 0.7
     * @param source <code>InputSource</code>, not null
     * @return <code>Class</code> array containing all mapped classes
     * @throws IntrospectionException
     * @throws SAXException
     * @throws IOException
     */
    public synchronized Class[] register(InputSource source) throws IntrospectionException, IOException, SAXException {
        Map xmlBeanInfoByClass = loadMultiMapping(source);	
        Set keySet = xmlBeanInfoByClass.keySet();
        Class mappedClasses[] = new Class[keySet.size()];
        int i=0;
        for (Iterator it=keySet.iterator(); it.hasNext(); ) {
            Class clazz = (Class) it.next();
            mappedClasses[i++] = clazz;
            XMLBeanInfo xmlBeanInfo = (XMLBeanInfo) xmlBeanInfoByClass.get(clazz);
            if (xmlBeanInfo != null) {
            	beanMap.put(clazz, xmlBeanInfo);
                getRegistry().put(clazz, xmlBeanInfo);
            }   
        }
        return mappedClasses;
    }
    
    /**
     * Loads the multi-mapping from the given <code>InputSource</code>.
     * @param mapping <code>InputSource</code>, not null
     * @return <code>Map</code> containing <code>XMLBeanInfo</code>'s
     * indexes by the <code>Class</code> they describe
     * @throws IOException
     * @throws SAXException
     */
    private synchronized Map loadMultiMapping(InputSource mapping) throws IOException, SAXException {
        // synchronized method so this digester is only used by
        // one thread at once
        if (multiMappingdigester == null) {
            multiMappingdigester = new MultiMappingBeanInfoDigester();
            multiMappingdigester.setXMLIntrospector(this);
        }
        Map multiBeanInfoMap = (Map) multiMappingdigester.parse(mapping);
        return multiBeanInfoMap;
    }
}
