/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.betwixt;

import java.io.IOException;
import java.io.Writer;
import java.util.Iterator;
import java.util.Map;

import jp.sf.maskat.core.layout.ComponentLibrary;
import jp.sf.maskat.core.layout.ComponentRegistry;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;
import org.apache.commons.betwixt.io.BeanWriter;
import org.apache.commons.betwixt.io.WriteContext;
import org.apache.commons.betwixt.strategy.NamespacePrefixMapper;

/**
 * レイアウト定義XMLを出力するクラスです。
 * 各プラグインごとの名前空間を付与するためにbetwixtのBeanWriterを
 * 拡張したクラスです。
 * 
 * <p>
 * 名前空間を持つ部品がレイアウト定義に存在する場合、その部品のprefixを
 * layoutDefタグに定義します。
 * </p>
 * <pre>
 * <layoutDef xmlns:custom="http://maskat.sourceforge.jp/widget/custom/2.1.0">
 * </pre>
 */
public class MaskatBeanWriter extends BeanWriter {

	/**
	 * デフォルトコンストラクタです。
	 * <p>
	 * 名前空間、cdata-section, textに対応したXMLIntrospector
	 * (MaskatXMLIntrospector)を設定します。
	 * </p>
	 */
    public MaskatBeanWriter() {
        super();
        this.setXMLIntrospector(new MaskatXMLIntrospector());
    }

    /**
     * コンストラクタです。
	 * <p>
	 * 名前空間、cdata-section, textに対応したXMLIntrospector
	 * (MaskatXMLIntrospector)を設定します。
	 * </p>
     * @param writer 出力先
     */
    public MaskatBeanWriter(Writer writer) {
        super(writer);
        this.setXMLIntrospector(new MaskatXMLIntrospector());
    }

    /**
     * {@inheritDoc}
     */
    protected void startElement(WriteContext context, String uri,
    		String localName, String qualifiedName, Attributes attr)
    			throws IOException, SAXException {
        if ("children".equals(qualifiedName) && attr.getLength() <= 0) {
            return;
        }
        /*
         * layoutDefタグの場合には名前空間接頭辞と名前空間の定義を書き込みます。
         */
        if ("layoutDef".equals(qualifiedName)) {
            NamespacePrefixMapper mapper = this.getXMLIntrospector()
                    .getConfiguration().getPrefixMapper();

            Map prefixMap = ((MaskatNamespacePrefixMapper) mapper)
                    .getPrefixes();
            AttributesImpl attribute = new AttributesImpl();

            for (Iterator ite = prefixMap.keySet().iterator(); ite.hasNext();) {
                String key = (String) ite.next();
                if (!isDefaultNamespaceURI((String) prefixMap.get(key))) {
                    attribute.addAttribute("", "", "xmlns:" + key, "NOTATION",
                            (String) prefixMap.get(key));
                }
            }
            attr = attribute;
        }
        /*
         * デフォルトの名前空間に属する部品の場合には名前空間接頭辞を除きます。 
         */
        if (isDefaultComponent(qualifiedName)) {
        	super.startElement(context, uri, localName, localName, attr);
        } else {
            super.startElement(context, uri, localName, qualifiedName, attr);
        }
    }

    /**
     * タグ名がデフォルト名前空間に属する部品であるか判定します。
     * 
     * @param qualifiedName 名前空間接頭辞を含むタグ名
     * @return デフォルト名前空間に属する部品の場合、true を返します。
     */
    private boolean isDefaultComponent(String qualifiedName) {
    	int pos = qualifiedName.indexOf(":");
    	if (pos == -1) {
    		return false;
    	}
    	String prefix =  qualifiedName.substring(0, pos);
    	String localName = qualifiedName.substring(pos + 1);
    	
        ComponentLibrary[] defaultLibrarys = ComponentRegistry
				.getDefaultLibrarys();
        for (int i = 0; i < defaultLibrarys.length; i++) {
        	if (defaultLibrarys[i].getPrefix().equals(prefix)) {
        		if (defaultLibrarys[i].containsComponent(localName)) {
        			return true;
        		}
        	}
        }
        return false;
    }

    /**
     * 名前空間がデフォルト名前空間であるか判定します。
     * 
     * @param namespace 名前空間
     * @return デフォルトの名前空間の場合、true を返します。
     */
    private boolean isDefaultNamespaceURI(String namespace) {
        ComponentLibrary[] defaultLibrarys = ComponentRegistry
                .getDefaultLibrarys();
        for (int i = 0; i < defaultLibrarys.length; i++) {
            if (defaultLibrarys[i].getNamespaceURI().equals(namespace)) {
                return true;
            }
        }
        return false;
    }

	/**
     * {@inheritDoc}
     */
    protected void endElement(WriteContext context, String uri,
            String localName, String qualifiedName) throws IOException,
                                                   SAXException {
        if ("children".equals(qualifiedName)) {
            return;
        }
        /*
         * デフォルトの名前空間に属する部品の場合には名前空間接頭辞を除きます。 
         */
        if (isDefaultComponent(qualifiedName)) {
        	super.endElement(context, uri, localName, localName);
        } else {
            super.endElement(context, uri, localName, qualifiedName);
        }
    }

	/**
     * {@inheritDoc}
     */
    public boolean getWriteEmptyElements() {
        return true;
    }
}
