/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.betwixt;

import java.util.HashMap;
import java.util.Map;

import jp.sf.maskat.core.MaskatElement;
import jp.sf.maskat.core.event.Component;
import jp.sf.maskat.core.event.Event;
import jp.sf.maskat.core.event.Param;
import jp.sf.maskat.core.event.RemoteUrl;
import jp.sf.maskat.core.event.Result;
import jp.sf.maskat.core.event.Source;
import jp.sf.maskat.core.event.Target;
import jp.sf.maskat.core.layout.Desc;

import org.apache.commons.betwixt.AttributeDescriptor;
import org.apache.commons.betwixt.ElementDescriptor;
import org.apache.commons.betwixt.strategy.ValueSuppressionStrategy;

/**
 * イベント定義XMLの要素に定義されている属性の出力制御を行うクラスです。
 * <p>
 * 属性に定義されている値がデフォルト値である場合はその属性を出力しません。
 */
public class EventValueSuppressionStrategy extends ValueSuppressionStrategy {

	private static final String DOT = ".";

	private Map attributeDefaultValues = new HashMap();

	private MaskatElement currentElement = null;

	/**
	 * デフォルトコンストラクタです。
	 * <p>
	 * イベント定義要素の属性にデフォルト値を定義します。
	 * </p>
	 */
	public EventValueSuppressionStrategy() {
		attributeDefaultValues.put(createAttributeKey(Event.class, "type"),
				Event.TYPE_DEFAULT);
		attributeDefaultValues.put(createAttributeKey(Event.class, "async"),
				String.valueOf(Event.ASYNC_DEFAULT));
		attributeDefaultValues.put(createAttributeKey(Event.class, "timeout"),
				String.valueOf(Event.TIMEOUT_DEFAULT));
		attributeDefaultValues.put(createAttributeKey(Param.class, "soap"),
				String.valueOf(Param.SOAP_DEFAULT));
		attributeDefaultValues.put(createAttributeKey(Source.class,
				"sendBlankElement"), String
				.valueOf(Source.SEND_BLANK_ELEMENT_DEFAULT));
		attributeDefaultValues.put(createAttributeKey(Source.class, "type"),
				Source.TYPE_DEFAULT);
		attributeDefaultValues.put(createAttributeKey(Source.class, "min"),
				String.valueOf(Source.MIN_DEFAULT));
		attributeDefaultValues.put(createAttributeKey(Source.class, "max"),
				String.valueOf(Source.MAX_DEFAULT));
		attributeDefaultValues.put(createAttributeKey(Source.class, "fromkey"),
				Source.FROM_KEY_DEFAULT);
		attributeDefaultValues.put(createAttributeKey(Target.class, "type"),
				Target.TYPE_DEFAULT);
		attributeDefaultValues.put(createAttributeKey(Result.class, "soap"),
				String.valueOf(Result.SOAP_DEFAULT));
	}

	/**
	 * クラス名と属性名からユニークなキーを生成します。
	 * <p>
	 * ここで生成したユニークなキーはデフォルト値マップのキーとなります。
	 * </p>
	 * @param clazz attributeNameで指定した属性を持つクラス
	 * @param attributeName 属性名
	 * @return キー
	 */
	private String createAttributeKey(Class clazz, String attributeName) {
		StringBuffer sb = new StringBuffer(clazz.getName());
		sb.append(DOT);
		sb.append(attributeName);
		return sb.toString();
	}

	/**
     * {@inheritDoc}
     */
	public boolean suppressAttribute(AttributeDescriptor attributeDescriptor,
			String value) {
		if (attributeDescriptor.getPropertyName() != null && value != null
				&& isDefaultValue(attributeDescriptor.getPropertyName(),
						value)) {
			return true;
		}
		return ValueSuppressionStrategy.DEFAULT.suppressAttribute(
				attributeDescriptor, value);
	}

	/**
	 * 指定された値がデフォルト値であるか判定します。
	 * 
	 * @param attribute 属性名
	 * @param value 値
	 * @return true デフォルト値
	 */
	private boolean isDefaultValue(String attribute, String value) {
		String attributeKey = null;
		if (currentElement == null) {
			attributeKey = attribute;
		} else {
			attributeKey = (new StringBuffer(currentElement.getClass()
					.getName()).append(DOT).append(attribute).toString());
		}
		return (attributeDefaultValues.containsKey(attributeKey) && value
				.equals(attributeDefaultValues.get(attributeKey)));
	}

	/**
     * {@inheritDoc}
     */
	public boolean suppressElement(ElementDescriptor element,
			String namespaceUri, String localName, String qualifiedName,
			Object value) {

		if (value instanceof MaskatElement) {
			currentElement = (MaskatElement) value;

			if (value instanceof Event) {
				return !((Event) value).isEnable();
			}
			if (value instanceof Component) {
				return !((Component) value).hasEnabledEvents();
			}
			if (value instanceof RemoteUrl) {
				return ((RemoteUrl) value).getUrl() == null
						|| "".equals(((RemoteUrl) value).getUrl());
			}
			if (value instanceof Param) {
				return !((Param) value).isEnable();
			}
			if (value instanceof Result) {
				return !((Result) value).isEnable();
			}
			if (value instanceof Desc) {
				return ((Desc) value).getComment() == null;
			}
		}
		return ValueSuppressionStrategy.DEFAULT.suppressElement(element,
				namespaceUri, localName, qualifiedName, value);
	}
}
