/*
 * Copyright (c) 2006-2011 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core;

import java.util.List;
import java.util.Map;

/**
 * マスカットが利用する要素の基本インターフェースです。
 * <p>
 * マスカットの要素であるレイアウト、イベントのインターフェースであり
 * 要素を制御するための各メソッドを定義しています。
 * </p>
 */
public interface MaskatElement extends Cloneable {
	
	/**
	 * この要素の親要素を取得します。
	 * 
	 * @return この要素の親要素
	 */
	public MaskatElement getParent();

	/**
	 * この要素の親となる要素を設定します。
	 * 
	 * @param parent この要素の親となる要素
	 */
	public void setParent(MaskatElement parent);

	/**
	 * この要素が持っている子要素の配列を取得します。
	 * 
	 * @return この要素が持っている子要素の配列
	 */
	public List getChildren();

	/**
	 * この要素に子となる要素を追加します。
	 * 
	 * @param child この要素に追加する子要素
	 */
	public void addChild(Object child);

	/**
	 * この要素から子要素を削除します。
	 * 
	 * @param obj この要素から削除する子要素
	 */
	public void removeChild(Object obj);

	/**
	 * この要素が持っている全ての子要素を削除します。
	 */
	public void removeAllChildren();

	/**
	 * この要素が持っている子要素の中から指定されたクラスであるものを削除します。
	 * 
	 * @param clazz 子要素の中から削除を行いたいクラス
	 */
	public void removeAllByType(Class clazz);

	/**
	 * この要素の階層構造順に呼び出すMaskatElementVisitorを受け入れます。
	 * 
	 * @param visitor MaskatElementVisitorのインスタンス
	 */
	public void accept(MaskatElementVisitor visitor);

	/**
	 * この要素の複製（ディープコピー）を生成します。
	 * 
	 * @return 複製された要素
	 * @throws CloneNotSupportedException 複製することができない場合
	 */
	public Object clone() throws CloneNotSupportedException;

	/**
	 * 指定された要素配下にある子要素が指定されたクラスであるものを取得します。
	 * <p>TODO: リファクタリングして消したいメソッド</p>
	 * 
	 * @param parent 検索を行う要素
	 * @param descendantClass 子要素の中から取得したいクラス
	 * @return 該当する要素のリスト
	 */
	public List getAllDescendants(MaskatElement parent, Class descendantClass);

	/**
	 * この要素が持つ属性のデフォルト値のマップを取得します。
	 * <p>
	 * 取得されるマップは属性名がキー、値がデフォルト値となります。
	 * </p>
	 * 
	 * @return デフォルト値マップ
	 */
	public Map getDefaultValues();

	/**
	 * この要素が持っている子要素の中から指定クラスと指定位置であるものを
	 * 取得します。
	 * 
	 * @param clazz　子要素の中から取得したいクラス
	 * @param idx  clazzと該当した要素の中で取得したい位置
	 * @return 該当する要素
	 */
	public Object getChildByTypeIdx(Class clazz, int idx);

	/**
	 * この要素が持つ子要素の中から指定された要素の位置を取得します。
	 * 
	 * @param child 対象要素
	 * @return 要素位置
	 */
	public int getChildIdx(Object child);

	/**
	 * この要素の指定位置に子要素を追加します。
	 * 
	 * @param child この要素に追加する子要素
	 * @param idx 追加を行いたい位置
	 */
	public void addChildToIdx(Object child, int idx);

	/**
	 * この要素が持っている子要素の中から指定クラスである要素をリストで取得します。
	 * 
	 * @param clazz 子要素の中から取得したいクラス
	 * @return 該当する要素のリスト
	 */
	public List getTypedChildren(Class clazz);
}
