/*
 * Copyright (c) 2006-2009 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.MaskatElement;
import jp.sf.maskat.core.layout.ComponentLibrary;
import jp.sf.maskat.core.layout.DynaProperty;

import org.apache.commons.beanutils.BasicDynaClass;


public class DynaComponentClass extends BasicDynaClass {
	
	private static final long serialVersionUID = 5110101941810053916L;

	public static final int NONE = 0;

	public static final int RESIZABLE_WIDTH = 0x01;

	public static final int RESIZABLE_HEIGHT = 0x02;

	public static final int RESIZABLE_BOTH = RESIZABLE_WIDTH | RESIZABLE_HEIGHT;

	public static final int NO_FOCUS = 0x04;

	public static final int CONTAINER = 0x08;
	
	public static final int COMPONENT = 0x10;
	
	public static final int ITEM = 0x20;

	public static final int MOVABLE_TOP = 0x40;
	
	public static final int MOVABLE_LEFT = 0x80;
	
	public static final int MOVABLE_BOTH = MOVABLE_TOP | MOVABLE_LEFT;
	
	public static final int WIDGET_NAME = 0x100;
	
	private ComponentLibrary library;
	
	private String[] eventTypes;
	
	private int flag;
	
	private  Map dafaultValues = null;
	
	private Map childWidgets = null;
	
	private String qualifiedName; 	

	public DynaComponentClass() {
		super();
	}
	
	public DynaComponentClass(
				String name,
				Class dynaBeanClass,
				DynaProperty[] properties) {
		
		super(name, dynaBeanClass, properties);
		this.dafaultValues = new HashMap();
		this.childWidgets = new HashMap();
	}
	
	public DynaComponentClass(
				ComponentLibrary library,
				String name,
				DynaProperty[] properties,
				String[] eventTypes,
				Map values,
				Map childWidgets,
				int flag) {
		
		super(name, null, null);
		this.library = library;
		this.flag = flag;
		if (library.getPrefix() != null) {
			this.qualifiedName = library.getPrefix() + ":" + name;
		}
		this.dafaultValues = (values == null) ? new HashMap() : values;
		this.childWidgets = (childWidgets == null) ? new HashMap() : childWidgets;

		setDynaBeanClass(isContainer() ? DynaContainer.class
				: DynaComponent.class);
		setProperties(properties);
		setEventTypes(eventTypes);
	}
	
	protected void setProperties(DynaProperty[] properties) {
		List list = new ArrayList();
		list.add(new DynaProperty("parent", MaskatElement.class));
		list.add(new DynaProperty("children", List.class));
		
		if (isWidgetName()) {
			list.add(new DynaProperty("name", String.class));
		}
		if (isTopMovable()) {
			list.add(new DynaProperty("top", Integer.class));
		}
		if (isLeftMovable()) {
			list.add(new DynaProperty("left", Integer.class));
		}
		if (isWidthResizable()) {
			list.add(new DynaProperty("width", Integer.class));
		}
		if (isHeightResizable()) {
			list.add(new DynaProperty("height", Integer.class));
		}
		if (canFocus()) {
			list.add(new DynaProperty("tabIndex", Integer.class));
		}
		if (properties != null) {
			for (int i = 0; i < properties.length; i++) {
				list.add(properties[i]);
			}
		}
		super.setProperties((DynaProperty[]) list.toArray(new DynaProperty[list
				.size()]));
	}
	
	public void setDynaProperties(DynaProperty[] properties) {
        setProperties(properties);
    }
	
	public void setDynaBeanClass(Class dynaBeanClass) {
        super.setDynaBeanClass(dynaBeanClass);
    }
	
	public String[] getEventTypes() {
		return eventTypes;
	}

	public Map getDefaultValues() {
		return dafaultValues;
	}

	public boolean isAddChildWidget(String namespaceURI, String name) {
		if (this.childWidgets.size() <= 0) {
			return isContainer();
		}
		Map map = (Map) this.childWidgets.get(namespaceURI);
		if (map == null) {
			return false;
		}
		return map.containsKey(name);
	}
	
	protected void setEventTypes(String[] eventTypes) {
		this.eventTypes = eventTypes;
	}

	public ComponentLibrary getLibrary() {
		return library;
	}

	public boolean isFlagSet(int check) {
		return (flag & check) > 0;
	}

	public boolean isWidthResizable() {
		return isFlagSet(RESIZABLE_WIDTH);
	}
	
	public boolean isHeightResizable() {
		return isFlagSet(RESIZABLE_HEIGHT);
	}
	
	public boolean isTopMovable() {
		return isFlagSet(MOVABLE_TOP);
	}
	
	public boolean isLeftMovable() {
		return isFlagSet(MOVABLE_LEFT);
	}
	
	public boolean isContainer() {
		return isFlagSet(CONTAINER);
	}
	
	public boolean isComponent() {
		return isFlagSet(COMPONENT);
	}
	
	public boolean isItem() {
		return isFlagSet(ITEM);
	}
	
	public boolean isWidgetName() {
		return isFlagSet(WIDGET_NAME);
	}
	
	public boolean canFocus() {
		return !isFlagSet(NO_FOCUS);
	}

	public String getQualifiedName() {
		return this.qualifiedName;
	}
	
	public void setqualifiedName(String name) {
		this.qualifiedName = name;
	}
}
