/* Copyright 2007 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest.sample.minesweeper;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import jp.sourceforge.jlogtest.Logger;
import jp.sourceforge.jlogtest.LoggerFactory;
import jp.sourceforge.jlogtest.annotations.Log;
import jp.sourceforge.jlogtest.annotations.Sequence;
import jp.sourceforge.jlogtest.annotations.TargetLogs;
import jp.sourceforge.jlogtest.junit.JLogTestRunner;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;

@RunWith(JLogTestRunner.class)
@TargetLogs(@Sequence(log = @Log("initBoard")))
public class MineSweeperTest {
	
	private static final Point LEFT_TOP = new Point(0, 0);
	private static final Size boardSize = new Size(5, 6);
	private static final Logger initBoardLog = LoggerFactory.getLogger("initBoard");
	
	@Before
	public void setUp() throws Exception {
		JLogTestRunner.setLogDirectory("src/test/resources/jlogtest");
	}

	@Test
	public void 座標を指定してオープンできる() throws Exception {
		final MineSweeper ms = getMineBasedMineSweeper(LEFT_TOP);

		// 最初はオープンしていない
		assertFalse(ms.isOpen(LEFT_TOP));
		
		// 左上のセルをクリックする
		ms.open(LEFT_TOP);
		
		// オープンしている
		assertTrue(ms.isOpen(LEFT_TOP));
		// 周りに地雷が3つ存在することがわかる
		assertEquals(" 3", ms.getCellOutput(LEFT_TOP));
	}

	@Test
	public void 地雷を踏むとゲームオーバー() throws Exception {
		final MineSweeper ms = getMineBasedMineSweeper(LEFT_TOP);

		// 地雷のないセルをクリックする
		//（仕様上一回目に地雷のあるセルをクリックすることはできないので）
		ms.open(LEFT_TOP);

		// ゲームオーバーではない
		assertFalse(ms.gameOver());

		// 地雷のあるセルをクリックする
		ms.open(new Point(1, 0));
		
		// 地雷を踏んでしまったことがわかる
		assertEquals("＊", ms.getCellOutput(new Point(1, 0)));
		// ゲームオーバーであることがわかる
		assertTrue(ms.gameOver());
	}

	@Test
	public void 一回目に地雷は踏めない() throws Exception {
		final MineSweeper ms = getMineBasedMineSweeper(LEFT_TOP);

		// 本来地雷があるはずのセルをクリックする
		ms.open(new Point(1, 0));
		
		// 一回目は地雷は踏めない
		assertEquals(" 5", ms.getCellOutput(new Point(1, 0)));
		// ゲームオーバーにはならない
		assertFalse(ms.gameOver());
	}

	@Test
	public void オープンしていないセルは四角で表示される() throws Exception {
		final MineSweeper ms = getMineBasedMineSweeper(LEFT_TOP);

		assertEquals("□", ms.getCellOutput(new Point(0, 0)));
	}

	@Test
	public void オープンしていないセルには旗を立てることができる() throws Exception {
		final MineSweeper ms = getMineBasedMineSweeper(LEFT_TOP);
		final Point point = new Point(1, 0);

		// 旗を立てる
		ms.toggleFlag(point);
		
		// 旗が立てられている
		assertTrue(ms.isFlagged(point));
		assertEquals(" F", ms.getCellOutput(point));
		
		// 他のセルには旗は立てられていない
		assertFalse(ms.isFlagged(LEFT_TOP));
		assertEquals("□", ms.getCellOutput(LEFT_TOP));
		
		// 旗を取り除く
		ms.toggleFlag(point);

		// 旗は立てられていない
		assertFalse(ms.isFlagged(point));
		assertEquals("□", ms.getCellOutput(point));
	}

	@Test
	public void あと立てるべき旗の数を取得できる() throws Exception {
		final MineSweeper ms = getMineBasedMineSweeper(LEFT_TOP);

		// 最初は爆弾の個数そのもの
		assertEquals(boardSize.getCellNum() - 1, ms.getNumOfRemainingFlags());
		
		// 旗を立てる
		ms.toggleFlag(new Point(1, 0));
		
		// 旗が1本減る
		assertEquals(boardSize.getCellNum() - 2, ms.getNumOfRemainingFlags());
	}

	@Test
	public void 開いていないセルと地雷の数が等しくなったらゲームクリア() throws Exception {
		final MineSweeper ms = getMineBasedMineSweeper(LEFT_TOP);

		// ゲームはクリアしていない
		assertFalse(ms.isCleared());
		
		// 地雷のない最後のセルを開く
		ms.open(LEFT_TOP);
		
		// ゲームはクリア
		assertTrue(ms.isCleared());
	}

	@Test
	@TargetLogs(@Sequence(name = "feature1", log = @Log("boardOutputFeature")))
	public void 盤面をテキストで出力できる() throws Exception {
		final Logger output = LoggerFactory.getLogger("boardOutputFeature");

		final MineSweeper ms = getMineBasedMineSweeper(LEFT_TOP, new Point(1, 1));

		output.out("[初期状態]");
		output.out(ms.getBoardAsString());

		ms.open(LEFT_TOP);
		
		output.out("[1つセルを開いた後]");
		output.out(ms.getBoardAsString());
		
		// 地雷を踏む
		ms.open(new Point(1, 0));

		// ゲームオーバーになった場合はすべてをオープンにする
		output.out("[地雷を踏んでゲームオーバー]");
		output.out(ms.getBoardAsString());
	}

	@Test
	@TargetLogs(@Sequence(name = "feature1", log = @Log("boardOutputFeature")))
	public void 周りに地雷がないセルをオープンしたときは周囲もオープンする()
			throws Exception {

		final Logger output = LoggerFactory.getLogger("boardOutputFeature");

		// MineBasedではない
		final MineSweeper ms = getMineSweeper(new Point(0, 0), new Point(2, 0));

		ms.open(new Point(0, 2));

		// 周囲のセルも自動的にオープンされる
		output.out("[(0, 2)を開いた後]");
		output.out(ms.getBoardAsString());
	}

	@Test
	public void 指定した位置がボード上に存在するかを確かめられる() throws Exception {
		final MineSweeper ms = getMineBasedMineSweeper(LEFT_TOP);

		// ボード上に存在するかどうか
		assertTrue(ms.isInBoard(LEFT_TOP));
		assertTrue(ms.isInBoard(
				new Point(boardSize.getWidth() - 1, boardSize.getHeight() - 1)));
		assertFalse(ms.isInBoard(new Point(-1, 0)));
	}

	private static MineSweeper getMineBasedMineSweeper(final Point... excluded) {
		
		final MockPermutator permutator =
			MockPermutator.createMineBasedPermutator(boardSize, excluded);
		
		// JLogTestに生成されたボードを（確認のため）出力
		initBoardLog.out(permutator.getBoard());
		
		// 引数 : board size, num of mines, permutator
		return new MineSweeper(
				boardSize, boardSize.getCellNum() - excluded.length, permutator);
	}

	private static MineSweeper getMineSweeper(final Point... mines) {
		
		final MockPermutator permutator =
			MockPermutator.createPermutator(boardSize, mines);
		
		// JLogTestに生成されたボードを（確認のため）出力
		initBoardLog.out(permutator.getBoard());
		
		// 引数 : board size, num of mines, permutator
		return new MineSweeper(boardSize, mines.length, permutator);
	}
}
