/* Copyright 2007 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest.sample.minesweeper;

import java.io.IOException;

public class CommandLineInterface {
	
	private final IInput reader;
	private final IOutput writer;
	private final IRandomPermutator permutator;
	private MineSweeper mineSweeper;

	public CommandLineInterface(
			final IInput reader, final IOutput writer,
			final IRandomPermutator permutator) {

		this.reader = reader;
		this.writer = writer;
		this.permutator = permutator;
		mineSweeper = null;
	}
	
	public void mainLoop() {
		final Size boardSize;
		final int numOfMines;
		try {
			final int width = getInputAsInt("ボードの幅: ");
			final int height = getInputAsInt("ボードの高さ: ");
			numOfMines = getInputAsInt("地雷の数: ");
			boardSize = new Size(width, height);
		} catch (IOException e) {
			writer.outln("例外が発生しました: " + e.getMessage());
			return ;
		}
		mineSweeper = new MineSweeper(boardSize, numOfMines, permutator);
		
		writer.outln(mineSweeper.getBoardAsString());
		
		while (oneAction())
			;
	}
	
	private boolean oneAction() {
		try {
			{
				final CommandParser parser = new CommandParser();
				if (!parser.succeeded)
					return false;
				
				if (!mineSweeper.isInBoard(parser.point))
					return true;
				
				if ("o".equals(parser.command) && !mineSweeper.isOpen(parser.point)) {
					mineSweeper.open(parser.point);
					writer.outln(String.format(
							"(%d, %d)をオープン", parser.point.getX(), parser.point.getY()));
				} else if ("f".equals(parser.command) && !mineSweeper.isOpen(parser.point))
					mineSweeper.toggleFlag(parser.point);
				else
					return true;
			}
			
			if (mineSweeper.gameOver()) {
				writer.outln("ゲームオーバー");
				writer.outln(mineSweeper.getBoardAsString());
				return false;
			}
			
			if (mineSweeper.isCleared()) {
				writer.outln("おめでとうございます！");
				writer.outln(mineSweeper.getBoardAsString());
				return false;
			}
			
			writer.outln(
					String.format("残りの旗: %d", mineSweeper.getNumOfRemainingFlags()));
			writer.outln(mineSweeper.getBoardAsString());
			
			return true;
		} catch (IOException e) {
			writer.outln("例外が発生しました: " + e.getMessage());
			return false;
		}
	}
	
	private class CommandParser {
		
		private boolean succeeded;
		private Point point;
		private String command;
		
		private CommandParser() throws IOException {
			succeeded = false;
			while (true) {
				writer.out("コマンド (x y cmd): ");
				final String line = reader.getLine();
				if (line == null)
					break ;
				final String[] splitted = line.split("\\s+");
				if (splitted.length < 3)
					continue ;
				
				try {
					point = new Point(
							Integer.parseInt(splitted[0]),
							Integer.parseInt(splitted[1]));
					command = splitted[2];
					succeeded = true;
					break ;
				} catch (NumberFormatException e) {
				}
			}
		}
		
	}
	
	private int getInputAsInt(final String message) throws IOException {
		while (true) {
			try {
				writer.out(message);
				return Integer.parseInt(reader.getLine());
			} catch (NumberFormatException e) {
			}
		}
	}

	public static void main(final String[] args) throws NumberFormatException {
		final IInput input = new StdInput();
		final IOutput output = new StdOutput();

		final CommandLineInterface game =
			new CommandLineInterface(input, output, new RandomPermutator());
		game.mainLoop();
	}
}
