/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/util/ReflectFactory.java,v 1.7 2007/10/25 03:16:49 kohnosuke Exp $
 * $Revision: 1.7 $
 * $Date: 2007/10/25 03:16:49 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.util;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;

import jp.sourceforge.jeextension.styleeditor.StyleEditorPlugin;

/**
 * ReflectFactory
 */
public class ReflectFactory {
    
    /** invocation target class identifier */
    private Class fClass;
    
    /** invocation target method identifier */
    private String fName;
    
    /** invocation target object */
    private Object fTarget;
    
    /** signatures of invocation target method */
    private Class[] fTypes = new Class[0];
    
    /** parameters of invocation target method */
    private Object[] fParams = new Object[0];
    
    /**
     * Default Constructor.
     */
    public ReflectFactory() {
        super();
    }
    
    /**
     * Set object of target.
     * @param target    object of target
     */
    public void setTarget(Object target) {
        fTarget = target;
    }
    
    /**
     * Set name of class.
     * @param clazz    name of class
     */
    public void setClass(Class clazz) {
        fClass = clazz;
    }
    
    /**
     * Set name of class.
     * @param clazz    name of class
     */
    public void setClass(String clazz) {
        fClass = loadClass(clazz);
    }
    
    /**
     * Set name of method or field.
     * @param name    name of method or field
     */
    public void setName(String name) {
        fName = name;
    }
    
    /**
     * Set Type of method.
     * @param type  Type
     */
    public void setType(String type) {
        fTypes = new Class[] {loadClass(type)};
    }
    
    /**
     * Set Type of method.
     * @param type  Type
     */
    public void setType(String[] type) {
        fTypes = new Class[type.length];
        for (int i = 0; i < type.length; i++) {
            fTypes[i] = loadClass(type[i]);
        }
    }
    
    /**
     * Set Type of method.
     * @param type  Type
     */
    public void setType(Class[] type) {
        fTypes = type;
    }
    
    /**
     * Set Type of method.
     * @param type  Type
     */
    public void setType(Class type) {
        fTypes = new Class[] {type};
    }
    
    /**
     * Set Parameter of method.
     * @param param parameter
     */
    public void setParam(Object param) {
        fParams = new Object[] {param};
    }
    
    /**
     * Set Parameter of method.
     * @param param parameter
     */
    public void setParam(Object[] param) {
        fParams = param;
    }
    
    /**
     * Return Object that was returned by method of target.
     * 
     * @return  Object
     */
    public Object invoke() {
        Object obj = null;
        try {
            Method m = fClass.getDeclaredMethod(fName, fTypes);
            m.setAccessible(true);
            obj = m.invoke(fTarget, fParams);
            m.setAccessible(false);
        }
        catch (Exception e) {
            StyleEditorPlugin.log(e);
        }
        return obj;
    }
    
    /**
     * Return Object of field of target.
     * @return  Object
     */
    public Object get() {
        Object obj = null;
        try {
            Field f = fClass.getDeclaredField(fName);
            f.setAccessible(true);
            obj = f.get(fTarget);
            f.setAccessible(false);
        }
        catch (Exception e) {
            StyleEditorPlugin.log(e);
        }
        return obj;
    }
    
    /**
     * Set Object to field of target.
     */
    public void set() {
        try {
            Field f = fClass.getDeclaredField(fName);
            f.setAccessible(true);
            f.set(fTarget, fParams);
            f.setAccessible(false);
        }
        catch (Exception e) {
            StyleEditorPlugin.log(e);
        }
    }
    
    /**
     * Return Object that was returned by constructor of target.
     * 
     * @return  Object
     */
    public Object newInstance() {
        Object obj = null;
        try {
            Constructor c = fClass.getDeclaredConstructor(fTypes);
            c.setAccessible(true);
            obj = c.newInstance(fParams);
            c.setAccessible(false);
        }
        catch (Exception e) {
            StyleEditorPlugin.log(e);
        }
        return obj;
    }
    
    /**
     * Return Class for class name.
     * 
     * @param name  name of class by full identifier.
     * @return  Class
     */
    private Class loadClass(String name) {
        Class clazz = null;
        try {
            clazz = Class.forName(name);
        }
        catch (ClassNotFoundException e) {
            StyleEditorPlugin.log(e);
        }
        return clazz;
    }
}
