/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/text/RegExpRule.java,v 1.7 2007/10/25 03:16:49 kohnosuke Exp $
 * $Revision: 1.7 $
 * $Date: 2007/10/25 03:16:49 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.text;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.text.rules.ICharacterScanner;
import org.eclipse.jface.text.rules.IPredicateRule;
import org.eclipse.jface.text.rules.IToken;
import org.eclipse.jface.text.rules.Token;

/**
 * RegExpRule
 * @since 1.4.0
 * <!-- #ReqID:10017 RegExp -->
 * <!-- #IntID:00001 Refactoring -->
 */
public class RegExpRule implements IPredicateRule {
    
    /**
     * IToken when it's matching.
     */
    protected IToken fToken;
    
    /**
     * RegExp Pattern
     */
    protected Pattern fPattern;
    
    /**
     * Constructer using regexp strings, and IToken when it's matching.
     * @param regexp    regexp strings
     * @param token     IToken
     */
    public RegExpRule(String regexp, IToken token) {
        super();
        this.fToken = token;
        this.fPattern = Pattern.compile(regexp);
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.text.rules.IRule#evaluate(org.eclipse.jface.text.rules.ICharacterScanner)
     */
    public IToken evaluate(ICharacterScanner scanner) {
        int c = scanner.read();
        scanner.unread();
        if (c != ICharacterScanner.EOF) {
            // read to end of line.
            String line = readToEOL(scanner);
            // regexp pattern matching.
            Matcher m = fPattern.matcher(line);
            if (m.find()) {
                String match = m.group();
                int start = m.start();
                // return fToken to matching.
                if (start == 0 && match.length() > 0) {
                    unread(line.length() - m.end(), scanner);
                    return fToken;
                }
            }
            unread(line.length(), scanner);
        }
        return Token.UNDEFINED;
    }
    
    /**
     * size number times {@link ICharacterScanner#unread()} call.
     * @param size      unread count.
     * @param scanner   ICharacterScanner
     */
    protected void unread(int size, ICharacterScanner scanner) {
        for (int i = 0; i < size; i++) {
            scanner.unread();
        }
    }
    
    /**
     * read by ICharacterScanner to end of line.
     * @param scanner   ICharacterScanner
     * @return line
     */
    protected String readToEOL(ICharacterScanner scanner) {
        StringBuffer sb = new StringBuffer();
        boolean lineBreakDetected = false;
        while (true) {
            int i = scanner.read();
            char c = (char) i;
            if (i == ICharacterScanner.EOF) {
                scanner.unread();
                break;
            }
            // 
            if (c != '\n' && c != '\n' && lineBreakDetected) {
                scanner.unread();
                break;
            }
            // \r
            if (c == '\r') {
                lineBreakDetected = true;
            }
            // \r
            if (c == '\n') {
                lineBreakDetected = true;
            }
            sb.append(c);
        }
        return sb.toString();
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.text.rules.IPredicateRule#evaluate(org.eclipse.jface.text.rules.ICharacterScanner, boolean)
     */
    public IToken evaluate(ICharacterScanner scanner, boolean resume) {
        return evaluate(scanner);
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.text.rules.IPredicateRule#getSuccessToken()
     */
    public IToken getSuccessToken() {
        return fToken;
    }
}
