/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/text/ExtWordRule.java,v 1.6 2007/10/25 03:16:49 kohnosuke Exp $
 * $Revision: 1.6 $
 * $Date: 2007/10/25 03:16:49 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.text;

import java.util.Iterator;
import java.util.Map;

import org.eclipse.jdt.ui.text.IColorManager;
import org.eclipse.jface.text.rules.ICharacterScanner;
import org.eclipse.jface.text.rules.IToken;
import org.eclipse.jface.text.rules.IWordDetector;
import org.eclipse.jface.text.rules.Token;
import org.eclipse.jface.text.rules.WordRule;

/**
 * ExtWordRule
 * @since 1.0.0
 */
public class ExtWordRule extends WordRule implements IExtRule {
    
    /** keyword entry for define. */
    private String fWord;
    
    /** keyword case sensitive. */
    private String fCase;
    
    /** Buffer used for pattern detection */
    private final StringBuffer fBuffer = new StringBuffer();
    
    /**
     * 
     * @param detector IWordDetector object.
     */
    public ExtWordRule(IWordDetector detector) {
        super(detector);
    }

    /**
     * 
     * @param detector      IWordDetector object.
     * @param defaultToken  default IToken objet.
     */
    public ExtWordRule(IWordDetector detector, IToken defaultToken) {
        super(detector, defaultToken);
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.rule.IExtRule#activateColor(jp.sourceforge.jeextension.editor.editors.ColorManager)
     */
    public void activateColor(IColorManager cm) {
        // activate default token color device.
        Object obj = super.fDefaultToken.getData();
        if (obj instanceof IColorActivator) {
            ((IColorActivator) obj).activateColor(cm);
        }
        // activate words entries token color device.
        Iterator iterator = super.fWords.values().iterator();
        while (iterator.hasNext()) {
            IToken token = (IToken) iterator.next();
            obj = token.getData();
            if (obj instanceof IColorActivator) {
                ((IColorActivator) obj).activateColor(cm);
            }
        }
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.text.rules.WordRule#addWord(java.lang.String, org.eclipse.jface.text.rules.IToken)
     */
    public void addWord(String word, IToken token, String sensitive) {
        super.addWord(word, token);
        
        this.fWord = word;
        this.fCase = sensitive;
        
        if (fCase.equalsIgnoreCase("false")) {
            String upper = word.toUpperCase();
            String lower = word.toLowerCase();
            addWord(upper, token);
            addWord(lower, token);
        }
    }
    
    /**
     * 
     * @param wordRuleMap Map of managing words.
     */
    public void addWordRuleMap(Map wordRuleMap) {
        super.fWords.putAll(wordRuleMap);
    }
    
    /**
     * 
     * @return Map of managing words. 
     */
    public Map getWordRuleMap() {
        return super.fWords;
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.rule.IExtRule#getWords()
     */
    public String[] getWords() {
        return new String[] {
            this.fWord,
            this.fCase,
        };
    }
    
    /**
     * {@inheritDoc}
     * @see jp.sourceforge.jeextension.editor.rule.IExtRule#setWords(java.lang.String[])
     */
    public void setWords(String[] words) {
        String keyword = words[0];
        String casesesitive = words[1];
        
        IToken token = (IToken) super.fWords.remove(keyword);
        
        addWord(keyword, token, casesesitive);
    }
    
    /**
     * {@inheritDoc}
     */
    public IToken evaluate(ICharacterScanner scanner) {
        int c = scanner.read();
        if (fDetector.isWordStart((char) c)) {
            if (fColumn == UNDEFINED || (fColumn == scanner.getColumn() - 1)) {
                
                fBuffer.setLength(0);
                do {
                    fBuffer.append((char) c);
                    c = scanner.read();
                } while (isContinue(c));
                scanner.unread();
                
                IToken token = (IToken) fWords.get(fBuffer.toString());
                if (token != null) {
                    return token;
                }
                if (fDefaultToken.isUndefined()) {
                    unreadBuffer(scanner);
                }
                return fDefaultToken;
            }
        }
        
        scanner.unread();
        return Token.UNDEFINED;
    }
    
    /**
     * check to continue reading.
     * 
     * @param c read char
     * @return  if continue to read, return true.
     */
    private boolean isContinue(int c) {
        return c != ICharacterScanner.EOF && fDetector.isWordPart((char) c);
    }
}
