/*************************************************************************************************/
/*!
   	@file		main.h
	@author 	Fanzo
*/
/*************************************************************************************************/
#pragma		once

///////////////////////////////////////////////////////////////////////////////////////////////////
//include files
#include	"AsioKsSetting.h"

#pragma pack( push , 8 )		//set align

using namespace icubic;
using namespace icubic_audio;

///////////////////////////////////////////////////////////////////////////////////////////////////
// preprocessor deifne

///////////////////////////////////////////////////////////////////////////////////////////////////
// type define

///////////////////////////////////////////////////////////////////////////////////////////////////
// classes define

#ifdef cb_debug
cb_guid_define( IID_AsioDriver , 0x6FAF1685 , 0x2178459a , 0x9AD534C9 , 0xE6469310 );
#else
cb_guid_define( IID_AsioDriver , 0x1A80B58F , 0xE5F7442c , 0xA97F64DF , 0xC2C548A9 );
#endif

/**************************************************************************************************
"AsioKs" class 
**************************************************************************************************/
class AsioKs : 
		public Thread , 
		public ComObject , 
		public IASIO
{
	cb_com_begin()
	cb_com_query( IID_AsioDriver() , IASIO )
	cb_com_end( ComObject )

	enum State
	{
		Null_State , 
		Initialized_State , 
		Created_State , 
		Played_State , 
	};
	class Packet
	{
		cb_copy_impossible( Packet );
		
	public:
		KSSTREAM_HEADER m_header;
		OVERLAPPED		m_overlapped;
		Array<uint8>	m_buffer;
		bool			m_first;
		bool			m_wait;
		
		Packet() : m_wait( false ) , m_first( true )
		{
			MemoryZero( &m_header , sizeof( m_header ) );
			MemoryZero( &m_overlapped , sizeof( m_overlapped ) );
			m_overlapped.hEvent	= CreateEvent( NULL , FALSE , FALSE , NULL );
		}
		void Initialize
				(
				uint32			channel , 
				SampleFormat	samplefmt , 
				uint32			samplenum
				)
		{
			m_first		= true;
			m_wait		= false;
			int		byte	= GetSampleFormatSize( samplefmt );
			int		size	= channel * byte * samplenum;
		
			m_buffer.Resize( size );
			{
				float	init = 0.0f;
				uint8*	p = m_buffer.GetPtr();
				uint32	sampleoff;
				for( sampleoff = 0 ; sampleoff < samplenum * channel ; sampleoff++ )
				{
					ConvertFormat( FloatFormat() , &init , samplefmt , p );
					p += byte;
				}
			}
			m_header.Data							= m_buffer.GetPtr();
			m_header.FrameExtent					= size;
			m_header.DataUsed						= size;
			m_header.Size							= sizeof( m_header );
			m_header.PresentationTime.Numerator		= 1;
			m_header.PresentationTime.Denominator	= 1;
		}
	};
	enum CmdType
	{
		Play_cmd , 
		Stop_cmd
	};
	
// variable member
private:
	HWND						m_parent;
	instance<AsioKsSetting>		m_setting;
	State						m_state;

	AsioKsSetting::DeviceInfo	m_devinfo;
	int32						m_fmtid;
	
	// created
	uint32						m_buffersamples;
	wdm::PinStream				m_device;
	ASIOCallbacks*				m_callbacks;
	bool						m_need_timeinfo;
	
	// thread
	CmdType						m_cmd;
	HANDLE						m_event;
	HANDLE						m_event_r;
	Array<int32>				m_buffer;
	Array<int32*>				m_input_ptrs[2];
	Array<int32*>				m_output_ptrs[2];

	// timeinfo
	double						m_sample_position;
	ASIOTimeStamp				m_system_time;
	ASIOTime					m_asio_time;

// thread functions	
private:
//=================================================================================================
void GetSystemTime
		(
		ASIOTimeStamp*	ts
		)
{
	double	nanosec	= ( double )( ( unsigned long )timeGetTime() ) * 1000000.0;
	ts->hi	= ( unsigned long )( nanosec / (double)0x100000000 );
	ts->lo	= ( unsigned long )( nanosec - ( ts->hi * (double)0x100000000 ) );
}
//=================================================================================================
void ThreadBufferSwitch
		(
		int		index
		)
{
	cb_trace( L"id = %d ThreadBufferSwitch = %d\n" , ::GetCurrentThreadId() , index );
	if( m_callbacks == 0 || m_callbacks->bufferSwitch == 0 )
		return;
	m_callbacks->bufferSwitch( index , ASIOFalse );
}
//=================================================================================================
void ThreadBufferSwitchTimeInfo
		(
		int		index
		)
{
	cb_trace( L"id=%d ThreadBufferSwitchTimeInfo = %d\n" , ::GetCurrentThreadId() , index );
	if( m_callbacks == 0 || m_callbacks->bufferSwitchTimeInfo == 0 )
		return;
	m_callbacks->bufferSwitchTimeInfo( &m_asio_time , index , ASIOFalse );
}
//=================================================================================================
void ThreadCallbackSwitch
		(
		int		index
		)
{
	cb_trace( L"priority=%d\n" , ::GetThreadPriority( ::GetCurrentThread() ) );

	// pre initialize time
	m_sample_position	+= m_buffersamples;
	GetSystemTime( &m_system_time );
	getSamplePosition( &m_asio_time.timeInfo.samplePosition , &m_asio_time.timeInfo.systemTime );

	// buffer switch
	if( false == m_need_timeinfo )
		ThreadBufferSwitch( index );
	else if( true == m_need_timeinfo )
		ThreadBufferSwitchTimeInfo( index );

	// post time
	m_asio_time.timeInfo.flags &= ~( kSampleRateChanged | kClockSourceChanged );
}
//=================================================================================================
bool ThreadEvent
		(
		Packet	packet[] ,
		int		pknum
		)
{
	if( m_cmd == Play_cmd )
	{
		ResetEvent( m_event );
		SetEvent( m_event_r );
		cb_verify( true == m_device.Run() );
		int	pkoff;
		for( pkoff = 0 ; pkoff < pknum ; pkoff++ )
			SetEvent( packet[pkoff].m_overlapped.hEvent );
	}
	else if( m_cmd == Stop_cmd )
	{
		int	pkoff;
		for( pkoff = 0 ; pkoff < pknum ; pkoff++ )
		{
			if( packet[pkoff].m_wait == true )
				WaitForSingleObject( packet[pkoff].m_overlapped.hEvent , INFINITE );
			packet[pkoff].m_wait	= false;
		}
		cb_verify( true == m_device.Stop() );
		ResetEvent( m_event );
		SetEvent( m_event_r );
		return false;
	}
	return true;
}
//=================================================================================================
int ThreadProc()
{
	cb_trace( L"ThreadProc\n" );
	cb_verify( TRUE == ::SetThreadPriority( GetCurrentThread() , THREAD_PRIORITY_TIME_CRITICAL ) );

	// initialize header
    Packet	packet[ 2 ];
    {
		int		off;
		for( off = 0 ; off < _countof( packet ) ; off++ )
			packet[ off ].Initialize( m_output_ptrs[0].GetDatanum() , m_devinfo.m_formats[m_fmtid].m_sampletype.m_format , m_buffersamples );
	}
    // initialize events
	HANDLE  events[ _countof( packet ) + 1 ];
	events[ 0 ]	= m_event;
	{
		int		off;
		for( off = 0 ; off < _countof( packet ) ; off++ )
			events[ off + 1 ]	= packet[ off ].m_overlapped.hEvent;
	}
	// stream
	while( true )
	{
		int	evoff	= WaitForMultipleObjects( _countof( events ) , events , FALSE , INFINITE ) - WAIT_OBJECT_0;
		int	pkoff	= evoff - 1;
		if( 0 <= pkoff && pkoff < _countof( packet ) )
		{
			if( packet[ pkoff ].m_first == true || packet[ pkoff ].m_wait == true )
			{
				ThreadCallbackSwitch( pkoff );
				MultiToInterleave
						( 
						m_output_ptrs[0].GetDatanum() , 
						m_buffersamples , 
						int32align32lsb , 
						(void const*const*)m_output_ptrs[pkoff].GetConstPtr() , 
						m_devinfo.m_formats[m_fmtid].m_sampletype.m_format , 
						packet[ pkoff ].m_buffer.GetPtr() 
						);
				packet[ pkoff ].m_first	= false;
			}
			SetEvent( packet[ pkoff ].m_overlapped.hEvent );
			if( true == m_device.WriteData( &packet[ pkoff ].m_header , &packet[ pkoff ].m_overlapped ) )
				packet[ pkoff ].m_wait	= true;
			else
			{
				packet[ pkoff ].m_wait	= false;
				SetEvent( packet[ pkoff ].m_overlapped.hEvent );
			}
		}
		else if( evoff == 0 )
		{
			if( false == ThreadEvent( packet , _countof( packet ) ) )
				break;
		}
	}
	return 0;
}

// private functions
private:
//=================================================================================================
void Cmd
		(
		CmdType		cmd
		)
{
	m_cmd	= cmd;
	ResetEvent( m_event_r );
	SetEvent( m_event );
	WaitForSingleObject( m_event_r , INFINITE );
}
//=================================================================================================
bool GetCreateChannel
		(
		uint32		ch[2]
		)
{
	if( ch[0] > GetInputChannelMax()
	||  ch[1] > GetOutputChannelMax() )
		return false;
	
	int	index;
	for( index = 0 ; index < 2 ; index++ )
	{	
		if( ch[index] >= 1 )
		{
			int		choff;
			for( choff = 0 ; choff < m_devinfo.m_channels[index].GetDatanum() ; choff++ )
			{
				if( ch[index] <= m_devinfo.m_channels[index][choff] )
				{
					ch[index]	= m_devinfo.m_channels[index][choff];
					break;
				}
			}
			if( choff == m_devinfo.m_channels[index].GetDatanum() )
				return false;
		}
	}
	if( ch[0] == 0 && ch[1] == 0 )
		return false;
	return true;
}
//=================================================================================================
bool CreateBuffer
		(
		uint32	in_ch , 
		uint32	out_ch
		)
{
	// check
	uint32	ch[2] = { in_ch , out_ch };
	if( false == GetCreateChannel( ch ) )
		return false;

	// buffer
	m_buffer.Resize( 2 * ( ch[0] + ch[1] ) * m_buffersamples );
	MemoryZero( m_buffer.GetPtr() , m_buffer.GetDatanum() * sizeof( int32 ) );

	uint32	choff;
	int32	bufoff = 0;

	m_input_ptrs[0].Resize( ch[0] );
	m_input_ptrs[1].Resize( ch[0] );
	for( choff = 0 ; choff < ch[0] ; choff++ )
	{
		m_input_ptrs[0][choff]	= &m_buffer[bufoff];
		bufoff	+= m_buffersamples;
		m_input_ptrs[1][choff]	= &m_buffer[bufoff];
		bufoff	+= m_buffersamples;
	}
	m_output_ptrs[0].Resize( ch[1] );
	m_output_ptrs[1].Resize( ch[1] );
	for( choff = 0 ; choff < ch[1] ; choff++ )
	{
		m_output_ptrs[0][choff]	= &m_buffer[bufoff];
		bufoff	+= m_buffersamples;
		m_output_ptrs[1][choff]	= &m_buffer[bufoff];
		bufoff	+= m_buffersamples;
	}
	return true;
}
//=================================================================================================
bool CreateDevice()
{
	DestroyDevice();
	
	// create
	{
		HANDLE	handle = CreateFileW
				(
				m_devinfo.m_path.c_str() , 
				GENERIC_READ | GENERIC_WRITE,
				0,
				NULL,
				OPEN_EXISTING,
				FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED,
				NULL
				);
		if ( false == IsValidHandle( handle ) )
			return false;
		if( false == m_device.Create
				(
				handle , 
				m_devinfo.m_pinid , 
				m_output_ptrs[0].GetDatanum() , 
				m_devinfo.m_formats[m_fmtid].m_sampletype.m_type , 
				m_devinfo.m_formats[m_fmtid].m_sampletype.m_samplebit , 
				m_devinfo.m_formats[m_fmtid].m_samplerate
				) )
		{
			CloseHandle( handle );
			return false;
		}
		CloseHandle( handle );
	}	
	return true;
}
//=================================================================================================
void DestroyDevice()
{
	m_device.Destroy();
}
//=================================================================================================
uint32 GetInputChannelMax()
{
	return ( m_devinfo.m_channels[0].GetDatanum() == 0 ? 0 : m_devinfo.m_channels[0][ m_devinfo.m_channels[0].GetDatanum() - 1 ] );
}
//=================================================================================================
uint32 GetOutputChannelMax()
{
	return ( m_devinfo.m_channels[1].GetDatanum() == 0 ? 0 : m_devinfo.m_channels[1][ m_devinfo.m_channels[1].GetDatanum() - 1 ] );
}
// "IASIO" interface functions
public:
//=================================================================================================
ASIOBool init
		(
		void*	syshandle
		)
{
	cb_trace( L"id=%d init\n" , ::GetCurrentThreadId() );
	if( m_state != Null_State )
		return false;

	if( false == m_setting->Initialize( &m_devinfo , &m_fmtid , &m_buffersamples ) )
		return false;

	m_parent			= (HWND)syshandle;
	m_state				= Initialized_State;
	return true;
}
//=================================================================================================
void getDriverName
		(
		char*	name
		)
{
	cb_trace( L"id=%d getDriverName\n" , ::GetCurrentThreadId() );
#ifdef cb_debug
	strcpy_s( name , 32 , "AsioKs(debug)" );
#else
	strcpy_s( name , 32 , "AsioKs" );
#endif
}
//=================================================================================================
long getDriverVersion()
{
	cb_trace( L"id=%d getDriverVersion\n" , ::GetCurrentThreadId() );
	return 1;
}
//=================================================================================================
void getErrorMessage
		(
		char*	string
		)
{
	cb_trace( L"id=%d getErrorMessage\n" , ::GetCurrentThreadId() );
}
//=================================================================================================
ASIOError start()
{
	cb_trace( L"id=%d start\n" , ::GetCurrentThreadId() );
	if( m_state != Created_State )
		return ASE_InvalidMode;

	// initialize time
	m_sample_position	= 0.0;
	m_system_time.lo	= 0;
	m_system_time.hi	= 0;
	
	// create device
	if( false == CreateDevice() )
		return ASE_InvalidMode;
	if( false == Thread::Create() )
	{
		DestroyDevice();
		return ASE_InvalidMode;
	}
	Cmd( Play_cmd );
	m_state	= Played_State;
	return ASE_OK;
}
//=================================================================================================
ASIOError stop()
{
	cb_trace( L"id=%d stop\n" , ::GetCurrentThreadId() );
	if( m_state != Played_State )
		return ASE_InvalidMode;

	Cmd( Stop_cmd );
	Thread::Destroy();
	DestroyDevice();
	
	m_state	= Created_State;
	return ASE_OK;
}
//=================================================================================================
ASIOError getChannels
		(
		long*	numInputChannels , 
		long*	numOutputChannels
		)
{
	cb_trace( L"id=%d getChannels\n" , ::GetCurrentThreadId() );
	if( m_state == Null_State )
		return ASE_InvalidMode;

	store( numInputChannels , (long)GetInputChannelMax() );
	store( numOutputChannels , (long)GetOutputChannelMax() );
	return ASE_OK;
}
//=================================================================================================
ASIOError getLatencies
		(
		long*	inputLatency ,
		long*	outputLatency
		)
{
	cb_trace( L"id=%d getLatencies\n" , ::GetCurrentThreadId() );
	if( m_state == Null_State )
		return ASE_InvalidMode;

	store( inputLatency , (long)m_buffersamples );
	store( outputLatency , (long)m_buffersamples );
	return ASE_OK;
}
//=================================================================================================
ASIOError getBufferSize
		(
		long*	minSize ,
		long*	maxSize ,
		long*	preferredSize , 
		long*	granularity
		)
{
	cb_trace( L"id=%d getBufferSize\n" , ::GetCurrentThreadId() );
	if( m_state == Null_State )
		return ASE_InvalidMode;

	store( minSize , (long)m_buffersamples );
	store( maxSize , (long)m_buffersamples );
	store( preferredSize , (long)m_buffersamples );
	store( granularity , (long)0 );
	return ASE_OK;
}
//=================================================================================================
ASIOError canSampleRate
		(
		ASIOSampleRate	sampleRate
		)
{
	cb_trace( L"id=%d canSampleRate\n" , ::GetCurrentThreadId() );
	if( m_state == Null_State )
		return ASE_InvalidMode;

	int		fmtoff , fmtnum = m_devinfo.m_formats.GetDatanum();
	for( fmtoff = 0 ; fmtoff < fmtnum ; fmtoff++ )
	{
		if( m_devinfo.m_formats[fmtoff].m_samplerate == (uint32)sampleRate )
			return ASE_OK;
	}
	return ASE_NotPresent;
}
//=================================================================================================
ASIOError getSampleRate
		(
		ASIOSampleRate*	sampleRate
		)
{
	cb_trace( L"id=%d getSampleRate\n" , ::GetCurrentThreadId() );
	if( m_state == Null_State )
		return ASE_InvalidMode;
	*sampleRate	= m_devinfo.m_formats[m_fmtid].m_samplerate;
	return ASE_OK;
}
//=================================================================================================
ASIOError setSampleRate
		(
		ASIOSampleRate	sampleRate
		)
{
	cb_trace( L"id=%d setSampleRate\n" , ::GetCurrentThreadId() );
	if( m_state == Null_State )
		return ASE_InvalidMode;

	int		fmtoff , fmtnum = m_devinfo.m_formats.GetDatanum();
	for( fmtoff = 0 ; fmtoff < fmtnum ; fmtoff++ )
	{
		if( m_devinfo.m_formats[fmtoff].m_samplerate == (uint32)sampleRate )
			break;
	}
	if( fmtoff == fmtnum )
		return ASE_NoClock;
	
	if( m_state == Played_State )
	{
		stop();
		m_fmtid	= fmtoff;
		m_asio_time.timeInfo.sampleRate		= m_devinfo.m_formats[m_fmtid].m_samplerate;
		m_asio_time.timeInfo.flags			|= kSampleRateChanged;
		if ( 0 != m_callbacks && 0 != m_callbacks->sampleRateDidChange )
			m_callbacks->sampleRateDidChange( m_devinfo.m_formats[m_fmtid].m_samplerate );
		start();
	}
	else
	{
		m_fmtid	= fmtoff;
		m_asio_time.timeInfo.sampleRate		= m_devinfo.m_formats[m_fmtid].m_samplerate;
		m_asio_time.timeInfo.flags			|= kSampleRateChanged;
		if ( 0 != m_callbacks && 0 != m_callbacks->sampleRateDidChange )
			m_callbacks->sampleRateDidChange( m_devinfo.m_formats[m_fmtid].m_samplerate );
	}
	return ASE_OK;
}
//=================================================================================================
ASIOError getClockSources
		(
		ASIOClockSource*	clocks , 
		long*				numSources
		)
{
	cb_trace( L"id=%d getClockSources\n" , ::GetCurrentThreadId() );
	if( clocks != 0 )
	{
		clocks->index				= 0;
		clocks->associatedChannel	= -1;
		clocks->associatedGroup		= -1;
		clocks->isCurrentSource		= ASIOTrue;
		strcpy_s( clocks->name , _countof( clocks->name ) , "Internal");
	}
	store( numSources , (long)1 );
	return ASE_OK;
}
//=================================================================================================
ASIOError setClockSource
		(
		long reference
		)
{
	cb_trace( L"id=%d setClockSource\n" , ::GetCurrentThreadId() );
	if( reference < 0 || reference >0 )
		return ASE_InvalidParameter;

	m_asio_time.timeInfo.flags |= kClockSourceChanged;
	return ASE_OK;
}
//=================================================================================================
ASIOError getSamplePosition
		(
		ASIOSamples*	sPos , 
		ASIOTimeStamp*	tStamp
		)
{
	cb_trace( L"id=%d getSamplePosition\n" , ::GetCurrentThreadId() );

	tStamp->lo = m_system_time.lo;
	tStamp->hi = m_system_time.hi;
	if( m_sample_position >= (double)0x100000000 )
	{
		sPos->hi = (unsigned long)( m_sample_position / (double)0x100000000 );
		sPos->lo = (unsigned long)( m_sample_position - ( sPos->hi * (double)0x100000000 ) );
	}
	else
	{
		sPos->hi = 0;
		sPos->lo = ( unsigned long )m_sample_position;
	}
	return ASE_NotPresent;
}
//=================================================================================================
ASIOError getChannelInfo
		(
		ASIOChannelInfo*	info
		)
{
	cb_trace( L"id=%d getChannelInfo\n" , ::GetCurrentThreadId() );
	if( m_state == Null_State )
		return ASE_InvalidMode;
	if( info->isInput == ASIOTrue )
	{
		if( info->channel < 0 || info->channel >= (int32)GetInputChannelMax() )
			return ASE_InvalidParameter;

		info->isActive		= ASIOTrue;
		info->channelGroup	= 0;
		info->type			= ASIOSTInt32LSB;
		std::ostringstream	os;
		os << "dummy " << ( info->channel + 1 )  << "/" << GetInputChannelMax();
		strcpy_s( info->name , _countof( info->name ) , os.str().c_str() );
		return ASE_OK;
	}
	else if( info->isInput == ASIOFalse )
	{
		if( info->channel < 0 || info->channel >= (int32)GetOutputChannelMax() )
			return ASE_InvalidParameter;

		info->isActive		= ASIOTrue;
		info->channelGroup	= 0;
		info->type			= ASIOSTInt32LSB;
		std::ostringstream	os;
		os << "AsioKs " << ( info->channel + 1 )  << "/" << GetOutputChannelMax();
		strcpy_s( info->name , _countof( info->name ) , os.str().c_str() );
		return ASE_OK;
	}
	else
		return ASE_InvalidParameter;
}
//=================================================================================================
ASIOError createBuffers
		(
		ASIOBufferInfo*	buffer_infos , 
		long			buffer_num ,
		long			buffer_size , 
		ASIOCallbacks*	callbacks
		)
{
	cb_trace( L"id=%d createBuffers\n" , ::GetCurrentThreadId() );
	if( m_state != Initialized_State )
		return ASE_InvalidMode;
	if( buffer_size != m_buffersamples )
		return ASE_InvalidParameter;

	// check channel
	uint32	ch[2] = { 0 , 0 };
	int	buffer_off;
	for( buffer_off = 0 ; buffer_off < buffer_num ; buffer_off++ )
	{
		if( buffer_infos[buffer_off].isInput == ASIOTrue )
		{
			if( buffer_infos[buffer_off].channelNum < 0 ||  buffer_infos[buffer_off].channelNum >= (int32)GetInputChannelMax() )
				return ASE_InvalidMode;
			else
				ch[0]	= max( buffer_infos[buffer_off].channelNum + 1 , (int32)ch[0] );
		}
		if( buffer_infos[buffer_off].isInput == ASIOFalse )
		{
			if( buffer_infos[buffer_off].channelNum < 0 ||  buffer_infos[buffer_off].channelNum >= (int32)GetOutputChannelMax() )
				return ASE_InvalidMode;
			else
				ch[1]	= max( buffer_infos[buffer_off].channelNum + 1 , (int32)ch[1] );
		}
	}

	// create buffer
	if( false == CreateBuffer( ch[0] , ch[1] ) )
		return ASE_InvalidParameter;
		
	// create asio buffer
	for( buffer_off = 0 ; buffer_off < buffer_num ; buffer_off++ )
	{
		if( buffer_infos[buffer_off].isInput == ASIOTrue )
		{
			buffer_infos[buffer_off].buffers[0]	= m_input_ptrs[0][ buffer_infos[buffer_off].channelNum ];
			buffer_infos[buffer_off].buffers[1]	= m_input_ptrs[1][ buffer_infos[buffer_off].channelNum ];
		}
		else if( buffer_infos[buffer_off].isInput == ASIOFalse )
		{
			buffer_infos[buffer_off].buffers[0]	= m_output_ptrs[0][ buffer_infos[buffer_off].channelNum ];
			buffer_infos[buffer_off].buffers[1]	= m_output_ptrs[1][ buffer_infos[buffer_off].channelNum ];
		}
	}
	
	// timeinfo
	if ( 0 != callbacks && 0 != callbacks->asioMessage( kAsioSupportsTimeInfo , 0 , 0 , 0 ) )
	{
		m_need_timeinfo	= true;
		m_asio_time.timeInfo.speed				= 1.;
		m_asio_time.timeInfo.systemTime.hi		= 0;
		m_asio_time.timeInfo.systemTime.lo		= 0;
		m_asio_time.timeInfo.samplePosition.hi	= 0;
		m_asio_time.timeInfo.samplePosition.lo	= 0;
		m_asio_time.timeInfo.sampleRate			= m_devinfo.m_formats[m_fmtid].m_samplerate;
		m_asio_time.timeInfo.flags				= kSystemTimeValid | kSamplePositionValid | kSampleRateValid;
		m_asio_time.timeCode.speed				= 1.;
		m_asio_time.timeCode.timeCodeSamples.lo = 0;
		m_asio_time.timeCode.timeCodeSamples.hi = 0;
		m_asio_time.timeCode.flags				= kTcValid | kTcRunning ;
	}
	else
		m_need_timeinfo = false;	

	// params
	m_callbacks	= callbacks;
	m_state		= Created_State;
	return ASE_OK;
}
//=================================================================================================
ASIOError disposeBuffers()
{
	cb_trace( L"id=%d disposeBuffers\n" , ::GetCurrentThreadId() );
	if( m_state == Null_State )
		return ASE_InvalidMode;
	stop();
	m_callbacks	= 0;
	m_state		= Initialized_State;
	return ASE_OK;
}
//=================================================================================================
ASIOError controlPanel()
{
	cb_trace( L"id=%d controlPanel\n" , ::GetCurrentThreadId() );
	cb_verify( true == m_setting->Create( ComModule::GetModuleHandle() , m_parent ) );
	return ASE_OK;
//	return ASE_NotPresent;
}
//=================================================================================================
ASIOError future
		(
		long	selector ,
		void*	opt
		)
{
	cb_trace( L"id=%d future\n" , ::GetCurrentThreadId() );
	return ASE_NotPresent;
}
//=================================================================================================
ASIOError outputReady()
{
	cb_trace( L"id=%d outputReady\n" , ::GetCurrentThreadId() );
	return ASE_NotPresent;
}

// public functions
public:
//=================================================================================================
AsioKs() : 
		m_parent( NULL ) , 
		m_state( Null_State ) , 
		m_fmtid( -1 ) , 
		m_buffersamples( 256 ) , 
		m_callbacks( 0 ) , 
		m_need_timeinfo( false ) , 
		m_sample_position( 0.0 )
{
	m_system_time.lo	= 0;
	m_system_time.hi	= 0;
	MemoryZero( &m_asio_time , sizeof( m_asio_time ) );
	
	cb_trace( L"AsioKs\n" );
	m_event		= CreateEvent( NULL , FALSE , FALSE , NULL );
	m_event_r	= CreateEvent( NULL , FALSE , FALSE , NULL );
}
//=================================================================================================
~AsioKs()
{
	cb_trace( L"==============~AsioKs=============\n" );
	stop();
	disposeBuffers();
	m_setting->Destroy();
}
};
/**************************************************************************************************
"AsioKsFactory" class 
**************************************************************************************************/
class AsioKsFactory : public ComFactory
{
// variable member
private:

// private functions
private:

// "ComFactory" override functios
protected:
//=================================================================================================
HRESULT CreateComObject
		(
		IUnknown*		outer , 
		const GUID&		riid , 
		void**			pv
		)
{
	if( outer != NULL )
		return CLASS_E_NOAGGREGATION;

	ComInstance<AsioKs>	obj;
	void*	p = 0;
	if( S_OK != obj->QueryInterface( riid , &p ) )
		return E_NOINTERFACE;
	*pv	= p;
	return S_OK;
}
// public functions
public:
};
/**************************************************************************************************
"AsioKsModule" class 
**************************************************************************************************/
class AsioKsModule : public ComModule
{
// variable member
private:

// private functions
private:

// "ComModule" override functions
protected:
//=================================================================================================
HRESULT GetClassObject
		(
		const GUID&		clsid, 
		const GUID&		iid, 
		void**			ppv
		)
{
	if( IID_AsioDriver() == clsid )
	{
		ComInstance<AsioKsFactory>	obj;
		if( S_OK != obj->QueryInterface( iid , ppv ) )
			return E_NOINTERFACE;
		return S_OK;
	}
    return CLASS_E_CLASSNOTAVAILABLE;
}
//=================================================================================================
HRESULT RegisterServer()
{
#ifdef cb_debug
	if( false == RegistAsioDriver( GetModuleHandle() , IID_AsioDriver::get_const() , L"AsioKs Driver(debug)" , L"AsioKs(debug)" , L"Apartment" ) )
		return E_FAIL;
#else
	if( false == RegistAsioDriver( GetModuleHandle() , IID_AsioDriver::get_const() , L"AsioKs Driver" , L"AsioKs" , L"Apartment" ) )
		return E_FAIL;
#endif
	return S_OK;
}
//=================================================================================================
HRESULT UnregisterServer()
{
#ifdef cb_debug
	if( false == UnregistAsioDriver( IID_AsioDriver::get_const() , L"AsioKs Driver(debug)" ) )
		return E_FAIL;
#else
	if( false == UnregistAsioDriver( IID_AsioDriver::get_const() , L"AsioKs Driver" ) )
		return E_FAIL;
#endif
	return S_OK;
}
// public functions
public:
};
///////////////////////////////////////////////////////////////////////////////////////////////////
// global variable define

///////////////////////////////////////////////////////////////////////////////////////////////////
// global functions define


//using namespace icubic;		

#pragma pack( pop )			//release align
