/*
 *  Copyright (C) 2006  Takashi Kasuya <kasuya@sfc.keio.ac.jp>
 *
 * This library is free software; you can redistribute it and/or
 *@modify it under the terms of the GNU Lesser General Public
 *@License as published by the Free Software Foundation; either
 *@version 2.1 of the License, or (at your option) any later version.
 *@This library is distributed in the hope that it will be useful,
 *@but WITHOUT ANY WARRANTY; without even the implied warranty of
 *@MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *@Lesser General Public License for more details.
 *
 *@You should have received a copy of the GNU Lesser General Public
 *@License along with this library; if not, write to the Free Software
 *@Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package jp.ac.naka.ec.sip;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.sip.RequestEvent;
import javax.sip.address.Address;
import javax.sip.address.SipURI;
import javax.sip.header.ContactHeader;
import javax.sip.header.EventHeader;
import javax.sip.header.ExpiresHeader;
import javax.sip.header.ToHeader;
import javax.sip.message.Request;
import javax.sip.message.Response;
import javax.xml.parsers.ParserConfigurationException;

import jp.ac.naka.ec.entity.AbstractEntity;
import jp.ac.naka.ec.entity.Entity;
import jp.ac.naka.ec.entity.EntityContainer;
import jp.ac.naka.ec.entity.EntityContainerImpl;
import jp.ac.naka.ec.entity.EntityEvent;
import jp.ac.naka.ec.entity.EntityImpl;
import jp.ac.naka.ec.entity.EntityEvent.EventType;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.SAXException;

/**
 * 
 * @author Takashi Kasuya
 * 
 */
class SubscribeDelegate implements MethodDelegate {
	public static int MIN_EXPIRES = 5;

	// message-summary (X-Lite)
	private final static String[] acceptPackages = { "presence", "message-summary" };
	private static Log logger = LogFactory.getLog(SubscribeDelegate.class);
	private static SipCore core;
	private static EntityContainer container;
	private Map<String, Subscriber> subscribers = new HashMap<String, Subscriber>();

	public SubscribeDelegate() throws ParserConfigurationException,
			SAXException {
		core = SipCore.getInstance();
		container = EntityContainerImpl.getInstance();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.ac.naka.ec.sip.MethodDelegate#forward(javax.sip.RequestEvent)
	 */
	public boolean forward(RequestEvent evt) {

		Request req = evt.getRequest();
		ExpiresHeader expires;
		try {
			// Event, ExpiresHeader
			EventHeader event = (EventHeader) req.getHeader(EventHeader.NAME);
			if (!checkEventHeader(event)) {
				logger.warn("Bad event.");
				core.sendResponse(Response.BAD_EVENT, req);
				return false;
			}
			expires = (ExpiresHeader) req.getHeader(ExpiresHeader.NAME);

			
			ToHeader to = (ToHeader) req.getHeader(ToHeader.NAME);
			SipURI uri = (SipURI) to.getAddress().getURI();
			// Notify̏
			Entity target = container.getEntity(uri.toString());
			
			ContactHeader contact = (ContactHeader) req.getHeader(ContactHeader.NAME);
			
			Address addr = contact.getAddress();
			SipURI sourceURI = (SipURI) contact.getAddress().getURI();
			
			// Expires Out
			if (expires.getExpires() == 0) {
				Subscriber subscriber = subscribers.get(uri);
				subscriber.remove();
				// 200 OK
				core.sendResponse(Response.OK, evt, addr.getURI().toString());
				return true;
			}
			else if (!checkExpiresHeader(expires)) {
				logger.warn("Interval is too brief.");
				core.sendResponse(Response.INTERVAL_TOO_BRIEF, req);
				return false;

			}
			
			// sip client̂ߎb菈u
			if (target == null)
				target = container.getEntity(sourceURI.toString());
			
			Entity source = new EntityImpl(sourceURI);
			if (target == null ) {
				// Error
				core.sendResponse(Response.NOT_FOUND, req);
				return false;
			} else {
				//TODO F
				if (!isAuthorized(req)) {
					// 401M
					core.sendResponse(401, req);
					return false;
				} 
				
				// F؍ς
				EntityEvent ee = new EntityEvent(target, source, EventType.SUBSCRIBE);
				((AbstractEntity)target).handleEvent(ee);
				
			}
			
			Subscriber subscriber;
			String uri_str = uri.toString();
			if (subscribers.containsKey(uri)) {
				subscriber = subscribers.get(uri);
				subscriber.refresh();
			} else {
				subscriber = new Subscriber(expires.getExpires(), target, source);
				subscribers.put(uri_str, subscriber);
				if (target instanceof AbstractEntity) 
					((AbstractEntity)target).addSubscriber(source);
			}
			
			// 200 OK
			core.sendResponse(Response.OK, evt, addr.getURI().toString());
			
			// NotifyM
			if (target instanceof AbstractEntity) {
				((AbstractEntity)target).startNotify(source);
			}
		} catch (Exception e1) {
			//e1.printStackTrace();
			logger.warn("Error while send a response.", e1);
			return false;
		}

		return true;
	}

	private boolean isAuthorized(Request req) {
		// TODO Auto-generated method stub
		return true;
	}

	/**
	 * 
	 * @param expires
	 * @return
	 */
	private boolean checkExpiresHeader(ExpiresHeader expires) {
		int expire = expires.getExpires();
		return (expire > MIN_EXPIRES) ? true : false;
	}

	/**
	 * 
	 * @param header
	 * @return
	 */
	private boolean checkEventHeader(EventHeader header) {
		String type = header.getEventType();
		for (String str : acceptPackages) {
			if (str.equals(type)) {
				return true;
			}
		}
		return false;
	}

	class Subscriber implements Runnable {

		int expire;

		
		Entity target, source;
		ExecutorService ex = Executors.newSingleThreadExecutor();
		Thread th;

		public Subscriber(int expire, Entity listener, Entity source) {
			this.expire = expire;
			this.source = source;
			if (listener instanceof AbstractEntity) {
				this.target =  listener;
				ex.execute(this);
			}
		}

		public void run() {
			try {
				//System.out.println("start");
				Thread.sleep(expire * 1000);
				remove();
			} catch (InterruptedException e) {
			}
		}
		
		void remove() {
			EntityEvent evt = new EntityEvent(target, source, EntityEvent.EventType.REMOVE);
			((AbstractEntity)target).handleEvent(evt);
			((AbstractEntity)target).removeSubscriber(source);
			subscribers.remove(target.getURI().toString());
		}

		public void refresh() {
			ex.shutdownNow();
			ex.execute(this);
		}
	}
	
}
