/*
 * Copyright (c) 2002-2004 David Keiichi Watanabe
 * davew@xlife.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.acquisitionx;

import com.limegroup.gnutella.*;
import com.limegroup.gnutella.tigertree.TigerTreeCache;
import com.limegroup.gnutella.downloader.ManagedDownloader;
import com.limegroup.gnutella.downloader.IncompleteFileManager;
import com.limegroup.gnutella.licenses.LicenseFactory;
import com.limegroup.gnutella.gui.DaapManager;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.util.CommonUtils;
import java.io.File;
import java.io.PrintStream;
import java.io.InputStreamReader;
import java.util.Vector;
import java.util.Iterator;
import java.util.List;


public class AqMain
{

/* Instances */
    
    public static Thread cThread;
    public static Thread tThread;
    public static Thread dThread;
    //public static Thread gcThread;
    public static Thread dispatchThread;
    
    public static boolean coreHasStarted = false;
    public static String bundlePath;

/* Main */

    public static void main(String[] args) 
    {
        System.err.println(args[0]);
        bundlePath = args[0];

        (new Thread(new ReaderThread())).start();

        /* create app support directory */

            if (!CommonUtils.getUserSettingsDir().exists()) {
                CommonUtils.getUserSettingsDir().mkdirs();
            }

        /* start LW */

            RouterService service = new RouterService(new AqEventHandler());
            service.preGuiInit();
            service.start();    

        /* launch threads */

            cThread = new Thread(new ConnectionUpdate());
            tThread = new Thread(new TransferUpdate());
            dThread = new Thread(new DownloadRetry());
            //gcThread = new Thread(new TwoMinuteThread());
            dispatchThread = new Thread(new DispatchThread());

            cThread.setDaemon(true);
            tThread.setDaemon(true);
            dThread.setDaemon(true);
            //gcThread.setDaemon(true);
            dispatchThread.setDaemon(true);

            cThread.start();
            tThread.start();
            dThread.start();
            //gcThread.start();
            dispatchThread.start();

            coreHasStarted = true;
    }
    
    public static String getBundlePath() {
        return bundlePath;
    }
    
    public static void cleanIncompleteDirectory() throws Exception {
        File incompleteDir = SharingSettings.INCOMPLETE_DIRECTORY.getValue();
        File[] files = incompleteDir.listFiles();

        if (files != null) {
            for (int i=0; i<files.length; i++) {
                File f = files[i];
                if (f.getName().startsWith(IncompleteFileManager.PREVIEW_PREFIX)) {
                    f.delete();
                } else if (f.length() == 0) {
                    f.delete();
                }
            }
        }
    }

    public static void killSelf() {
        AqEvent.shouldSignalEvents = false;
        if (AqMain.cThread != null) AqMain.cThread.interrupt();
        if (AqMain.tThread != null) AqMain.tThread.interrupt();
        if (AqMain.dThread != null) AqMain.dThread.interrupt();
        //if (AqMain.gcThread != null) AqMain.gcThread.interrupt();
        if (AqMain.dispatchThread != null) AqMain.dispatchThread.interrupt();

        shutdown();
        System.exit(0);
    }
    
    public static void shutdown() {
        try {
            RouterService.getAcceptor().haltUPnP();
            Statistics.instance().shutdown();
            DaapManager.instance().stop();
            
            ConnectionSettings.FORCE_IP_ADDRESS.revertToDefault();
            ConnectionSettings.FORCED_IP_ADDRESS_STRING.revertToDefault();
            ConnectionSettings.EVER_ACCEPTED_INCOMING.setValue(
                RouterService.acceptedIncomingConnection());

            /* limewire.props */
            AqEvent.signalEvent(AqEvent.kShutdownSettingsHandler);
            SettingsHandler.save();
            
            /* fileurns.cache */
            AqEvent.signalEvent(AqEvent.kShutdownURNCache);
            UrnCache.instance().persistCache();
            CreationTimeCache.instance().persistCache();
            TigerTreeCache.instance().persistCache();
            LicenseFactory.persistCache();

            /* downloads.cache */
            AqEvent.signalEvent(AqEvent.kShutdownDownloadsCache);
            RouterService.getDownloadManager().writeSnapshot();

            /* clean incomplete */
            AqEvent.signalEvent(AqEvent.kShutdownCleanDirectory);
            AqMain.cleanIncompleteDirectory();

        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
    

/* System.in */

/* Incoming Pipe */

    /* 
        The following is a listening thread which receives control instructions from the primary Acquisition process.  When the pipe is broken (the parent process has exited), we calmly clean up everything and exit.
    */

    class ReaderThread implements Runnable 
    {
        public static Vector dispatches = new Vector();

        public void run() 
        {
            try 
            {
                Vector v = new Vector();
                StringBuffer fragment = new StringBuffer();
                int b;
            
                System.err.println("ReaderThread: Listening to System.in");
                
                /* read loop */
    
                    InputStreamReader reader = new InputStreamReader(System.in, "UTF-8");
                    while ((b = reader.read()) > -1)
                    {
                        char character = (char)b;
                        
                        if (character == '|' || character == '\n') {
                            v.add(fragment.toString());
                            fragment = new StringBuffer();
        
                            if (character == '\n') {
                                synchronized (ReaderThread.dispatches) {
                                    dispatches.add(v);
                                }
                                v = new Vector();
                            }
                        } else {
                            fragment.append(character);
                        }
                    }

                /* shutdown - under this new scheme, the following will likely never be executed */
                
                    System.err.println("ReaderThread: System.in EOF");
                    
                    /* kill thread that will terminate the JVM in a few seconds, *regardless* of what happens next */
                    
                    (new Thread() {
                        public void run() {
                            try {
                                Thread.sleep(10000); /* 10 seconds */
                            } catch (Exception e) {}
                            System.err.println("ReaderThreadTerminator: killing");
                            System.exit(0);
                        }
                    }).start();
                    
                    AqMain.killSelf();
                
            } catch (Exception e) {
                e.printStackTrace();
                System.exit(0);
            }
        }
    }


/* Dispatch */

    class DispatchThread implements Runnable {
        public void run() {
            while (true) {
                if (AqMain.coreHasStarted) 
                {
                    Vector dispatches = new Vector();

                    synchronized (ReaderThread.dispatches) {
                        dispatches.addAll(ReaderThread.dispatches);
                        ReaderThread.dispatches.removeAllElements();
                    }
                    
                    for (int i=0; i<dispatches.size(); i++) {
                        try {
                            AqDispatcher.dispatchCommand((Vector)dispatches.get(i));
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
                try {
                    Thread.sleep(300); /* 0.3 second */
                } catch (Exception e) {
                    break;
                }
            }
        }
    }


/* ConnectionUpdate */

    class ConnectionUpdate implements Runnable {
        public void run() {
            while (true) {

                ConnectionManager cm = RouterService.getConnectionManager();
                Iterator i;
                    
                try {

                    i = cm.getInitializedConnections().iterator();
                    while (i.hasNext()) publishUpdate((Connection)i.next());
    
                    i = cm.getInitializedClientConnections().iterator();
                    while (i.hasNext()) publishUpdate((Connection)i.next());
    
                    AqEvent.signalEvent(AqEvent.kLWEventConnectionsUpdated);
                    
                    /* Sleep */

                        int total = cm.getNumInitializedConnections() + cm.getNumInitializedClientConnections();
                        try {
                            if (total > 10)
                                Thread.sleep(5000);  /* 5 seconds */
                            else
                                Thread.sleep(2000);  /* 2 seconds */
                        } catch (InterruptedException e) {
                            break;
                        }

                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
        
        public void publishUpdate(Connection c) {
            AqEvent.signalEvent(AqEvent.kLWEventUpdateConnectionStats, c);
        }
    }


/* TransferUpdate */

    class TransferUpdate implements Runnable {
        public void run() {

            DownloadManager dm = RouterService.getDownloadManager();
            UploadManager um = RouterService.getUploadManager();

            while (true) {
                try {
                    List waiting = dm.getWaitingDownloads();
                    List active = dm.getActiveDownloads();
                    List uploads = um.getUploads();
                    
                    Iterator i;
                    
                    /* Downloads */
        
                        i = active.iterator();
                        while (i.hasNext()) publishUpdate((Downloader)i.next());
        
                        i = waiting.iterator();
                        while (i.hasNext()) publishUpdate((Downloader)i.next());
        
                        AqEvent.signalEvent(AqEvent.kLWEventDownloadsUpdated);
                        
                    /* Uploads */
            
                        i = uploads.iterator();
                        while (i.hasNext()) publishUpdate((Uploader)i.next());

                        AqEvent.signalEvent(AqEvent.kLWEventUploadsUpdated);

                    /* Sleep */

                        try {
                            Thread.sleep(1000);  /* 1 second */
                        } catch (InterruptedException e) {
                            break;
                        }
 
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
        
        public void publishUpdate(Downloader d) {
            if(d != null)
                AqEvent.signalEvent(AqEvent.kLWEventUpdateDownloadStats, d);
        }

        public void publishUpdate(Uploader u) {
            if(u != null)
                AqEvent.signalEvent(AqEvent.kLWEventUpdateUploadStats, u);
        }
    }
    

/* DownloadRetry */

    class DownloadRetry implements Runnable {
        public void run() {
            while (true) {

            /* Sleep */

                try {
                    Thread.sleep(60 * 60 * 1000);  /* 60 minutes */
                } catch (InterruptedException e) {
                    break;
                }
                    
            /* Downloads */

                try {
                    List waiting = RouterService.getDownloadManager().getWaitingDownloads();
                    Iterator i = waiting.iterator();
                    
                    while (i.hasNext()) {
                        ManagedDownloader d= (ManagedDownloader)i.next();
                        d.sendRequeryImmediately();
                    }

                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }
    

/* TwoMinuteThread */

    /* heavy
	class TwoMinuteThread implements Runnable {
        public void run() {
            while (true) {
                try {
                    Thread.sleep(120 * 1000);  // 120 seconds
                    System.gc();
                    RouterService.getHostCatcher().write(); // network.cache
                    TigerTreeCache.instance().persistCache();
                    CreationTimeCache.instance().persistCache();
               } catch (Exception e) {
                    break;
                }
            }
        }
    }
    */
