/*
 * Copyright (c) 2006 - 2006, the original author or authors. All rights
 * reserved. Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met: *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer. * Redistributions in binary
 * form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided
 * with the distribution. * Neither the name of the University of California,
 * Berkeley nor the names of its contributors may be used to endorse or promote
 * products derived from this software without specific prior written
 * permission. THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sf.bli.framework.validate;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

import jp.sf.bli.framework.logging.Logging;

/**
 * Validate実行Abstractクラス
 * 
 * @author takeda 2007/01/30
 */
public abstract class AbstractVlidateObject {

    /**
     * Validate設定プロパティのboolean型の設定を取得する
     * 
     * @param ログイン名
     * @param Validate設定プロパティファイルを割り当てたResourceBundle
     * @param Validateを行うフィールド名
     * @param Validate実行種類をあらわす文字列
     * @return 該当Validate実行種類を実行するなら true、しないなら false
     * @throws ValidateException
     */
    protected static boolean chkNameMakeBoolean( String loginName,
            ResourceBundle bundleConfing, String fieldName, String chkType )
            throws ValidateException {

        boolean rtnBoolean = false;

        try {

            rtnBoolean = Boolean.valueOf( bundleConfing.getString( fieldName
                    + "_" + chkType ) );

        } catch ( MissingResourceException e ) {

            Logging.debug( AbstractVlidateObject.class, "BLI000038", loginName,
                    "VlidateFrame chkNameMakeBooleanでMissingResourceExceptionが発生 Validate設定ファイル中の "
                            + fieldName + "_" + chkType
                            + "の設定が意図的にされていないなら問題ありません" );

            // 設定が無い事も可なので、この場合のMissingResourceExceptionは無視する
            return rtnBoolean;
        }

        return rtnBoolean;

    }


    /**
     * Validate設定プロパティのint型の設定を取得する
     * 
     * @param ユーザID
     * @param Validate設定プロパティファイルを割り当てたResourceBundle
     * @param Validateを行うフィールド名
     * @param Validate実行種類をあらわす文字列
     * @return int
     * @throws ValidateException
     */
    protected static int chkNameMakeInt( String loginName,
            ResourceBundle bundleConfing, String fieldName, String chkType )
            throws ValidateException {

        int rtnInt = 0;

        try {

            rtnInt = Integer.parseInt( bundleConfing.getString( fieldName + "_"
                    + chkType ) );

        } catch ( NumberFormatException e ) {

            e.printStackTrace();

            Logging
                    .error( AbstractVlidateObject.class, "BLI000039",
                            loginName,
                            "VlidateFrame chkNameMakeIntでNumberFormatExceptionが発生 Validate設定ファイル中の "
                                    + fieldName + "_" + chkType
                                    + "の設定に数字にならない文字列があります" );

            throw new ValidateException(
                    "VlidateFrame chkNameMakeIntでNumberFormatExceptionが発生 "
                            + fieldName + "_" + chkType + "の設定に数字にならない文字列があります" );

        } catch ( MissingResourceException e ) {

            // 設定が無い事も可なので、この場合のMissingResourceExceptionは無視する

            Logging.debug( AbstractVlidateObject.class, "BLI000040", loginName,
                    "VlidateFrame chkNameMakeIntでMissingResourceExceptionが発生 Validate設定ファイル中の "
                            + fieldName + "_" + chkType
                            + "の設定が意図的にされていないなら問題ありません" );

            return rtnInt;
        }

        return rtnInt;

    }


    protected static String chkNameMakeString( String loginName,
            ResourceBundle bundleConfing, String fieldName, String chkType )
            throws ValidateException {

        String rtnStr = null;

        try {

            rtnStr = bundleConfing.getString( fieldName + "_" + chkType );

        } catch ( MissingResourceException e ) {

            // 設定が無い事も可なので、この場合のMissingResourceExceptionは無視する
            Logging.debug( AbstractVlidateObject.class, "BLI000041", loginName,
                    "VlidateFrame chkNameMakeStringでMissingResourceExceptionが発生 Validate設定ファイル中の "
                            + fieldName + "_" + chkType
                            + "の設定が意図的にされていないなら問題ありません" );

            return rtnStr;
        }

        return rtnStr;
    }


    /**
     * エラーメッセージ整形メソッド ユーザID
     * 
     * @param Validate設定プロパティファイルを割り当てたResourceBundle
     * @param Validateを行うフィールド名
     * @param Validate実行種類をあらわす文字列
     * @param エラーメッセーじ改行文字列
     * @return エラーメッセージ
     * @throws ValidateException
     */
    protected static String errorMesMake( String loginName,
            ResourceBundle bundleMes, String fieldName, String chkType,
            String errorMesLinefeed, GetMessageDB getMessageDB )
            throws ValidateException {

        String rtnStr;
        try {

            if ( bundleMes != null ) {

                rtnStr = "ErrorCode:"
                        + bundleMes.getString( fieldName + "_" + chkType
                                + "_code" )
                        + " [ "
                        + bundleMes.getString( fieldName + "_" + chkType
                                + "_mes" ) + " ]" + errorMesLinefeed;

            } else {

                rtnStr = "ErrorCode:"
                        + getMessageDB.getErrorMesCode( fieldName + "_"
                                + chkType ) + " [ "
                        + getMessageDB.getErrorMes( fieldName + "_" + chkType )
                        + " ]" + errorMesLinefeed;

            }

        } catch ( MissingResourceException e ) {

            e.printStackTrace();

            Logging
                    .error(
                            AbstractVlidateObject.class,
                            "BLI000042",
                            loginName,
                            "VlidateFrame errorMesMakeでMissingResourceExceptionが発生 ValidateMessage設定ファイル中の "
                                    + fieldName + "_" + chkType + "の設定がありません" );

            throw new ValidateException(
                    "VlidateFrame errorMesMakeでMissingResourceExceptionが発生 "
                            + fieldName + "_" + chkType + "_errorMes"
                            + "の設定がありません" );
        }

        return rtnStr;
    }
}
