/*
 * Copyright (C) 2006-2007 BLI Project
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms,
 * with or without modification, are permitted provided
 * that the following conditions are
 * met:
 * 
 * 1. Redistributions of source code must retain
 * the above copyright notice, this list of conditions
 * and the following disclaimer as the first lines of this file unmodified.
 * 
 * 2. Redistributions  in  binary  form must reproduce 
 * the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.sf.bli.framework.batch;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.sf.bli.framework.common.BusinessLogicException;
import jp.sf.bli.framework.logging.Logging;
import jp.sf.bli.framework.validate.GetMessageDB;
import jp.sf.bli.framework.validate.ValidateCtrl;

/**
 * バッチ処理用 ビジネスロジック Validateクラスのabstractクラス
 * 
 * @author $Author: gackt $
 * @version $id$
 */
abstract public class BusinessLogicValidateBT {

    /** Validate Confign File Path */
    protected String       validateConfigFile  = "";

    /** Validate Message Confing File Path */
    protected String       validateMessageFile = "";

    protected GetMessageDB getMessageDB        = null;


    /**
     * @return this.validateConfigFile
     */
    public String getValidateConfigFile() {

        return this.validateConfigFile;
    }


    /**
     * @param validateConfigFile
     *            this.validateConfigFile = validateConfigFile
     */
    public void setValidateConfigFile( String validateConfigFile ) {

        this.validateConfigFile = validateConfigFile;
    }


    /**
     * @return this.validateMessageFile
     */
    public String getValidateMessageFile() {

        return this.validateMessageFile;
    }


    /**
     * @param validateMessageFile
     *            this.validateMessageFile = validateMessageFile
     */
    public void setValidateMessageFile( String validateMessageFile ) {

        this.validateMessageFile = validateMessageFile;
    }

    /**
     * @return this.getMessageDB
     */
    public GetMessageDB getGetMessageDB() {

        return this.getMessageDB;
    }


    /**
     * DBからValidate時のエラーメッセージを取得する場合はSpringにて<br />
     * jp.sf.bli.framework.validate.GetMessageDB を実装した<br />
     * クラスをこのメソッドでインジェクションする
     * 
     * @param getMessageDB
     *            this.getMessageDB = getMessageDB
     */
    public void setGetMessageDB( GetMessageDB getMessageDB ) {

        this.getMessageDB = getMessageDB;
    }

    /**
     * inputValueMapの値をValidateチェックして、<br />
     * 結果をoutputValueMapにresultFlagというKeyでInteger型をセットして返す<br />
     * 論理的にエラーになった場合はoutputValueMapのresultFlagに0をセットする。
     * 処理が障害によって止まったらBusinessLogicExceptionをスローする。<br />
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws BusinessLogicException
     *             実行時に各種Exceptionが起きた場合に<br />
     *             BusinessLogicException発生させてthrowする
     */
    public void doValidate( String loginName,
            Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap )
            throws BusinessLogicException {

        try {
            init( loginName, inputValueMap, outputValueMap );

            initProcess( loginName, inputValueMap, outputValueMap );

            doValidateMain( loginName, inputValueMap, outputValueMap );

            endProcess( loginName, inputValueMap, outputValueMap );

            end( loginName, inputValueMap, outputValueMap );

        } catch ( Exception e ) {

            Logging.fatal( this.getClass(), "BLI000032", loginName,
                    "doValidate:BlValidate catch Exception. " + e.getMessage() );

            e.printStackTrace();
            throw new BusinessLogicException( "");

        }
    }


    /**
     * BlValidateの初期処理前のシステムの初期処理を行う
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    protected void init( String loginName, Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception {

        Logging.debug( this.getClass(), "BLI000033", loginName,
                "doBizLogic:BlDo started." );
    }


    /**
     * BlValidateの実行を行う前に初期処理を行う
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    abstract protected void initProcess( String loginName,
            Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception;


    /**
     * BlValidateのメイン処理を行う<br />
     * validateメソッドで呼び出すように記述するとFrameworkのvalidateが使用可能
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    abstract protected void doValidateMain( String loginName,
            Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception;


    /**
     * BlValidateの実行後に終了処理を行う
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    abstract protected void endProcess( String loginName,
            Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception;


    /**
     * BlValidateの終了後の最終処理を行う
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    protected void end( String loginName, Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception {

        Logging.debug( this.getClass(), "BLI000034", loginName,
                "doBizLogic:BlDo Ended." );

    }



    /**
     * doValidateMainメソッドから呼び出してValidate処理を行う
     * 
     * @param loginName
     * @param checkList
     * @param erroLineFeedType
     * @return
     * @throws Exception
     */
    public String validate( String loginName, List< Object[] > checkList,
            int erroLineFeedType ) throws Exception {

        String errorMes = "";

        Iterator itr = checkList.iterator();

        while ( itr.hasNext() ) {

            Object[] obj = ( Object[] ) itr.next();

            errorMes = errorMes
                    + ValidateCtrl.validate( loginName, ( String ) obj[0],
                            obj[1], erroLineFeedType, this.validateConfigFile,
                            this.validateMessageFile, this.getMessageDB );

        }

        return errorMes;

    }


}
