/*
 * Copyright (C) 2006-2007 BLI Project
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms,
 * with or without modification, are permitted provided
 * that the following conditions are
 * met:
 * 
 * 1. Redistributions of source code must retain
 * the above copyright notice, this list of conditions
 * and the following disclaimer as the first lines of this file unmodified.
 * 
 * 2. Redistributions  in  binary  form must reproduce 
 * the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.sf.bli.framework.batch;

import java.util.Map;

import jp.sf.bli.framework.common.BusinessLogicException;
import jp.sf.bli.framework.logging.Logging;

/**
 * バッチ処理用 ビジネスロジック実行クラスのabstractクラス
 * 
 * @author $Author: gackt $
 * @version $id$
 */
abstract public class BusinessLogicDoBT {

    /**
     * BlCtrlから呼び出されてビジネスロジックの処理を行う。<br />
     * 記述するビジネスロジック処理は実装を最小にしてオブジェクトの<br />
     * 粒度を小さくする。<br />
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws BusinessLogicException
     */
    public void doBizLogic( String loginName,
            Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap )
            throws BusinessLogicException {

        try {
            init( loginName, inputValueMap, outputValueMap );

            initProcess( loginName, inputValueMap, outputValueMap );

            doBizLogicMain( loginName, inputValueMap, outputValueMap );

            endProcess( loginName, inputValueMap, outputValueMap );

            end( loginName, inputValueMap, outputValueMap );

        } catch ( Exception e ) {

            Logging.fatal( this.getClass(), "BLI000029", loginName,
                    "doBizCtrl:BLCtrl catch Exception. " + e.getMessage() );

            e.printStackTrace();
            throw new BusinessLogicException( "");

        }
    }


    /**
     * BlDoの初期処理前のシステムの初期処理を行う
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    protected void init( String loginName, Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception {

        Logging.debug( this.getClass(), "BLI000030", loginName,
                "doBizLogic:BlDo started." );
    }


    /**
     * BlDoの実行を行う前に初期処理を行う
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    abstract protected void initProcess( String loginName,
            Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception;


    /**
     * BlDoのメイン処理を行う。<br />
     * 処理結果をoutputValueMapにresultFlagというKeyでInteger型をセットして返す<br />
     * 論理的にエラーになった場合はoutputValueMapのresultFlagに0をセットする。<br />
     * 処理が障害によって止まったらBusinessLogicExceptionをスローする。<br />
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    abstract protected void doBizLogicMain( String loginName,
            Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception;


    /**
     * BlDoの実行後に終了処理を行う
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    abstract protected void endProcess( String loginName,
            Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception;


    /**
     * BlDoの終了後の最終処理を行う
     * 
     * @param loginName
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    protected void end( String loginName, Map< String, Object > inputValueMap,
            Map< String, Object > outputValueMap ) throws Exception {

        Logging.debug( this.getClass(), "BLI000031", loginName,
                "doBizLogic:BlDo Ended." );

    }

}
