/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.core;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import benten.core.BentenConstants;
import benten.core.io.Files;
import benten.core.io.XliffFileFilter;
import benten.core.model.BentenXliff;
import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.valueobject.BentenConvertToXliffProcessInput;
import benten.twa.filter.messages.BentenConvertToXliffMessages;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import blanco.xliff.BlancoXliffUtil;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffHeader;
import blanco.xliff.valueobject.BlancoXliffTool;

/**
 * 翻訳対象を XLIFF に変換
 *
 * <pre>
 * 翻訳対象物を XLIFF に変換します。
 *   1.  プロジェクト内にインポートされた翻訳対象物を、XLIFF に変換します。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳対象物－翻訳中間形式変換機能: HTML－XLIFF変換機能」に対応します。
 *
 * @author KASHIHARA Shinji
 */
public class BentenConvertToXliffProcessImpl extends AbstractTraverseDir implements BentenConvertToXliffProcess {
	/**
	 * HTML－XLIFF変換機能のためのメッセージ。
	 */
	protected static final BentenConvertToXliffMessages fMsg = new BentenConvertToXliffMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenConvertToXliffProcessInput fInput;

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * 処理クラスのリスト。
	 */
	protected List<BentenTwaFilterProcessor> fTwaProcessors = new ArrayList<BentenTwaFilterProcessor>();

	/**
	 * コンストラクタ。
	 */
	public BentenConvertToXliffProcessImpl() {
		fTwaProcessors.add(new BentenTwaFilterHtmlProcessor());
		fTwaProcessors.add(new BentenTwaFilterTextProcessor());
		fTwaProcessors.add(new BentenTwaFilterJavaPropertyResourceBundleProcessor());
		fTwaProcessors.add(new BentenTwaFilterEcma376DocxProcessor());
		fTwaProcessors.add(new BentenTwaFilterEcma376XlsxProcessor());
		fTwaProcessors.add(new BentenTwaFilterEcma376PptxProcessor());
	}

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenConvertToXliffProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenConvertToXliffProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenConvertToXliffProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File dirSource = new File(fInput.getSourcedir());
		if (dirSource.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE004(fInput.getSourcedir()));
		}
		if (dirSource.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE005(fInput.getSourcedir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenConvertToXliffProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) throws IOException {
				fCounter++;
			}

			@Override
			protected void processFilterdFile(final File file, final String baseDir) throws IOException {
				// こちらはカウントしません。
			}

			/**
			 * カウンタ数の取得。
			 * @return カウンタ数。
			 */
			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(input);
		inner.processDir(dirSource);
		beginTask(inner.getCounter());

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		processDir(dirSource);

		if (progress(fMsg.getCoreP003(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		// 処理できるファイルかどうかを判定。
		for (BentenTwaFilterProcessor processor : fTwaProcessors) {
			if (processor.canProcess(file)) {
				return true;
			}
		}
		return false;
	}

	@Override
	public void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenConvertToXliffProcessImpl#processFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		final File destFile = new File(fInput.getTargetdir(), Files.relativePath(new File(fInput.getSourcedir()), file));
		final File destDir = destFile.getParentFile();

		// 該当ディレクトリーが無い場合には、ディレクトリー作成。
		// すでに存在する場合は、それはそれで良い。

		if (destDir.exists() == false) {
			if (destDir.mkdirs() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE006(fInput.getTargetdir()));
			}
		} else {
			if (destDir.isDirectory() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE007(fInput.getTargetdir()));
			}
		}

		final File outFile = XliffFileFilter.appendExtension(new File(destDir, file.getName()));

		final BentenXliff xliff = BentenXliff.newInstance(outFile);
		final BlancoXliffFile xliffFile = xliff.createBlancoXliffFile();
		xliffFile.setSourceLanguage(fInput.getTranssourcelang());
		xliffFile.setTargetLanguage(fInput.getTranstargetlang());

		final BlancoXliffHeader header = xliffFile.getHeader();

		final BlancoXliffTool tool = new BlancoXliffTool();
		tool.setToolId(BentenConstants.PROJECT_NAME_LOWER);
		tool.setToolName(BentenConstants.PROJECT_NAME);
		BlancoXliffUtil.setTool(header, tool);

		final String path = Files.relativePath(new File(fInput.getSourcedir()).getParentFile(), file);
		final HelpTransUnitId id = new HelpTransUnitId(fInput.getTranstargetid() + path);

		for (BentenTwaFilterProcessor processor : fTwaProcessors) {
			if (processor.canProcess(file)) {
				processor.extractXliffFromSource(file, xliffFile, id, fResultInfo);
				break;
			}
		}

		xliff.save();

		getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
	}
}
