/*******************************************************************************
 * Copyright (c) 2010 IGA Tosiki.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
/*
 * Copyright (C) 2010 IGA Tosiki.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package benten.twa.filter.core;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.Iterator;
import java.util.Properties;
import java.util.Map.Entry;

/**
 * プロパティファイル処理の共通処理を担う抽象クラス。
 * 
 * @author IGA Tosiki
 */
abstract class AbstractJavaPropertiesFileProcessor {

	/**
	 * 1 つのプロパティファイルを処理します。
	 * 
	 * @param bytesFile 入力プロパティファイル・バイト配列。UTF-8 でエンコードされている必要があります。
	 * @param writer 出力プロパティファイル・ライター。
	 * @return 値の箇所を 1 箇所でもライターに出力処理したかどうか。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	boolean process(final byte[] bytesFile, final Writer writer) throws IOException {
		boolean isProcessed = false;

		final String text = new String(bytesFile, "UTF-8"); //$NON-NLS-1$

		final StringReader reader = new StringReader(text);

		final Properties props = new Properties();
		props.load(new ByteArrayInputStream(bytesFile));

		try {
			for (;;) {
				if (processComment(reader, writer)) {
					// 頭からやり直し。
					continue;
				}

				final StringBuffer strbufKey = new StringBuffer();
				if (processKey(strbufKey, reader, writer)) {
					// 頭からやり直し。
					continue;
				}

				// ここからは右辺。

				final StringBuffer strbufValue = new StringBuffer();
				final StringBuffer strbufOriginalValueLine = new StringBuffer();
				processValue(strbufKey.toString(), strbufValue, strbufOriginalValueLine, reader, writer);

				// 取得されたキーについて、実際にプロパティファイルから読み込みできることをチェックします。

				// keyに \ が入ることを考慮して文字列を取得します。
				final String strKey = getNormalizedKeyString(strbufKey.toString());
				if (strKey.trim().length() == 0) {
					// このファイルは処理できないのでスキップ扱いします。
					return false;
				}

				final String inputValue = (String) props.get(strKey);
				if (inputValue == null) {
					System.out.println("Fail to read key [" + strKey + "] from properties file."); //$NON-NLS-1$ //$NON-NLS-2$
					final Iterator<Entry<Object, Object>> ite = props.entrySet().iterator();
					for (; ite.hasNext();) {
						final Object obj = ite.next();
						System.out.println("  " + obj.toString()); //$NON-NLS-1$
					}
					return false;
				}

				if (replaceValue(props, strKey, writer)) {
					isProcessed = true;
				} else {
					writer.write(strbufOriginalValueLine.toString());
				}
			}
		} catch (EOFException ex) {
			// ファイルが終端に到達しました。
		} finally {
			reader.close();
		}

		return isProcessed;
	}

	/**
	 * コメントの処理。
	 * 
	 * @param reader 入力リーダー。
	 * @param writer 出力ライター。
	 * @return 処理中断すべき場合は true。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	public static boolean processComment(final Reader reader, final Writer writer) throws IOException {
		for (;;) {
			// 該当行がコメント行かどうかをチェックします。
			reader.mark(1);
			final int iReadCheck = reader.read();
			if (iReadCheck < 0) {
				// チェックの最中にファイルの終端に到達しました。
				throw new EOFException();
			}

			final char cReadCheck = (char) iReadCheck;
			if (cReadCheck == '#' || cReadCheck == '!') {
				// 該当行はコメント行でした。行末までコピーします。
				reader.reset();
				copyRemainCurrentLine(reader, writer);
				return true;
			} else if (cReadCheck == ' ') {
				writer.write(cReadCheck);
			} else {
				// 最後に読み込んだ文字は巻き戻しします。
				reader.reset();
				return false;
			}
		}
	}

	/**
	 * キーの処理。
	 * 
	 * @param strbufKey 読み取ったキー。
	 * @param reader 入力リーダー。
	 * @param writer 出力ライター。
	 * @return 処理中断すべき場合は true。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	public static boolean processKey(final StringBuffer strbufKey, final Reader reader, final Writer writer)
			throws IOException {
		for (;;) {
			reader.mark(1);
			final int iRead1 = reader.read();
			if (iRead1 < 0) {
				// チェックの最中にファイルの終端に到達しました。
				throw new EOFException();
			}

			final char cRead1 = (char) iRead1;

			if (cRead1 == '\\') {
				// エスケープ開始。
				writer.write(cRead1);
				reader.mark(1);
				final int iRead2 = reader.read();
				if (iRead2 < 0) {
					// エスケープ中なのに行末に来てしまいました。
					// 仕方がないので、なにもなかったことにします。

					// そもそも、キー読み込み最中に行末に到達するのは困ります。
					// そのようなプロパティファイルは処理すべきはでないものと判断し、処理中断します。
					// チェックの最中にファイルの終端に到達しました。
					throw new EOFException();
				}

				final char cRead2 = (char) iRead2;
				switch (cRead2) {
				case '\n':
				case '\r':
					// 改行が現れました。これはキーではありませんでした。
					reader.reset();
					copyRemainCurrentLine(reader, writer);
					return true;
				default:
					writer.write(cRead2);
					strbufKey.append(cRead1);
					strbufKey.append(cRead2);
					// キー途中のスペース表現の場合はここに含まれる。
					// 意味としての改行やユニコード表現もこれに含まれる。
					break;
				}
			} else if (cRead1 == '\n' || cRead1 == '\r') {
				// 改行が現れました。これはキーではありませんでした。
				reader.reset();
				copyRemainCurrentLine(reader, writer);
				return true;
			} else if (cRead1 == '=' || cRead1 == ':') {
				// 左辺が終了しました。
				writer.write(cRead1);
				return false;
			} else {
				// エスケープではありません。そのまま書き込みます。
				writer.write(cRead1);
				strbufKey.append(cRead1);
			}
		}
	}

	/**
	 * 値の処理。
	 * 
	 * @param strKey キー値。
	 * @param strbufValue 読み込んだ値。
	 * @param strbufOriginalValueLine 読み込んだもとの文字列。
	 * @param reader 入力リーダー。
	 * @param writer 出力リーダー。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	public static void processValue(final String strKey, final StringBuffer strbufValue,
			final StringBuffer strbufOriginalValueLine, final Reader reader, final Writer writer) throws IOException {
		boolean isValueStarted = false;
		for (;;) {
			reader.mark(1);
			final int iRead = reader.read();
			if (iRead < 0) {
				reader.reset();
				// ファイルの終端です。
				return;
			}

			final char cRead = (char) iRead;
			if (cRead == '\n' || cRead == '\r') {
				reader.reset();
				// 通常の行末です。
				return;
			}

			if (isValueStarted == false && cRead == ' ') {
				// 値の開始していない状態による空白は無視します。
				writer.write(cRead);
				continue;
			}

			strbufOriginalValueLine.append(cRead);

			// ここまでくれば、値は開始しています。
			isValueStarted = true;

			if (cRead == '\\') {
				// エスケープ開始。
				for (;;) {
					reader.mark(1);
					final int iRead2 = reader.read();
					if (iRead2 < 0) {
						throw new EOFException();
					}
					final char cRead2 = (char) iRead2;
					strbufOriginalValueLine.append(cRead2);
					if (cRead2 == '\n' || cRead2 == '\r') {
						// エスケープ中の改行です。これは特別に無視します。
					} else {
						strbufValue.append(cRead2);
						break;
					}
				}
			} else {
				strbufValue.append(cRead);
			}
		}
	}

	/**
	 * 与えられたリーダーについて、現在行の残りをコピーします。
	 * 
	 * @param reader 入力リーダー。
	 * @param writer 出力ライター。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	public static void copyRemainCurrentLine(final Reader reader, final Writer writer) throws IOException {
		boolean isLineBreaked = false;
		for (;;) {
			reader.mark(1);
			final int iRead = reader.read();
			if (iRead < 0) {
				reader.reset();
				break;
			}
			final char cRead = (char) iRead;
			if (cRead == '\n' || cRead == '\r') {
				isLineBreaked = true;
			} else {
				if (isLineBreaked) {
					reader.reset();
					break;
				}
			}
			writer.write(cRead);
		}
	}

	/**
	 * エスケープ文字が含まれたキー文字列を、エスケープが除去されたキー文字列に正常化します。
	 * 
	 * @param keyStringWithEscape
	 *            エスケープが含まれたキー文字列。
	 * @return エスケープが除去されたキー文字列。
	 */
	public static String getNormalizedKeyString(final String keyStringWithEscape) {
		final Properties prop = new Properties();
		try {
			prop.load(new ByteArrayInputStream((keyStringWithEscape + "=dummy").getBytes("UTF-8"))); //$NON-NLS-1$ //$NON-NLS-2$
		} catch (UnsupportedEncodingException e) {
			throw new IllegalArgumentException("An unexpected exception has occurred.", e); //$NON-NLS-1$
		} catch (IOException e) {
			throw new IllegalArgumentException("An unexpected exception has occurred.", e); //$NON-NLS-1$
		}

		final Iterator<Object> ite = prop.keySet().iterator();
		if (ite.hasNext() == false) {
			throw new IllegalArgumentException("An unexpected exception has occurred."); //$NON-NLS-1$
		}
		return (String) ite.next();
	}

	/**
	 * プロパティの値を処理します。
	 * @param props 入力プロパティ。
	 * @param key キー。
	 * @param writer ライター。
	 * @return 値の箇所をライターに出力処理したかどうか。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	abstract boolean replaceValue(final Properties props, final String key, final Writer writer) throws IOException;

	/**
	 * エスケープされたプロパティファイル上の値の文字列を取得します。
	 * 
	 * @param value
	 *            値。
	 * @return プロパティファイル上におけるエスケープされた値。
	 */
	public static String getEscapedPropertiesValue(final String value) {
		String result = null;
		final Properties prop = new Properties();
		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		try {
			prop.setProperty("key", value); //$NON-NLS-1$
			prop.store(outStream, ""); //$NON-NLS-1$
			outStream.flush();
			outStream.close();

			final byte[] buf = outStream.toByteArray();
			final BufferedReader reader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(buf),
					"UTF-8")); //$NON-NLS-1$
			for (;;) {
				final String line = reader.readLine();
				if (line == null) {
					break;
				}
				if (line.startsWith("#")) { //$NON-NLS-1$
					continue;
				}
				if (line.trim().length() == 0) {
					continue;
				}
				if (line.startsWith("key=") == false) { //$NON-NLS-1$
					continue;
				}
				result = line;
			}

			reader.close();

			if (result == null) {
				throw new IllegalArgumentException("An unexpected exception has occurred."); //$NON-NLS-1$
			}
			return result.substring("key=".length()); //$NON-NLS-1$
		} catch (IOException e) {
			throw new IllegalArgumentException("An unexpected exception has occurred. ", e); //$NON-NLS-1$
		}
	}
}
