﻿using System;
using System.Collections.Generic;
using System.Reflection;

namespace NaGet.Packages.Install
{
	/// <summary>
	/// 依存関係を解決したりするための便利メソッド群
	/// </summary>
	public sealed class DependeciesResolver
	{
		private DependeciesResolver()
		{
		}
		
		/// <summary>
		/// 依存解決済みInstallation配列を生成する
		/// </summary>
		/// <param name="insts">元のインストールリスト</param>
		/// <param name="pkgListsMan">インストール確認などをおこなうパッケージリストのマネージャ</param>
		/// <param name="resolved">解決済みInstallation配列</param>
		/// <param name="dependencies">依存によって必要とされたInstallation配列。ない場合は空配列</param>
		public static void ResolveInstallations(Installation[] insts, PackageListsManager pkgListsMan, out Installation[] resolved, out Installation[] dependencies)
		{
			List<Installation> depInsts;
			
			// 依存関係のインストールリスト
			depInsts = CreateRequiresInstallations(insts, pkgListsMan, null);
			dependencies = depInsts.ToArray();
			
			// 依存解決をしたインストールリスト
			depInsts.AddRange(insts);
			resolved = depInsts.ToArray();
		}
		
		/// <summary>
		/// Requires依存による依存パッケージの検索をし、それを解決するためのインストールリストを返す
		/// </summary>
		/// <param name="insts">対象インストールリスト</param>
		/// <param name="pkgListsMan">現在インストールされているか否かの判断に使われるパッケージリスト</param>
		/// <param name="preferencial">優先して使用をするパッケージのリスト。<c>null</c>でもよい</param>
		/// <returns>Requires依存による依存</returns>
		public static List<Installation> CreateRequiresInstallations(Installation[] insts, PackageListsManager pkgListsMan, ICollection<Package> preferencial)
		{
			List<Installation> reqInsts = new List<Installation>();
			
			foreach (Entry entry in DeleteDuplicatedEntries(CreateRequiresEntries(insts))) {
				List<InstalledPackage> instPkgs = NaGet.Utils.MeargeList<InstalledPackage>(
					pkgListsMan.installedPkgList.GetPackagesForEntry(entry),
					pkgListsMan.systemInstalledPkgList.GetPackagesForEntry(entry)
				);
				
				if (instPkgs.Count <= 0) {
					if (Array.Exists(insts, delegate(Installation inst) {
					                 	return entry.Match(inst.InstalledPackage);
					                 })) {
						continue;
					} else {
						VersionComparetor vc = new VersionComparetor();
						Package pkgToBeInstall = null;
						
						if (preferencial != null) {
							foreach (Package pkg in preferencial) {
								if (entry.Match(pkg)) {
									pkgToBeInstall = pkg;
									break;
								}
							}
						}
						if (pkgToBeInstall == null) {
							foreach (Package pkg in pkgListsMan.availablePkgList.GetPackagesForEntry(entry)) {
								if ((pkgToBeInstall == null) || (vc.Compare(pkgToBeInstall.Version, pkg.Version) < 0)) {
									pkgToBeInstall = pkg;
								}
							}
						}
						
						if (pkgToBeInstall != null) {
							reqInsts.Add(new Installation(pkgToBeInstall));
						}
					}
				}
			}
			
			return reqInsts;
		}
		
		/// <summary>
		/// 与えられたエントリのイテレータに対し、重複しているものを取り除いて返す
		/// </summary>
		/// <remarks>現段階の実装では、重複の判断は名前のみである</remarks>
		/// <param name="entries">エントリのイテレータ</param>
		/// <returns>重複しているものを取り除いたエントリのイテレータ</returns>
		private static IEnumerable<Entry> DeleteDuplicatedEntries(IEnumerable<Entry> entries)
		{
			LinkedList<string> pkgNames = new LinkedList<string>();
			
			foreach (Entry entry in entries) {
				if (! pkgNames.Contains(entry.Name)) {
					// 新出ならば返す
					yield return entry;
					pkgNames.AddFirst(entry.Name);
				}
			}
		}
		
		/// <summary>
		/// Requires依存エントリのイテレータを返す
		/// </summary>
		/// <remarks>現段階の実装では再帰的な依存は検索しない</remarks>
		/// <param name="insts">対象のインストールリスト</param>
		/// <returns>依存エントリのイテレータ。重複に関して解決は行わない</returns>
		private static IEnumerable<Entry> CreateRequiresEntries(Installation[] insts)
		{
			foreach (Installation inst in insts) {
				Entry[] deps = inst.InstalledPackage.Requires;
				if (deps == null) continue;
				foreach (Entry dep in deps) {
					yield return dep;
				}
			}
		}
		
		
	}
}
