﻿using System;
using System.Net;
using System.IO;
using System.Collections;
using System.Threading;
using NaGet.SubCommands;

namespace NaGet.Net
{

/// <summary>
/// ダウンロードイベントオブジェクト
/// </summary>
public class DownloadEventArgs : NaGetEventArgs
{
	/// <summary>
	/// イベントの種類
	/// </summary>
	public DownloadEventType Type;

	/// <summary>
	/// ダウンロード済みのバイト数
	/// </summary>
	public long DownloadSize;

	/// <summary>
	/// ダウンロードする総バイト数
	/// </summary>
	public long MaxSize;
	
	/// <summary>
	/// コンストラクタ
	/// </summary>
	/// <param name="type">イベントの種類</param>
	/// <param name="msg">イベントメッセージ</param>
	/// <param name="pos">ダウンロード済みのバイト数</param>
	/// <param name="max">ダウンロードする総バイト数</param>
	public DownloadEventArgs(DownloadEventType type, string msg, long pos, long max)
	{
		Type = type;
		DownloadSize = pos;
		MaxSize = max;
		
		TaskMessage = msg;
		TaskProgressPercent = (max > 0)? (100.0f * pos / max) : -1;
	}
}

/// <summary>
/// DownloadEventArgsでのイベントの種類
/// </summary>
public enum DownloadEventType {
	INITED,
	CONNECTED,
	STARTED,
	DOWNLOADING,
	ERROR,
	COMPLETED
}

/// <summary>
/// ダウンロード処理を行うクラス
/// </summary>
public class Downloader : NaGetTask
{
	/// <summary>
	/// デフォルトで使うプロキシ
	/// </summary>
	public static IWebProxy DefaultProxy = WebRequest.GetSystemWebProxy();
	
	/// <summary>
	/// 通信に使うプロキシ
	/// </summary>
	public IWebProxy Proxy;
	
	/// <summary>
	/// イベントハンドラ
	/// </summary>
	public event EventHandler<DownloadEventArgs> DownloadEventRaised;
	
	/// <summary>
	/// アクセスURL
	/// </summary>
	protected string url;
	
	/// <summary>
	/// 保存先
	/// </summary>
	protected string filepath;
	
	/// <summary>
	/// リクエストオブジェクト
	/// </summary>
	protected WebRequest request;
	
	/// <summary>
	/// レスポンスオブジェクト。応答がくるまではnullである。
	/// </summary>
	protected WebResponse response;
	
	/// <summary>
	/// ダウンロード要求時のキャッシュレベル。デフォルトではキャッシュ無視
	/// </summary>
	public System.Net.Cache.RequestCacheLevel CacheLevel = System.Net.Cache.RequestCacheLevel.NoCacheNoStore;
	
	private bool cancelCalled = false;
	
	private bool done = false;
	
	/// <summary>
	/// ダウンロード時にHTTPヘッダなどから取得した本来のファイル名
	/// </summary>
	private string downloadedFileName = null;
	
	/// <summary>
	/// ダウンロード時にHTTPヘッダなどから取得した本来のファイル名
	/// </summary>
	public string DownloadedFileName {
		get { return downloadedFileName; }
	}
	
	/* ダウンロードの処理時間計測器 */
	private System.Diagnostics.Stopwatch stopwatch;

	/// <summary>
	/// ダウンロードを行う
	/// </summary>
	/// <param name="url">ダウンロードするリソースのURL</param>
	/// <param name="filepath">保存先ファイルパス</param>
	public void Download(string url, string filepath)
	{
		this.url = url;
		this.filepath = filepath;
		
		try {
			Run();
		} catch (NotSupportedException e) {
			throw new NotSupportedException("Not supported download location for downloader.", e);
		}
	}
	
	public override bool Done {
		get { return done; }
	}
	
	public override bool Running {
		get { return request != null && (!done); }
	}

	public override void Run()
	{
		RaiseDownloadEvent(DownloadEventType.INITED, 0, -1);
		
		try {
			request = WebRequest.Create(url);
			request.Proxy = (Proxy == null)? DefaultProxy : Proxy;
			request.CachePolicy = new System.Net.Cache.RequestCachePolicy(CacheLevel);
			
			if (cancelCalled) {
				throw new NaGetTaskCanceledException(string.Empty);
			}
			
			try {
				response = request.GetResponse();
			} catch (WebException e) {
				if (cancelCalled) { // キャンセル時
					throw new NaGetTaskCanceledException(string.Empty, e);
				} else {
					throw new WebException(e.Message, e);
				}
			}
			
			if (cancelCalled) {
				throw new NaGetTaskCanceledException(string.Empty);
			}
			
			try {
				downloadedFileName = getFileNameFromWebResponse(response);
			} catch (Exception) {
			}
			
			if (File.Exists(filepath)) { // ファイルが存在するとき削除
				File.Delete(filepath);
			}
			
			RaiseDownloadEvent(DownloadEventType.CONNECTED, 0, -1);
			
			using (Stream stream = response.GetResponseStream() )
			using (FileStream fs = new FileStream(filepath,
				                    FileMode.Create,
				                    FileAccess.Write) ) {
				try {
					File.SetAttributes(filepath, FileAttributes.Hidden);
					long contentLength = response.ContentLength;
	
					RaiseDownloadEvent(DownloadEventType.STARTED, 0, contentLength);
					
					stopwatch = new System.Diagnostics.Stopwatch();
					stopwatch.Start();
	
					Timer timer = new Timer(new TimerCallback(
						delegate(object obj) {
							try {
								RaiseDownloadEvent(DownloadEventType.DOWNLOADING, fs.Position, contentLength);
							} catch (ObjectDisposedException) {
							}
						}), null, 0, 1000);
					
					try {
						byte[] data = new byte[4096];
						int size = 0;
						while ((size = stream.Read(data,0,data.Length)) > 0) {
							fs.Write(data, 0, size);
							
							if (cancelCalled) {
								throw new NaGetTaskCanceledException(string.Empty);
							}
						}
					} finally {
						timer.Dispose();
					}
					
					File.SetAttributes(filepath, FileAttributes.Normal);
					
					RaiseDownloadEvent(DownloadEventType.COMPLETED, fs.Position, contentLength);
				} catch (IOException ex) {
					if (cancelCalled) {
						throw new NaGetTaskCanceledException(string.Empty);
					} else {
						RaiseDownloadEvent(DownloadEventType.ERROR, 0, 0);
						throw new IOException(ex.Message, ex);
					}
				} finally {
					if (stopwatch != null) {
						stopwatch.Stop();
						stopwatch = null;
					}
				}
			}
			
			// 更新日を補完
			if (File.Exists(filepath)) {
				if (response is HttpWebResponse) {
					File.SetLastWriteTime(filepath, ((HttpWebResponse) response).LastModified);
				} else if (response is FtpWebResponse) {
					File.SetLastWriteTime(filepath, ((FtpWebResponse) response).LastModified);
				}
			}
		} finally {
			if (response != null) {
				response.Close();
			}
			
			request = null;
			done = true;
		}
	}
	
	protected void RaiseDownloadEvent(DownloadEventType type, long pos, long max)
	{
		if (DownloadEventRaised != null) {
			DownloadEventArgs e = new DownloadEventArgs(type, string.Empty, pos, max);
			
			switch (e.Type) {
				case DownloadEventType.CONNECTED:
					e.TaskMessage = "接続しました";
					break;
				case DownloadEventType.STARTED:
				case DownloadEventType.DOWNLOADING:
				case DownloadEventType.COMPLETED:
					if (e.TaskProgressPercent >= 0) {
						e.TaskMessage = string.Format("{0} bytes ({1} %)", e.DownloadSize, (int) e.TaskProgressPercent);
					} else {
						e.TaskMessage = string.Format("{0} bytes", e.DownloadSize);
					}
					
					
					if (stopwatch != null && stopwatch.IsRunning && stopwatch.ElapsedMilliseconds > 3000) {
						long bpers = e.DownloadSize * 1000 / stopwatch.ElapsedMilliseconds;
						if ((e.TaskProgressPercent >= 0) && (bpers > 0)) {
							TimeSpan rest = TimeSpan.FromSeconds((max - e.DownloadSize) / bpers);
							e.TaskMessage += string.Format(" 推定残り時間:{0} ({1}/s)", rest, NaGet.Utils.FormatSize(bpers));
						} else {
							e.TaskMessage += string.Format(" ({0}/s)", NaGet.Utils.FormatSize(bpers));
						}
					}
					
					break;
			}
			
			DownloadEventRaised(this, e);
		}
	}
	
	public override bool Cancelable {
		get { return !(this.Done || cancelCalled); }
	}
	
	public override bool Cancel()
	{
		if (this.Done || cancelCalled) {
			return false;
		}
		
		cancelCalled = true;
		if (request != null) {
			try {
				request.Abort();
			} catch (WebException) {
			}
		}
		return true;
	}
	
	/// <summary>
	/// Webレスポンスからダウンロードしたファイルの名前を取得
	/// </summary>
	/// <param name="response"></param>
	/// <returns></returns>
	private string getFileNameFromWebResponse(WebResponse response)
	{
		if (response is HttpWebResponse) {
			string contentDisposition = ((HttpWebResponse) response).Headers["Content-Disposition"];
			
			// TODO check license for http://www.atmarkit.co.jp/fdotnet/dotnettips/618downnoname/downnoname.html
			if (! string.IsNullOrEmpty(contentDisposition)) {
				System.Text.RegularExpressions.Regex re = new System.Text.RegularExpressions.Regex(
					@"filename\s*=\s*(?:""(?<filename>[^""]*)""|(?<filename>[^;]*))",
					System.Text.RegularExpressions.RegexOptions.IgnoreCase);
				
				System.Text.RegularExpressions.Match m = re.Match(contentDisposition);
				if (m.Success) {
					return m.Groups["filename"].Value;
				}
			}
		}
		
		return NaGet.Utils.Url2filename(response.ResponseUri.ToString());
	}
}

}


