// DocumentView.h
// (c) 2003 exeal

#ifndef _DOCUMENT_VIEW_H_
#define _DOCUMENT_VIEW_H_
#include "Window.h"
#include <list>


namespace Manah {
namespace Windows {

// CDocument class definition
/////////////////////////////////////////////////////////////////////////////

class CView;
class CController;

typedef std::list<CView*>::const_iterator	VIEW_POSITION;

class CDocument : public CObject {
	// RXgN^
public:
	CDocument(CController* pController = 0);
	virtual ~CDocument();	// destructor destories all containing views
private:
	CDocument(const CDocument& rhs);

	// Zq
private:
	operator =(const CDocument& rhs);

	// \bh
public:
	CController*	GetController() const;
	// r[
	void					AddView(CView* pWnd);
	virtual VIEW_POSITION	GetFirstViewPosition() const;
	virtual CView*			GetNextView(VIEW_POSITION& vp) const;
	unsigned long			GetViewCount() const;
	void					RemoveView(CView* pWnd);
	void					SetupAllViews(LPARAM lParam = 0L);
	void					UpdateAllViews(CView* pSender, LPARAM lParam = 0L, void* pHint = 0);
	// hLg
	tstring			GetExtensionName() const;
	const tstring&	GetPathName() const;
	const tstring&	GetTitle() const;
	virtual bool	IsModified() const;
	virtual void	SetModified(bool bModified = true);
	void			SetPathName(const tstring& strPathName);
	void			SetTitle(const tstring& strTitle);

protected:
	virtual void	OnChangedViewList();
	virtual void	OnCloseDocument();

	// f[^o
protected:
	CController*		m_pController;
	tstring				m_strTitle;
	tstring				m_strPathName;
	bool				m_bModified;
	std::list<CView*>	m_listViews;
};


// CView class definition
/////////////////////////////////////////////////////////////////////////////

class CView : public CWindow {
	friend class CDocument;

	// RXgN^
public:
	CView();
	CView(const CView& rhs);	// creates new view belongs to no documents
	virtual ~CView();

	// \bh
public:
	virtual bool	Create(HWND hwndParent, const RECT* pRect,	// always fail (return false)
						DWORD dwStyle, DWORD dwExStyle) throw(runtime_error);
	CDocument*		GetDocument() const;
protected:
	virtual void	OnDocumentSetup(LPARAM lHint = 0);
	virtual void	OnUpdate(CView* pSender, LPARAM lHint, void* pHint);

	// f[^o
protected:
	CDocument*	m_pDocument;
};


// CController class definition
/////////////////////////////////////////////////////////////////////////////

class CController : public CWindow {
	// ^
public:
	typedef CDocument	DocumentClass;
	typedef CView		ViewClass;

	// RXgN^
public:
	CController();
	~CController();
private:
	CController(const CController& rhs);	// never been invoked
	operator =(const CController& rhs);		// never been invoked

	// \bh
public:
	virtual bool	Create(HWND hwndParent, const RECT* lpRect,
						DWORD dwStyle, DWORD dwExStyle, DWORD dwViewStyle, DWORD dwViewExStyle);
	CDocument*		GetDocument() const;
	int				GetPadding() const;
	void			SetPadding(int nPadding);
protected:
	virtual void	_AdjustView();
	template<class Controller>
	bool			_Create(HWND hwndParent, const RECT* lpRect,
						DWORD dwStyle, DWORD dwExStyle,
						DWORD dwViewStyle, DWORD dwViewExStyle);

	// bZ[Wnh
protected:
	virtual void OnSize(UINT nType, int cx, int cy);

protected:
	static LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);

	// f[^o
protected:
	CDocument*	m_pDocument;
	CView*		m_pView;
private:
	int			m_nPadding;
};


// CDocument class implementation
/////////////////////////////////////////////////////////////////////////////

inline CDocument::CDocument(CController* pController /* = 0 */)
		: m_pController(pController), m_strTitle(L""), m_strPathName(L""), m_bModified(false) {
}

inline CDocument::~CDocument() {
	for(list<CView*>::iterator it = m_listViews.begin(); it != m_listViews.end(); ++it) {
		if((*it)->IsWindow())
			(*it)->DestroyWindow();
		delete *it;
	}
	m_listViews.erase(m_listViews.begin(), m_listViews.end());
}

inline void CDocument::AddView(CView* pView) {
	AssertValid();
	assert(pView != 0);
	pView->m_pDocument = this;
	m_listViews.push_back(const_cast<CView*>(pView));
}

inline CController* CDocument::GetController() const {
	AssertValid();
	return m_pController;
}

inline tstring CDocument::GetExtensionName() const {
	AssertValid();

	tstring::size_type	i = m_strTitle.rfind(L".");
	if(i == tstring::npos)
		return L"";
	return m_strTitle.substr(i + 1);
}

inline VIEW_POSITION CDocument::GetFirstViewPosition() const {
	AssertValid();
	return m_listViews.begin();
}

inline CView* CDocument::GetNextView(VIEW_POSITION& vp) const {
	AssertValid();
	if(vp == m_listViews.end())
		return 0;

	CView*	pView = *vp;
	++vp;
	return pView;
}

inline const tstring& CDocument::GetPathName() const {
	AssertValid();
	return m_strPathName;
}

inline const tstring& CDocument::GetTitle() const {
	AssertValid();
	return m_strTitle;
}

inline unsigned long CDocument::GetViewCount() const {
	AssertValid();
	return m_listViews.size();
}

inline bool CDocument::IsModified() const {
	AssertValid();
	return m_bModified;
}

inline void CDocument::RemoveView(CView* pView) {
	AssertValid();
	pView->m_pDocument = 0;
	m_listViews.remove(pView);
}

inline void CDocument::SetModified(bool bModified /* = true */) {
	AssertValid();
	m_bModified = bModified;
}

inline void CDocument::SetPathName(const tstring& strPathName) {
	AssertValid();
	m_strPathName = strPathName;
}

inline void CDocument::SetTitle(const tstring& strTitle) {
	AssertValid();
	m_strTitle = const_cast<tstring&>(strTitle);
}

inline void CDocument::SetupAllViews(LPARAM lParam /* = 0L */) {
	AssertValid();

	std::list<CView*>::iterator	it = m_listViews.begin();
	while(it != m_listViews.end()) {
		(*it)->OnDocumentSetup();
		++it;
	}
}

inline void CDocument::UpdateAllViews(CView* pSender, LPARAM lParam /* = 0L */, void* pHint /* = 0 */) {
	AssertValid();
	
	std::list<CView*>::iterator	it = m_listViews.begin();
	while(it != m_listViews.end()) {
		if(*it != pSender)
			(*it)->OnUpdate(pSender, lParam, pHint);
		++it;
	}
}

inline void CDocument::OnChangedViewList() {
	if(m_listViews.empty())
		OnCloseDocument();
}

inline void CDocument::OnCloseDocument() {
}


// CView class implementation
/////////////////////////////////////////////////////////////////////////////

inline CView::CView() : m_pDocument(0) {
}

inline CView::CView(const CView& rhs) : CWindow(rhs), m_pDocument(0) {
}

inline CView::~CView() {
}

inline bool CView::Create(HWND hwndParent, const RECT* pRect,
		DWORD dwStyle, DWORD dwExStyle) throw(runtime_error) {
	return false;
}

inline CDocument* CView::GetDocument() const {
	AssertValid();
	return m_pDocument;
}

inline void CView::OnDocumentSetup(LPARAM lHint /* = 0 */) {
}

inline void CView::OnUpdate(CView* pSender, LPARAM lHint, void* pHint) {
	InvalidateRect(0);
}


// CController class implementation
/////////////////////////////////////////////////////////////////////////////

inline CController::CController() : m_pDocument(0), m_pView(0), m_nPadding(0) {
}

inline CController::~CController() {
	delete m_pDocument;
}

inline void CController::_AdjustView() {
	AssertValidAsWindow();
	if(m_pView == 0 || !m_pView->IsWindow())
		return;

	RECT	rect;

	GetClientRect(&rect);
	::InflateRect(&rect, -m_nPadding, -m_nPadding);
	m_pView->MoveWindow(&rect);
}

template<class Controller>
inline bool CController::_Create(HWND hwndParent, const RECT* lpRect,
		DWORD dwStyle, DWORD dwExStyle, DWORD dwViewStyle, DWORD dwViewExStyle) {
	AssertValid();
	assert(::IsWindow(hwndParent));
	assert(lpRect != 0 && m_pDocument == 0);

	if(IsWindow())
		return false;

	HINSTANCE	hInstance = reinterpret_cast<HINSTANCE>(::GetWindowLong(hwndParent, GWL_HINSTANCE));
	WNDCLASSEX	wc;
	wc.cbSize = sizeof(WNDCLASSEX);
	if(!::GetClassInfoEx(hInstance, _T("ControllerPane"), &wc)) {
		wc.style			= CS_BYTEALIGNCLIENT | CS_BYTEALIGNWINDOW | CS_DBLCLKS;
		wc.lpfnWndProc		= reinterpret_cast<WNDPROC>(CController::WndProc);
		wc.cbClsExtra		= 0;
		wc.cbWndExtra		= sizeof(long);
		wc.hInstance		= ::GetModuleHandle(0);
		wc.hIcon			= 0;
		wc.hIconSm			= 0;
		wc.hCursor			= ::LoadCursor(0, IDC_ARROW);
		wc.hbrBackground	= ::GetSysColorBrush(COLOR_3DFACE);
		wc.lpszClassName	= _T("ControllerPane");
		wc.lpszMenuName		= 0;
		if(!::RegisterClassEx(&wc))
			return false;
	}

	m_hWnd = ::CreateWindowEx(dwExStyle, _T("ControllerPane"), _T(""), dwStyle,
			lpRect->left, lpRect->top, lpRect->right - lpRect->left, lpRect->bottom - lpRect->top,
			hwndParent, 0, hInstance, reinterpret_cast<void*>(this));
	if(m_hWnd == 0)
		return false;

	m_pDocument = new Controller::DocumentClass(reinterpret_cast<Controller*>(this));
	m_pView = new Controller::ViewClass();
	m_pDocument->AddView(m_pView);
	if(m_pView->Create(m_hWnd, lpRect, dwViewStyle, dwViewExStyle))
		return true;
	else {
		delete m_pDocument;
		return false;
	}
}

inline bool CController::Create(HWND hwndParent, const RECT* lpRect,
		DWORD dwStyle, DWORD dwExStyle, DWORD dwViewStyle, DWORD dwViewExStyle) {
	AssertValid();
	return _Create<CController>(hwndParent, lpRect,
			dwStyle, dwExStyle, dwViewStyle, dwViewExStyle);
}

inline CDocument* CController::GetDocument() const {
	AssertValid();
	return m_pDocument;
}

inline int CController::GetPadding() const {
	AssertValid();
	return m_nPadding;
}

inline void CController::SetPadding(int nPadding) {
	AssertValid();
	assert(nPadding >= 0);

	m_nPadding = nPadding;
	if(IsWindowVisible())
		_AdjustView();
}

inline void CController::OnSize(UINT nType, int cx, int cy) {
	_AdjustView();
}

inline LRESULT CALLBACK CController::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	CController*	pController;
	if(message == WM_CREATE) {
		pController = reinterpret_cast<CController*>(reinterpret_cast<CREATESTRUCTW*>(lParam)->lpCreateParams);
		pController->m_hWnd = hWnd;
		pController->SetWindowLong(
			GWL_USERDATA, reinterpret_cast<long>(reinterpret_cast<CREATESTRUCTW*>(lParam)->lpCreateParams));
		return pController->DispatchEvent(message, wParam, lParam);
	} else {
		pController = reinterpret_cast<CController*>(::GetWindowLong(hWnd, GWL_USERDATA));
		return (pController != 0 && pController->m_hWnd == hWnd) ?
			pController->DispatchEvent(message, wParam, lParam) : ::DefWindowProc(hWnd, message, wParam, lParam);
	}
}

} // namespace Windows
} // namespace Manah

#endif /* _DOCUMENT_VIEW_H_ */

/* [EOF] */