/****************************************************************************
**
** Copyright (C) 2016 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of Qt Creator.
**
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
****************************************************************************/

#include "rmcmakeoperation.h"

#include "addkeysoperation.h"
#include "addcmakeoperation.h"
#include "findkeyoperation.h"
#include "findvalueoperation.h"
#include "getoperation.h"
#include "rmkeysoperation.h"

#include <iostream>

// CMake file stuff:
const char COUNT[] = "CMakeTools.Count";
const char PREFIX[] = "CMakeTools.";

// CMake:
const char ID[] = "Id";

QString RmCMakeOperation::name() const
{
    return QString("rmCMake");
}

QString RmCMakeOperation::helpText() const
{
    return QString("remove a cmake tool");
}

QString RmCMakeOperation::argumentsHelpText() const
{
    return QString("    --id <ID>  The id of the cmake tool to remove.\n");
}

bool RmCMakeOperation::setArguments(const QStringList &args)
{
    for (int i = 0; i < args.count(); ++i) {
        const QString current = args.at(i);
        const QString next = ((i + 1) < args.count()) ? args.at(i + 1) : QString();

        if (current == "--id") {
            if (next.isNull()) {
                std::cerr << "No parameter for --id given." << std::endl << std::endl;
                return false;
            }
            ++i; // skip next;
            m_id = next;
            continue;
        }
    }

    if (m_id.isEmpty())
        std::cerr << "No id given." << std::endl << std::endl;

    return !m_id.isEmpty();
}

int RmCMakeOperation::execute() const
{
    QVariantMap map = load("cmaketools");
    if (map.isEmpty())
        return 0;

    QVariantMap result = RmCMakeData{m_id}.rmCMake(map);
    if (result == map)
        return 2;

    return save(result, "cmaketools") ? 0 : 3;
}

#ifdef WITH_TESTS
bool RmCMakeOperation::test() const
{
    // Add cmakes:
    QVariantMap map = AddCMakeOperation::initializeCMake();
    const QVariantMap emptyMap = map;
    map = AddCMakeData{"testId", "name", "/tmp/test",
                       {{"ExtraKey", QVariant("ExtraValue")}}}.addCMake(map);
    map = AddCMakeData{"testId2", "other name", "/tmp/test2", {}}.addCMake(map);

    QVariantMap result = RmCMakeData{"nonexistent"}.rmCMake(QVariantMap());
    if (!result.isEmpty())
        return false;

    result = RmCMakeData{"nonexistent"}.rmCMake(map);
    if (result != map)
        return false;

    // Remove from map with both testId and testId2:
    result = RmCMakeData{"testId2"}.rmCMake(map);
    if (result == map
            || result.value(COUNT, 0).toInt() != 1
            || !result.contains(QString::fromLatin1(PREFIX) + "0")
            || result.value(QString::fromLatin1(PREFIX) + "0") != map.value(QString::fromLatin1(PREFIX) + "0"))
        return false;

    // Remove from map with both testId and testId2:
    result = RmCMakeData{"testId"}.rmCMake(map);
    if (result == map
            || result.value(COUNT, 0).toInt() != 1
            || !result.contains(QString::fromLatin1(PREFIX) + "0")
            || result.value(QString::fromLatin1(PREFIX) + "0") != map.value(QString::fromLatin1(PREFIX) + "1"))
        return false;

    // Remove from map without testId!
    result = RmCMakeData{"testId2"}.rmCMake(result);
    if (result != emptyMap)
        return false;

    return true;
}
#endif

QVariantMap RmCMakeData::rmCMake(const QVariantMap &map) const
{
    // Find count of cmakes:
    bool ok;
    int count = GetOperation::get(map, COUNT).toInt(&ok);
    if (!ok || count < 0) {
        std::cerr << "Error: Count found in cmake tools file seems wrong." << std::endl;
        return map;
    }

    QVariantList cmList;
    for (int i = 0; i < count; ++i) {
        QVariantMap cmData = GetOperation::get(map, QString::fromLatin1(PREFIX) + QString::number(i)).toMap();
        if (cmData.value(ID).toString() != m_id)
            cmList.append(cmData);
    }

    QVariantMap newMap = AddCMakeOperation::initializeCMake();
    for (int i = 0; i < cmList.count(); ++i)
        newMap.insert(QString::fromLatin1(PREFIX) + QString::number(i), cmList.at(i));
    newMap.insert(COUNT, cmList.count());

    return newMap;
}

