/****************************************************************************
**
** Copyright (C) 2021 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the QtQml module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qqmlcppbinding_p.h"

#include <QtQml/qqmlengine.h>
#include <QtQml/qqmlcontext.h>
#include <QtCore/qmetaobject.h>

#include <private/qqmltypedata_p.h>
#include <private/qqmlpropertybinding_p.h>
#include <private/qqmlbinding_p.h>
#include <private/qv4qmlcontext_p.h>
#include <private/qqmlproperty_p.h>
#include <private/qqmlbinding_p.h>

QT_BEGIN_NAMESPACE

template<typename CreateBinding>
inline decltype(auto) createBindingInScope(QObject *thisObject, CreateBinding create)
{
    QQmlEngine *qmlengine = qmlEngine(thisObject);
    Q_ASSERT(qmlengine);
    QV4::ExecutionEngine *v4 = qmlengine->handle();
    Q_ASSERT(v4);

    QQmlData *ddata = QQmlData::get(thisObject);
    Q_ASSERT(ddata && ddata->outerContext);
    QQmlRefPointer<QQmlContextData> ctxtdata = QQmlRefPointer<QQmlContextData>(ddata->outerContext);

    QV4::Scope scope(v4);
    QV4::ExecutionContext *executionCtx = v4->scriptContext();
    QV4::Scoped<QV4::QmlContext> qmlContext(
            scope, QV4::QmlContext::create(executionCtx, ctxtdata, thisObject));

    return create(ctxtdata, qmlContext);
}

QUntypedPropertyBinding
QQmlCppBinding::createBindingForBindable(const QV4::ExecutableCompilationUnit *unit,
                                         QObject *thisObject, qsizetype functionIndex,
                                         QObject *bindingTarget, int metaPropertyIndex,
                                         int valueTypePropertyIndex, const QString &propertyName)
{
    Q_UNUSED(propertyName);

    QV4::Function *v4Function = unit->runtimeFunctions.value(functionIndex, nullptr);
    if (!v4Function) {
        // TODO: align with existing logging of such
        qCritical() << "invalid JavaScript function index (internal error)";
        return QUntypedPropertyBinding();
    }
    if (metaPropertyIndex < 0) {
        // TODO: align with existing logging of such
        qCritical() << "invalid meta property index (internal error)";
        return QUntypedPropertyBinding();
    }

    const QMetaObject *mo = bindingTarget->metaObject();
    Q_ASSERT(mo);
    QMetaProperty property = mo->property(metaPropertyIndex);
    Q_ASSERT(valueTypePropertyIndex == -1 || QString::fromUtf8(property.name()) == propertyName);

    return createBindingInScope(
            thisObject,
            [&](const QQmlRefPointer<QQmlContextData> &ctxt, QV4::ExecutionContext *scope) {
                auto index = QQmlPropertyIndex(property.propertyIndex(), valueTypePropertyIndex);
                return QQmlPropertyBinding::create(property.metaType(), v4Function, thisObject,
                                                   ctxt, scope, bindingTarget, index);
            });
}

void QQmlCppBinding::createBindingForNonBindable(const QV4::ExecutableCompilationUnit *unit,
                                                 QObject *thisObject, qsizetype functionIndex,
                                                 QObject *bindingTarget, int metaPropertyIndex,
                                                 int valueTypePropertyIndex,
                                                 const QString &propertyName)
{
    Q_UNUSED(propertyName);

    QV4::Function *v4Function = unit->runtimeFunctions.value(functionIndex, nullptr);
    if (!v4Function) {
        // TODO: align with existing logging of such
        qCritical() << "invalid JavaScript function index (internal error)";
        return;
    }
    if (metaPropertyIndex < 0) {
        // TODO: align with existing logging of such
        qCritical() << "invalid meta property index (internal error)";
        return;
    }

    const QMetaObject *mo = bindingTarget->metaObject();
    Q_ASSERT(mo);
    QMetaProperty property = mo->property(metaPropertyIndex);
    Q_ASSERT(valueTypePropertyIndex != -1 || QString::fromUtf8(property.name()) == propertyName);

    createBindingInScope(
            thisObject,
            [&](const QQmlRefPointer<QQmlContextData> &ctxt, QV4::ExecutionContext *scope) -> void {
                QQmlBinding *binding = QQmlBinding::create(property.metaType(), v4Function,
                                                           thisObject, ctxt, scope);
                // almost as in qv4objectwrapper.cpp:535
                Q_ASSERT(!property.isAlias()); // we convert aliases to (almost) real properties
                binding->setTarget(bindingTarget, property.propertyIndex(), false,
                                   valueTypePropertyIndex);
                QQmlPropertyPrivate::setBinding(binding);
            });
}

QT_END_NAMESPACE
