/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.ui.internal.binding;

import java.util.Collection;
import java.util.Iterator;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.provider.AdapterFactoryItemDelegator;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IItemPropertyDescriptor;
import org.eclipse.jsr220orm.core.IEntityModelManager;
import org.eclipse.jsr220orm.metadata.MetadataElement;
import org.eclipse.swt.widgets.Widget;

public abstract class EmfAbstractBinding implements Adapter, IEmfWidgetBinding {
    protected AdapterFactory adapterFactory;
    protected EObject notifier;
    protected int featureId;
    protected IItemPropertyDescriptor item;
    protected Collection choices;

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.jsr220orm.ui.internal.binding.IEmfWidgetBinding#bind(org.eclipse.emf.common.notify.AdapterFactory,
     *      org.eclipse.swt.widgets.Widget, int)
     */
    public void bind(AdapterFactory adapterFactory, Widget widget, int featureId) {
        this.adapterFactory = adapterFactory;
        this.featureId = featureId;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.jsr220orm.ui.internal.binding.IEmfWidgetBinding#setNotifier(org.eclipse.emf.common.notify.Notifier)
     */
    public void setEObject(EObject notifier) {
        uninstallListeners();
        try {
            this.notifier = notifier;
            setValuesFromModel();
        } finally {
            installListeners();
        }
    }

    protected void installListeners() {
        if (this.notifier != null) {
            this.notifier.eAdapters().add(this);
        }
    }

    protected void uninstallListeners() {
        if (this.notifier != null) {
            while(this.notifier.eAdapters().remove(this));
        }
    }

    public void notifyChanged(Notification notification) {
        if (notification.getEventType() == Notification.REMOVING_ADAPTER) {
            return;
        }
        uninstallListeners();
        try {
            setValuesFromModel();
        } finally {
            installListeners();
        }
    }

    private void setValuesFromModel() {
        AdapterFactoryItemDelegator contentProvider = new AdapterFactoryItemDelegator(
                adapterFactory);
        EStructuralFeature feature = (EStructuralFeature) notifier.eClass()
                .getEAllStructuralFeatures().get(featureId);
        for (Iterator i = contentProvider.getPropertyDescriptors(notifier)
                .iterator(); i.hasNext();) {
            IItemPropertyDescriptor itemPropertyDescriptor = (IItemPropertyDescriptor) i
                    .next();
            if (feature.equals(itemPropertyDescriptor.getFeature(notifier))) {
                item = itemPropertyDescriptor;
                break;
            }
        }
        if (item != null) {
            IItemLabelProvider labelProvider = item.getLabelProvider(notifier);
            Collection choices = getChoiceValuesFromModel(item, notifier);
            if (choices != null) {
                setChoicesFromModel(labelProvider, choices);
            }
            setValueFromModel(labelProvider, getValueFromModel(notifier, feature));
        }
    }

    protected Object getValueFromModel(EObject notifier, EStructuralFeature feature) {
        return notifier.eGet(feature);
    }
    
    protected Collection getChoiceValuesFromModel(IItemPropertyDescriptor item, EObject notifier) {
        return item.getChoiceOfValues(notifier);
    }

    protected Object getCurrentValueFromModel() {
        AdapterFactoryItemDelegator contentProvider = new AdapterFactoryItemDelegator(
                adapterFactory);
        EStructuralFeature feature = (EStructuralFeature) notifier.eClass()
                .getEAllStructuralFeatures().get(featureId);
        return notifier.eGet(feature);
    }

    /**
     * Update the model setting our target feature to value.
     * 
     * @see #setValuesFromWidget(Runnable)
     */
    protected void setValuesFromWidget(final Object value) {
        AdapterFactoryItemDelegator contentProvider = 
        	new AdapterFactoryItemDelegator(adapterFactory);
        final EStructuralFeature feature = (EStructuralFeature)notifier.eClass()
                .getEAllStructuralFeatures().get(featureId);
        setValuesFromWidget(new Runnable() {
        	public void run() {
        		notifier.eSet(feature, value);
        	}
        });
    }

    /**
     * Update the model by invoking r.run(). This is for cases when more 
     * than one change needs to be made to the model.
     * 
     * @see #setValuesFromWidget(Object)
     */
    protected void setValuesFromWidget(Runnable r) {
    	try {
        	uninstallListeners();
    		MetadataElement amd = (MetadataElement)getTarget();
    		IEntityModelManager mm = (IEntityModelManager)amd.adapt(
    				IEntityModelManager.class);
    		if (mm != null) {
    			mm.executeModelChanges(r);
    		} else {
    			r.run();
    		}
    	} finally {
    		installListeners();
    	}
    }
    
    protected abstract void setChoicesFromModel(IItemLabelProvider provider,
            Collection choices);

    protected abstract void setValueFromModel(IItemLabelProvider provider,
            Object value);

    public Notifier getTarget() {
        return notifier;
    }

    public void setTarget(Notifier newTarget) {
        // notifier = newTarget;
    }

    public boolean isAdapterForType(Object type) {
        return notifier != null && type instanceof Class
                && notifier.getClass().isAssignableFrom((Class) type);
    }
}
