/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io.ast;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.Annotation;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.internal.ui.javaeditor.EditorUtility;
import org.eclipse.jsr220orm.core.OrmPlugin;
import org.eclipse.jsr220orm.generic.io.AnnotationRegistry;
import org.eclipse.jsr220orm.generic.reflect.RClass;

/**
 * Creates RClass instances from Eclipse AST trees. This automatically looks
 * up resources in the project and caches and builds new trees as needed.
 */
public class AstRClassFactory {
	
	protected final IJavaProject javaProject;
	protected final ASTParser parser;
	protected final AnnotationRegistry annotationRegistry;
	
	protected Map <String, RClass> nameClassMap = new HashMap();
	protected AstState astState;
	
	protected static final java.lang.annotation.Annotation[] EMPTY_ANNOTATION_ARRAY = 
		new java.lang.annotation.Annotation[0];
	
	public AstRClassFactory(IJavaProject javaProject, 
			AnnotationRegistry annotationRegistry) {
		this.javaProject = javaProject;
		this.annotationRegistry = annotationRegistry;
		parser = ASTParser.newParser(AST.JLS3);
	}

	/**
	 * Get readly to lookup classes and so on. Call {@link #finish()} in a
	 * finally block when done.
	 */
	public void begin() {
		begin(null);
	}
	
	/**
	 * Get readly to lookup classes and so on using previously state 
	 * information captured from a previous AST tree. Call {@link #finish()}
	 * in a finally block when done.
	 */
	public void begin(AstState astState) {
		nameClassMap.clear();
		if (astState == null) {
			astState = new AstState();
		}
		this.astState = astState;
	}
	
	/**
	 * Release all resources and return state information for future use.
	 */
	public AstState finish() {
		nameClassMap.clear();
		AstState ans = astState;
		astState = null;
		return ans;
	}
	
	/**
	 * Find a class by name. 
	 */
	public RClass findClass(String fullClassName, boolean forModification) {
		RClass ans = (RClass)nameClassMap.get(fullClassName);
		if (ans == null) {
			try {
                if(!javaProject.isOpen()){
                    return null;
                }
				IType type = javaProject.findType(fullClassName);
				if (type == null) {
					return null;
				}
				ICompilationUnit cu = type.getCompilationUnit();
	            boolean wasWC = cu.isWorkingCopy();
	            if (forModification && !wasWC) {
//	            	IEditorInput ei = EditorUtility.getEditorInput(type);
//	            	IWorkingCopyManager wcm = JavaUI.getWorkingCopyManager();
//					wcm.connect(ei);
//	            	cu = wcm.getWorkingCopy(ei);
//	            	wcm.disconnect(ei);
	            	
	            	EditorUtility.openInEditor(type, false);
	            	
//	                IEditorPart javaEditor = JavaUI.openInEditor(type);
//	                cu = type.getCompilationUnit();
	                // UIHACK
//	                Control focusControl = 
//	                	PlatformUI.getWorkbench().getDisplay().getFocusControl();
//	                if (focusControl != null) {
//		                focusControl.setFocus();
//	                }
	            }
	            
	            if (cu == null) {	// binary type
					// TODO handle binary type in findClass
					return null;
				} else {
					parser.setSource(cu);
					// must call setResolveBindings AFTER setting source!
					parser.setResolveBindings(true);
					CompilationUnit root = (CompilationUnit)parser.createAST(null);
					TypeDeclaration td = findTypeDeclaration(root, 
							type.getElementName());
					ImportListHelper importListHelper = new ImportListHelper(root);
					ans = new AstRClass(this, td, fullClassName, cu, root,
							importListHelper);
				}
				nameClassMap.put(fullClassName, ans);
			} catch (Exception e) {
				OrmPlugin.log(e);
			}
		}
		return ans;
	}

	protected TypeDeclaration findTypeDeclaration(CompilationUnit root, 
			String className) {
		for (Iterator i = root.types().iterator(); i.hasNext(); ) {
			Object o = i.next();
			if (o instanceof TypeDeclaration) {
				TypeDeclaration td = (TypeDeclaration)o;
				if (td.getName().getFullyQualifiedName().equals(className)) {
					return td;
				}
			}
		}
		return null;
	}

	public AnnotationRegistry getAnnotationRegistry() {
		return annotationRegistry;
	}
	
	public AstState getAstState() {
		return astState;
	}

	/**
	 * Create actual java.lang.annotation.Annotation instances mapped to all
	 * AST Annotation instances returned by i.
	 */
	public Map<Class,java.lang.annotation.Annotation> createAnnotations(
			AstRAnnotatedElement owner, Iterator i, ImportListHelper importListHelper) {
		Map<Class,java.lang.annotation.Annotation> ans = new HashMap();
		for (; i.hasNext(); ) {
			Object o = i.next();
			if (o instanceof Annotation) {
				Annotation ann = (Annotation)o;
				Class annotationType = annotationRegistry.getClass(
						ann.getTypeName().getFullyQualifiedName());
				if (annotationType != null) {
					AstAnnotationProxyHandler h = 
						new AstAnnotationProxyHandler(owner,
								annotationType, importListHelper, ann, 
								(List)null);
					ans.put(annotationType, h.newProxy());
				}
			}
		}
		return ans;
	}

	public IJavaProject getJavaProject() {
		return javaProject;
	}
	
}
