package org.eclipse.jsr220orm.generic.io;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.persistence.AttributeOverride;
import javax.persistence.AttributeOverrides;

import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;

/**
 * This manages a collection of AttributeOverride annotations and provides
 * map like access.
 */
public class AttributeOverrideMap {

	protected final RAnnotatedElement element;
	protected final Map<String, AttributeOverride> map;
	protected AttributeOverrides aos;
	protected AttributeOverride ao;
	
	/**
	 * Create using annotations from element. If entityIO is not null then
	 * any problems with the annotations discovered during reading are added
	 * to it (e.g. having AttributeOverride and AttributeOverrides).
	 */
	public AttributeOverrideMap(RAnnotatedElement element, EntityIO entityIO) {
		this.element = element;
		map = new LinkedHashMap();
		aos = element.getAnnotation(AttributeOverrides.class);
		ao = element.getAnnotation(AttributeOverride.class);
		if (aos != null) {
			if (ao != null && entityIO != null) {
				entityIO.addProblem("Cannot use AttributeOverride and " +
						"AttributeOverrides together", ao);
				ao = null;
			}
			AttributeOverride[] value = aos.value();
			if (value != null) {
				for (AttributeOverride override : value) {
					String name = override.name();
					if (map.containsKey(name)) {
						if (entityIO != null) {
							entityIO.addProblem(
									"Duplicate AttributeOverride: " + name, 
									override);
						}
					} else {
						map.put(name, override);
					}
				}
			}
		} else if (ao != null) {
			map.put(ao.name(), ao);
		}
	}
	
	public AttributeOverride get(String name) {
		return map.get(name);
	}

	public AttributeOverride remove(String name) {
		return map.remove(name);
	}
	
	public Collection<AttributeOverride> values() {
		return map.values();
	}

	/**
	 * Get rid of AttributeOverride's that have no column and so on. This will 
	 * happen if all the column values are set to defaults. This might also 
	 * remove the AttributeOverrides itself if we are now empty or downgrade
	 * to a single AttributeOverride etc.
	 */
	public void cleanupAfterMetaDataUpdates() {
		
		// get rid of AttributeOverride's with no column
		for (Iterator i = map.values().iterator(); i.hasNext(); ) {
			AnnotationEx override = (AnnotationEx)i.next();
			if (!override.hasValue("column")) {
				override.delete();
				i.remove();
			}
		}
		if (aos != null && ((AnnotationEx)aos).getArraySize("value") == 0) {
			((AnnotationEx)aos).delete();
			aos = null;
		} else if (ao != null && !((AnnotationEx)ao).hasValue("column")) {
			((AnnotationEx)ao).delete();
			ao = null;
		}
		
		// now see how many AO's we have in total and consolidate
		if (aos == null) {
			return; // 0 or 1 (in a standalone AO) so nothing to do
		}
		if (ao != null) { 
			// add to AOS which already contains at least 1 AO
			copy((AnnotationEx)ao, ((AnnotationEx)aos).appendArrayElement("value"));
			((AnnotationEx)ao).delete();
			ao = null;
			return;
		}
		if (((AnnotationEx)aos).getArraySize("value") == 1) {
			// convert to a standalone AO
			ao = element.getAnnotation(AttributeOverride.class, true);
			copy((AnnotationEx)aos.value()[0], (AnnotationEx)ao);
			((AnnotationEx)aos).delete();
			aos = null;
		}
	}
	
	/**
	 * Copy the values from AttributeOverride src to dest including the
	 * column values. 
	 */
	protected void copy(AnnotationEx src, AnnotationEx dest) {
		dest.set("name", src.get("name"));
		AnnotationEx sc = (AnnotationEx)src.get("column");
		AnnotationEx dc = (AnnotationEx)dest.get("column");
		for (String name : sc.getValueNames()) {
			dc.set(name, sc.get(name));
		}
	}
	
	/**
	 * Add a new AttributeOverride to the end of our list. The annotation is
	 * only actually added if at least one of its values is set to something
	 * non-default.
	 */
	public AttributeOverride append() {
		if (aos == null) {
			aos = element.getAnnotation(AttributeOverrides.class, true);
		}
		return (AttributeOverride)((AnnotationEx)aos).appendArrayElement("value");
	}	
	
}
