# This file tests the tclFCmd.c file.
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1996-1997 Sun Microsystems, Inc.
# Copyright (c) 1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: fCmd.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $
#

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest 2
    namespace import -force ::tcltest::*
}

tcltest::testConstraint testsetplatform [string equal testsetplatform [info commands testsetplatform]]
tcltest::testConstraint testchmod [string equal testchmod [info commands testchmod]]

# Several tests require need to match results against the unix username
set user {}
if {$tcl_platform(platform) == "unix"} {
    catch {set user [exec whoami]}
    if {$user == ""} {
	catch {regexp {^[^(]*\(([^)]*)\)} [exec id] dummy user}
    }
    if {$user == ""} {
	set user "root"
    }
}

proc createfile {file {string a}} {
    set f [open $file w]
    puts -nonewline $f $string
    close $f
    return $string
}

# 
# checkcontent --
#
#  Ensures that file "file" contains only the string "matchString"
#  returns 0 if the file does not exist, or has a different content
#
proc checkcontent {file matchString} {
    if {[catch {
	set f [open $file]
	set fileString [read $f]
	close $f 
    }]} {
	return 0
    }
    return [string match $matchString $fileString]
}

proc openup {path} {
    testchmod 777 $path
    if {[file isdirectory $path]} {
	catch {
	    foreach p [glob -directory $path *] {
		openup $p
	    }
	}
    }
}

proc cleanup {args} {
    if {$::tcl_platform(platform) == "macintosh"} {
	set wd [list :]
    } else {
	set wd [list .]
    }
    foreach p [concat $wd $args] {
	set x ""
	catch {
	    set x [glob -directory $p tf* td*]
	}
	foreach file $x {
	    if {[catch {file delete -force -- $file}]} {
		catch {openup $file}
		catch {file delete -force -- $file}
	    }
	}
    }
}

proc contents {file} {
    set f [open $file r]
    set r [read $f]
    close $f
    set r
}

cd [temporaryDirectory]

set ::tcltest::testConstraints(fileSharing) 0
set ::tcltest::testConstraints(notFileSharing) 1

if {$tcl_platform(platform) == "macintosh"} {
    catch {file delete -force foo.dir}
    file mkdir foo.dir
    if {[catch {file attributes foo.dir -readonly 1}] == 0} {
    	set ::tcltest::testConstraints(fileSharing) 1
    	set ::tcltest::testConstraints(notFileSharing) 0
    }
    file delete -force foo.dir
}

set ::tcltest::testConstraints(xdev) 0

if {$tcl_platform(platform) == "unix"} {
    if {[catch {set m1 [exec df .]; set m2 [exec df /tmp]}] == 0} {
	set m1 [string range $m1 0 [expr [string first " " $m1]-1]]
	set m2 [string range $m2 0 [expr [string first " " $m2]-1]]
	if {$m1 != "" && $m2 != "" && $m1 != $m2 && [file exists $m1] && [file exists $m2]} {
	    set ::tcltest::testConstraints(xdev) 1
	}
    }
}

set root [lindex [file split [pwd]] 0]

# A really long file name
# length of long is 1216 chars, which should be greater than any static
# buffer or allowable filename.

set long "abcdefghihjllmnopqrstuvwxyz01234567890"
append long $long
append long $long
append long $long
append long $long
append long $long

test fCmd-1.1 {TclFileRenameCmd} {notRoot} {
    cleanup
    createfile tf1
    file rename tf1 tf2
    glob tf*
} {tf2}

test fCmd-2.1 {TclFileCopyCmd} {notRoot} {
    cleanup
    createfile tf1
    file copy tf1 tf2
    lsort [glob tf*]
} {tf1 tf2}

test fCmd-3.1 {FileCopyRename: FileForceOption fails} {notRoot} {
    list [catch {file rename -xyz} msg] $msg
} {1 {bad option "-xyz": should be -force or --}}
test fCmd-3.2 {FileCopyRename: not enough args} {notRoot} {
    list [catch {file rename xyz} msg] $msg
} {1 {wrong # args: should be "file rename ?options? source ?source ...? target"}}
test fCmd-3.3 {FileCopyRename: Tcl_TranslateFileName fails} {notRoot} {
    list [catch {file rename xyz ~_totally_bogus_user} msg] $msg
} {1 {user "_totally_bogus_user" doesn't exist}}
test fCmd-3.4 {FileCopyRename: Tcl_TranslateFileName passes} {notRoot} {
    cleanup
    list [catch {file copy tf1 ~} msg] $msg
} {1 {error copying "tf1": no such file or directory}}
test fCmd-3.5 {FileCopyRename: target doesn't exist: stat(target) != 0} {notRoot} {
    cleanup
    list [catch {file rename tf1 tf2 tf3} msg] $msg
} {1 {error renaming: target "tf3" is not a directory}}
test fCmd-3.6 {FileCopyRename: target tf3 is not a dir: !S_ISDIR(target)} \
	{notRoot} {
    cleanup
    createfile tf3
    list [catch {file rename tf1 tf2 tf3} msg] $msg
} {1 {error renaming: target "tf3" is not a directory}}
test fCmd-3.7 {FileCopyRename: target exists & is directory} {notRoot} {
    cleanup
    file mkdir td1
    createfile tf1 tf1
    file rename tf1 td1
    contents [file join td1 tf1]
} {tf1}
test fCmd-3.8 {FileCopyRename: too many arguments: argc - i > 2} {notRoot} {
    cleanup
    list [catch {file rename tf1 tf2 tf3} msg] $msg
} {1 {error renaming: target "tf3" is not a directory}}
test fCmd-3.9 {FileCopyRename: too many arguments: argc - i > 2} {notRoot} {
    cleanup
    list [catch {file copy -force -- tf1 tf2 tf3} msg] $msg
} {1 {error copying: target "tf3" is not a directory}}
test fCmd-3.10 {FileCopyRename: just 2 arguments} {notRoot} {
    cleanup
    createfile tf1 tf1
    file rename tf1 tf2
    contents tf2
} {tf1}
test fCmd-3.11 {FileCopyRename: just 2 arguments} {notRoot} {
    cleanup
    createfile tf1 tf1
    file rename -force -force -- tf1 tf2
    contents tf2
} {tf1}
test fCmd-3.12 {FileCopyRename: move each source: 1 source} {notRoot} {
    cleanup
    createfile tf1 tf1
    file mkdir td1
    file rename tf1 td1
    contents [file join td1 tf1]
} {tf1}
test fCmd-3.13 {FileCopyRename: move each source: multiple sources} {notRoot} {
    cleanup
    createfile tf1 tf1
    createfile tf2 tf2
    createfile tf3 tf3
    createfile tf4 tf4
    file mkdir td1
    file rename tf1 tf2 tf3 tf4 td1
    list [contents [file join td1 tf1]] [contents [file join td1 tf2]] \
	[contents [file join td1 tf3]] [contents [file join td1 tf4]]
} {tf1 tf2 tf3 tf4}
test fCmd-3.14 {FileCopyRename: FileBasename fails} {notRoot} {
    cleanup
    file mkdir td1
    list [catch {file rename ~_totally_bogus_user td1} msg] $msg
} {1 {user "_totally_bogus_user" doesn't exist}}
test fCmd-3.15 {FileCopyRename: source[0] == '\0'} {notRoot unixOrPc} {
    cleanup
    file mkdir td1
    list [catch {file rename / td1} msg] $msg
} {1 {error renaming "/" to "td1": file already exists}}
test fCmd-3.16 {FileCopyRename: break on first error} {notRoot} {
    cleanup
    createfile tf1 
    createfile tf2 
    createfile tf3 
    createfile tf4 
    file mkdir td1
    createfile [file join td1 tf3]
    list [catch {file rename tf1 tf2 tf3 tf4 td1} msg] $msg
} [subst {1 {error renaming "tf3" to "[file join td1 tf3]": file already exists}}]

test fCmd-4.1 {TclFileMakeDirsCmd: make each dir: 1 dir} {notRoot} {
    cleanup
    file mkdir td1
    glob td*
} {td1}
test fCmd-4.2 {TclFileMakeDirsCmd: make each dir: multiple dirs} {notRoot} {
    cleanup
    file mkdir td1 td2 td3
    lsort [glob td*]
} {td1 td2 td3}
test fCmd-4.3 {TclFileMakeDirsCmd: stops on first error} {notRoot} {
    cleanup
    createfile tf1
    catch {file mkdir td1 td2 tf1 td3 td4}
    glob td1 td2 tf1 td3 td4
} {td1 td2 tf1}
test fCmd-4.4 {TclFileMakeDirsCmd: Tcl_TranslateFileName fails} {notRoot} {
    cleanup
    list [catch {file mkdir ~_totally_bogus_user} msg] $msg
} {1 {user "_totally_bogus_user" doesn't exist}}
test fCmd-4.5 {TclFileMakeDirsCmd: Tcl_SplitPath returns 0: *name == '\0'} \
	{notRoot} {
    cleanup
    list [catch {file mkdir ""} msg] $msg
} {1 {can't create directory "": no such file or directory}}
test fCmd-4.6 {TclFileMakeDirsCmd: one level deep} {notRoot} {
    cleanup
    file mkdir td1
    glob td1
} {td1}
test fCmd-4.7 {TclFileMakeDirsCmd: multi levels deep} {notRoot} {
    cleanup
    file mkdir [file join td1 td2 td3 td4]
    glob td1 [file join td1 td2]
} "td1 [file join td1 td2]"
test fCmd-4.8 {TclFileMakeDirsCmd: already exist: lstat(target) == 0} {notRoot} {
    cleanup
    file mkdir td1
    set x [file exists td1]
    file mkdir td1
    list $x [file exists td1]
} {1 1}
test fCmd-4.9 {TclFileMakeDirsCmd: exists, not dir} {notRoot} {
    cleanup
    createfile tf1
    list [catch {file mkdir tf1} msg] $msg
} [subst {1 {can't create directory "[file join tf1]": file already exists}}]
test fCmd-4.10 {TclFileMakeDirsCmd: exists, is dir} {notRoot} {
    cleanup
    file mkdir td1
    set x [file exists td1]
    file mkdir td1
    list $x [file exists td1]
} {1 1}
test fCmd-4.11 {TclFileMakeDirsCmd: doesn't exist: errno != ENOENT} \
	{unixOnly notRoot testchmod} {
    cleanup
    file mkdir td1/td2/td3
    testchmod 000 td1/td2
    set msg [list [catch {file mkdir td1/td2/td3/td4} msg] $msg]
    testchmod 755 td1/td2
    set msg
} {1 {can't create directory "td1/td2/td3": permission denied}}
test fCmd-4.12 {TclFileMakeDirsCmd: doesn't exist: errno != ENOENT} {macOnly} {
    cleanup
    list [catch {file mkdir nonexistentvolume:} msg] $msg
} {1 {can't create directory "nonexistentvolume:": invalid argument}}
test fCmd-4.13 {TclFileMakeDirsCmd: doesn't exist: errno == ENOENT} {notRoot} {
    cleanup
    set x [file exists td1]
    file mkdir td1
    list $x [file exists td1]
} {0 1}
test fCmd-4.14 {TclFileMakeDirsCmd: TclpCreateDirectory fails} \
	{unixOnly notRoot} {
    cleanup
    file delete -force foo
    file mkdir foo
    file attr foo -perm 040000
    set result [list [catch {file mkdir foo/tf1} msg] $msg]
    file delete -force foo
    set result
} {1 {can't create directory "foo/tf1": permission denied}}
test fCmd-4.15 {TclFileMakeDirsCmd: TclpCreateDirectory fails} {macOnly} {
    list [catch {file mkdir ${root}:} msg] $msg
} [subst {1 {can't create directory "${root}:": no such file or directory}}]
test fCmd-4.16 {TclFileMakeDirsCmd: TclpCreateDirectory succeeds} {notRoot} {
    cleanup
    file mkdir tf1
    file exists tf1
} {1}

test fCmd-5.1 {TclFileDeleteCmd: FileForceOption fails} {notRoot} {
    list [catch {file delete -xyz} msg] $msg
} {1 {bad option "-xyz": should be -force or --}}
test fCmd-5.2 {TclFileDeleteCmd: not enough args} {notRoot} {
    list [catch {file delete -force -force} msg] $msg
} {1 {wrong # args: should be "file delete ?options? file ?file ...?"}}
test fCmd-5.3 {TclFileDeleteCmd: 1 file} {notRoot} {
    cleanup
    createfile tf1
    createfile tf2
    file mkdir td1
    file delete tf2
    glob tf* td*
} {tf1 td1}
test fCmd-5.4 {TclFileDeleteCmd: multiple files} {notRoot} {
    cleanup
    createfile tf1
    createfile tf2
    file mkdir td1
    set x [list [file exists tf1] [file exists tf2] [file exists td1]]
    file delete tf1 td1 tf2
    lappend x [file exists tf1] [file exists tf2] [file exists tf3]
} {1 1 1 0 0 0}
test fCmd-5.5 {TclFileDeleteCmd: stop at first error} {notRoot unixOrPc} {
    cleanup
    createfile tf1
    createfile tf2
    file mkdir td1
    catch {file delete tf1 td1 $root tf2}
    list [file exists tf1] [file exists tf2] [file exists td1]
} {0 1 0}
test fCmd-5.6 {TclFileDeleteCmd: Tcl_TranslateFileName fails} {notRoot} {
    list [catch {file delete ~_totally_bogus_user} msg] $msg
} {1 {user "_totally_bogus_user" doesn't exist}}
test fCmd-5.7 {TclFileDeleteCmd: Tcl_TranslateFileName succeeds} {notRoot} {
    catch {file delete ~/tf1}
    createfile ~/tf1
    file delete ~/tf1
} {}
test fCmd-5.8 {TclFileDeleteCmd: file doesn't exist: lstat(name) != 0} {notRoot} {
    cleanup
    set x [file exists tf1]
    file delete tf1
    list $x [file exists tf1]
} {0 0}    
test fCmd-5.9 {TclFileDeleteCmd: is directory} {notRoot} {
    cleanup
    file mkdir td1
    file delete td1
    file exists td1
} {0}
test fCmd-5.10 {TclFileDeleteCmd: TclpRemoveDirectory fails} {notRoot} {
    cleanup
    file mkdir [file join td1 td2]
    list [catch {file delete td1} msg] $msg
} {1 {error deleting "td1": directory not empty}}
test fCmd-5.11 {TclFileDeleteCmd: TclpRemoveDirectory with cwd inside} {notRoot} {
    cleanup
    set dir [pwd]
    file mkdir [file join td1 td2]
    cd [file join td1 td2]
    set res [list [catch {file delete -force [file dirname [pwd]]} msg]]
    cd $dir
    lappend res [file exists td1] $msg
} {0 0 {}}
test fCmd-5.12 {TclFileDeleteCmd: TclpRemoveDirectory with bad perms} {unixOnly} {
    cleanup
    file mkdir [file join td1 td2]
    #exec chmod u-rwx [file join td1 td2]
    file attributes [file join td1 td2] -permissions u+rwx
    set res [list [catch {file delete -force td1} msg]]
    lappend res [file exists td1] $msg
} {0 0 {}}

test fCmd-6.1 {CopyRenameOneFile: bad source} {notRoot} {
    # can't test this, because it's caught by FileCopyRename
} {}
test fCmd-6.2 {CopyRenameOneFile: bad target} {notRoot} {
    # can't test this, because it's caught by FileCopyRename
} {}
test fCmd-6.3 {CopyRenameOneFile: lstat(source) != 0} {notRoot} {
    cleanup
    list [catch {file rename tf1 tf2} msg] $msg
} {1 {error renaming "tf1": no such file or directory}}
test fCmd-6.4 {CopyRenameOneFile: lstat(source) == 0} {notRoot} {
    cleanup
    createfile tf1
    file rename tf1 tf2
    glob tf*
} {tf2}
test fCmd-6.5 {CopyRenameOneFile: lstat(target) != 0} {notRoot} {
    cleanup
    createfile tf1
    file rename tf1 tf2
    glob tf*
} {tf2}
test fCmd-6.6 {CopyRenameOneFile: errno != ENOENT} {unixOnly notRoot testchmod} {
    cleanup
    file mkdir td1
    testchmod 000 td1
    createfile tf1
    set msg [list [catch {file rename tf1 td1} msg] $msg]
    testchmod 755 td1
    set msg
} {1 {error renaming "tf1" to "td1/tf1": permission denied}}
test fCmd-6.7 {CopyRenameOneFile: errno != ENOENT} {pcOnly 95} {
    cleanup
    createfile tf1
    list [catch {file rename tf1 $long} msg] $msg
} [subst {1 {error renaming "tf1" to "$long": file name too long}}]
test fCmd-6.8 {CopyRenameOneFile: errno != ENOENT} {macOnly} {
    cleanup
    createfile tf1
    list [catch {file rename tf1 $long} msg] $msg
} [subst {1 {error renaming "tf1" to "$long": file name too long}}]
test fCmd-6.9 {CopyRenameOneFile: errno == ENOENT} {unixOnly notRoot} {
    cleanup
    createfile tf1
    file rename tf1 tf2
    glob tf*
} {tf2}
test fCmd-6.10 {CopyRenameOneFile: lstat(target) == 0} {notRoot} {
    cleanup
    createfile tf1
    createfile tf2
    list [catch {file rename tf1 tf2} msg] $msg
} {1 {error renaming "tf1" to "tf2": file already exists}}
test fCmd-6.11 {CopyRenameOneFile: force == 0} {notRoot} {
    cleanup
    createfile tf1
    createfile tf2
    list [catch {file rename tf1 tf2} msg] $msg
} {1 {error renaming "tf1" to "tf2": file already exists}}
test fCmd-6.12 {CopyRenameOneFile: force != 0} {notRoot} {
    cleanup
    createfile tf1
    createfile tf2
    file rename -force tf1 tf2
    glob tf*
} {tf2}
test fCmd-6.13 {CopyRenameOneFile: source is dir, target is file} {notRoot} {
    cleanup
    file mkdir td1
    file mkdir td2
    createfile [file join td2 td1]
    list [catch {file rename -force td1 td2} msg] $msg
} [subst {1 {can't overwrite file "[file join td2 td1]" with directory "td1"}}]
test fCmd-6.14 {CopyRenameOneFile: source is file, target is dir} {notRoot} {
    cleanup
    createfile tf1
    file mkdir [file join td1 tf1]
    list [catch {file rename -force tf1 td1} msg] $msg
} [subst {1 {can't overwrite directory "[file join td1 tf1]" with file "tf1"}}]
test fCmd-6.15 {CopyRenameOneFile: TclpRenameFile succeeds} {notRoot} {
    cleanup
    file mkdir [file join td1 td2]
    file mkdir td2
    createfile [file join td2 tf1]
    file rename -force td2 td1
    file exists [file join td1 td2 tf1]
} {1}
test fCmd-6.16 {CopyRenameOneFile: TclpCopyRenameOneFile fails} {notRoot} {
    cleanup
    file mkdir [file join td1 td2]
    createfile [file join td1 td2 tf1]
    file mkdir td2
    list [catch {file rename -force td2 td1} msg] $msg
} [subst {1 {error renaming "td2" to "[file join td1 td2]": file already exists}}]

test fCmd-6.17 {CopyRenameOneFile: errno == EINVAL} {notRoot} {
    cleanup
    list [catch {file rename -force $root tf1} msg] $msg
} [subst {1 {error renaming "$root" to "tf1": trying to rename a volume or move a directory into itself}}]
test fCmd-6.18 {CopyRenameOneFile: errno != EXDEV} {notRoot} {
    cleanup
    file mkdir [file join td1 td2]
    createfile [file join td1 td2 tf1]
    file mkdir td2
    list [catch {file rename -force td2 td1} msg] $msg
} [subst {1 {error renaming "td2" to "[file join td1 td2]": file already exists}}]
test fCmd-6.19 {CopyRenameOneFile: errno == EXDEV} {unixOnly notRoot} {
    cleanup /tmp
    createfile tf1
    file rename tf1 /tmp
    glob tf* /tmp/tf1
} {/tmp/tf1}
test fCmd-6.20 {CopyRenameOneFile: errno == EXDEV} {pcOnly} {
    catch {file delete -force c:/tcl8975@ d:/tcl8975@}
    file mkdir c:/tcl8975@
    if [catch {file rename c:/tcl8975@ d:/}] {
	set msg d:/tcl8975@
    } else {
	set msg [glob c:/tcl8975@ d:/tcl8975@]
	file delete -force d:/tcl8975@
    }
    file delete -force c:/tcl8975@
    set msg
} {d:/tcl8975@}
test fCmd-6.21 {CopyRenameOneFile: copy/rename: S_ISDIR(source)} \
	{unixOnly notRoot} {
    cleanup /tmp
    file mkdir td1
    file rename td1 /tmp
    glob td* /tmp/td*
} {/tmp/td1}
test fCmd-6.22 {CopyRenameOneFile: copy/rename: !S_ISDIR(source)} \
	{unixOnly notRoot} {
    cleanup /tmp
    createfile tf1
    file rename tf1 /tmp
    glob tf* /tmp/tf*
} {/tmp/tf1}
test fCmd-6.23 {CopyRenameOneFile: TclpCopyDirectory failed} \
	{unixOnly notRoot xdev} {
    cleanup /tmp
    file mkdir td1/td2/td3
    file attributes td1 -permissions 0000
    set msg [list [catch {file rename td1 /tmp} msg] $msg]
    file attributes td1 -permissions 0755
    set msg 
} {1 {error renaming "td1": permission denied}}
test fCmd-6.24 {CopyRenameOneFile: error uses original name} \
	{unixOnly notRoot} {
    cleanup
    file mkdir ~/td1/td2
    set td1name [file join [file dirname ~] [file tail ~] td1]
    file attributes $td1name -permissions 0000
    set msg [list [catch {file copy ~/td1 td1} msg] $msg]
    file attributes $td1name -permissions 0755
    file delete -force ~/td1
    set msg
} {1 {error copying "~/td1": permission denied}}
test fCmd-6.25 {CopyRenameOneFile: error uses original name} \
	{unixOnly notRoot} {
    cleanup
    file mkdir td2
    file mkdir ~/td1
    set td1name [file join [file dirname ~] [file tail ~] td1]
    file attributes $td1name -permissions 0000
    set msg [list [catch {file copy td2 ~/td1} msg] $msg]
    file attributes $td1name -permissions 0755
    file delete -force ~/td1
    set msg
} {1 {error copying "td2" to "~/td1/td2": permission denied}}
test fCmd-6.26 {CopyRenameOneFile: doesn't use original name} \
	{unixOnly notRoot} {
    cleanup
    file mkdir ~/td1/td2
    set td2name [file join [file dirname ~] [file tail ~] td1 td2]
    file attributes $td2name -permissions 0000
    set msg [list [catch {file copy ~/td1 td1} msg] $msg]
    file attributes $td2name -permissions 0755
    file delete -force ~/td1
    set msg
} "1 {error copying \"~/td1\" to \"td1\": \"[file join [file dirname ~] [file tail ~] td1 td2]\": permission denied}"
test fCmd-6.27 {CopyRenameOneFile: TclpCopyDirectory failed} \
	{unixOnly notRoot xdev} {
    cleanup /tmp
    file mkdir td1/td2/td3
    file mkdir /tmp/td1
    createfile /tmp/td1/tf1
    list [catch {file rename -force td1 /tmp} msg] $msg
} {1 {error renaming "td1" to "/tmp/td1": file already exists}}
test fCmd-6.28 {CopyRenameOneFile: TclpCopyDirectory failed} \
	{unixOnly notRoot xdev} {
    cleanup /tmp
    file mkdir td1/td2/td3
    file attributes td1/td2/td3 -permissions 0000
    set msg [list [catch {file rename td1 /tmp} msg] $msg]
    file attributes td1/td2/td3 -permissions 0755
    set msg
} {1 {error renaming "td1" to "/tmp/td1": "td1/td2/td3": permission denied}}
test fCmd-6.29 {CopyRenameOneFile: TclpCopyDirectory passed} \
	{unixOnly notRoot xdev} {
    cleanup /tmp
    file mkdir td1/td2/td3
    file rename td1 /tmp
    glob td* /tmp/td1/t*
} {/tmp/td1/td2}
test fCmd-6.30 {CopyRenameOneFile: TclpRemoveDirectory failed} \
	{unixOnly notRoot} {
    cleanup
    file mkdir foo/bar
    file attr foo -perm 040555
    set catchResult [catch {file rename foo/bar /tmp} msg]
    set msg [lindex [split $msg :] end]
    catch {file delete /tmp/bar}
    catch {file attr foo -perm 040777}
    catch {file delete -force foo}
    list $catchResult $msg
} {1 { permission denied}}
test fCmd-6.31 {CopyRenameOneFile: TclpDeleteFile passed} \
	{unixOnly notRoot xdev} {
    catch {cleanup /tmp}
    file mkdir /tmp/td1
    createfile /tmp/td1/tf1
    file rename /tmp/td1/tf1 tf1
    list [file exists /tmp/td1/tf1] [file exists tf1]
} {0 1}
test fCmd-6.32 {CopyRenameOneFile: copy} {notRoot} {
    cleanup
    list [catch {file copy tf1 tf2} msg] $msg
} {1 {error copying "tf1": no such file or directory}}
catch {cleanup /tmp}

test fCmd-7.1 {FileForceOption: none} {notRoot} {
    cleanup
    file mkdir [file join tf1 tf2]
    list [catch {file delete tf1} msg] $msg
} {1 {error deleting "tf1": directory not empty}}
test fCmd-7.2 {FileForceOption: -force} {notRoot} {
    cleanup
    file mkdir [file join tf1 tf2]
    file delete -force tf1
} {}
test fCmd-7.3 {FileForceOption: --} {notRoot} {
    createfile -tf1
    file delete -- -tf1
} {}
test fCmd-7.4 {FileForceOption: bad option} {notRoot} {
    createfile -tf1
    set msg [list [catch {file delete -tf1} msg] $msg]
    file delete -- -tf1
    set msg
} {1 {bad option "-tf1": should be -force or --}}
test fCmd-7.5 {FileForceOption: multiple times through loop} {notRoot} {
    createfile --
    createfile -force
    file delete -force -force -- -- -force
    list [catch {glob -- -- -force} msg] $msg
} {1 {no files matched glob patterns "-- -force"}}

test fCmd-8.1 {FileBasename: basename of ~user: argc == 1 && *path == ~} \
	{unixOnly notRoot knownBug} {
    # Labelled knownBug because it is dangerous [Bug: 3881]
    file mkdir td1
    file attr td1 -perm 040000
    set result [list [catch {file rename ~$user td1} msg] $msg]
    file delete -force td1
    set result
} "1 {error renaming \"~$user\" to \"td1/[file tail ~$user]\": permission denied}"
test fCmd-8.2 {FileBasename: basename of ~user: argc == 1 && *path == ~} {
    string equal [file tail ~$user] ~$user
} 0
test fCmd-8.3 {file copy and path translation: ensure correct error} {
    list [catch {file copy ~ [file join this file doesnt exist]} res] $res
} [list 1 \
  "error copying \"~\" to \"[file join this file doesnt exist]\":\
  no such file or directory"]

test fCmd-9.1 {file rename: comprehensive: EACCES} {unixOnly notRoot} {
    cleanup
    file mkdir td1
    file mkdir td2
    file attr td2 -perm 040000
    set result [list [catch {file rename td1 td2/} msg] $msg]
    file delete -force td2
    file delete -force td1
    set result
} {1 {error renaming "td1" to "td2/td1": permission denied}}
test fCmd-9.2 {file rename: comprehensive: source doesn't exist} {notRoot} {
    cleanup
    list [catch {file rename tf1 tf2} msg] $msg
} {1 {error renaming "tf1": no such file or directory}}
test fCmd-9.3 {file rename: comprehensive: file to new name} {notRoot testchmod} {
    cleanup
    createfile tf1
    createfile tf2
    testchmod 444 tf2
    file rename tf1 tf3
    file rename tf2 tf4
    list [lsort [glob tf*]] [file writable tf3] [file writable tf4]
} {{tf3 tf4} 1 0}    
test fCmd-9.4 {file rename: comprehensive: dir to new name} {unixOrPc notRoot testchmod} {
    cleanup
    file mkdir td1 td2
    testchmod 555 td2
    file rename td1 td3
    file rename td2 td4
    list [lsort [glob td*]] [file writable td3] [file writable td4]
} {{td3 td4} 1 0}    
test fCmd-9.5 {file rename: comprehensive: file to self} {notRoot testchmod} {
    cleanup
    createfile tf1 tf1
    createfile tf2 tf2
    testchmod 444 tf2
    file rename -force tf1 tf1
    file rename -force tf2 tf2
    list [contents tf1] [contents tf2] [file writable tf1] [file writable tf2]
} {tf1 tf2 1 0}    
test fCmd-9.6 {file rename: comprehensive: dir to self} {notRoot unixOrPc testchmod} {
    cleanup
    file mkdir td1
    file mkdir td2
    testchmod 555 td2
    file rename -force td1 .
    file rename -force td2 .
    list [lsort [glob td*]] [file writable td1] [file writable td2]
} {{td1 td2} 1 0}    
test fCmd-9.7 {file rename: comprehensive: file to existing file} {notRoot testchmod} {
    cleanup
    createfile tf1
    createfile tf2
    createfile tfs1
    createfile tfs2
    createfile tfs3
    createfile tfs4
    createfile tfd1
    createfile tfd2
    createfile tfd3
    createfile tfd4
    testchmod 444 tfs3
    testchmod 444 tfs4
    testchmod 444 tfd2
    testchmod 444 tfd4
    set msg [list [catch {file rename tf1 tf2} msg] $msg]
    file rename -force tfs1 tfd1
    file rename -force tfs2 tfd2
    file rename -force tfs3 tfd3
    file rename -force tfs4 tfd4
    list [lsort [glob tf*]] $msg [file writable tfd1] [file writable tfd2] [file writable tfd3] [file writable tfd4] 
} {{tf1 tf2 tfd1 tfd2 tfd3 tfd4} {1 {error renaming "tf1" to "tf2": file already exists}} 1 1 0 0}
test fCmd-9.8 {file rename: comprehensive: dir to empty dir} {notRoot testchmod} {
    # Under unix, you can rename a read-only directory, but you can't
    # move it into another directory.

    cleanup
    file mkdir td1
    file mkdir [file join td2 td1]
    file mkdir tds1
    file mkdir tds2
    file mkdir tds3
    file mkdir tds4
    file mkdir [file join tdd1 tds1]
    file mkdir [file join tdd2 tds2]
    file mkdir [file join tdd3 tds3]
    file mkdir [file join tdd4 tds4]
    if {$tcl_platform(platform) != "unix" && $tcl_platform(platform) != "macintosh"} {
	testchmod 555 tds3
	testchmod 555 tds4
    }
    if {$tcl_platform(platform) != "macintosh"} {
    	testchmod 555 [file join tdd2 tds2]
    	testchmod 555 [file join tdd4 tds4]
    }
    set msg [list [catch {file rename td1 td2} msg] $msg]
    file rename -force tds1 tdd1
    file rename -force tds2 tdd2
    file rename -force tds3 tdd3
    file rename -force tds4 tdd4
    if {$tcl_platform(platform) != "unix" && $tcl_platform(platform) != "macintosh"} {
	set w3 [file writable [file join tdd3 tds3]]
	set w4 [file writable [file join tdd4 tds4]]
    } else {
	set w3 0
	set w4 0
    }
    list [lsort [glob td*]] $msg [file writable [file join tdd1 tds1]] \
    [file writable [file join tdd2 tds2]] $w3 $w4
} [subst {{td1 td2 tdd1 tdd2 tdd3 tdd4} {1 {error renaming "td1" to "[file join td2 td1]": file already exists}} 1 1 0 0}]
test fCmd-9.9 {file rename: comprehensive: dir to non-empty dir} {notRoot testchmod} {
    cleanup
    file mkdir tds1
    file mkdir tds2
    file mkdir [file join tdd1 tds1 xxx]
    file mkdir [file join tdd2 tds2 xxx]
    if {$tcl_platform(platform) != "unix" && $tcl_platform(platform) != "macintosh"} {
	testchmod 555 tds2
    }
    set a1 [list [catch {file rename -force tds1 tdd1} msg] $msg]
    set a2 [list [catch {file rename -force tds2 tdd2} msg] $msg]
    if {$tcl_platform(platform) != "unix" && $tcl_platform(platform) != "macintosh"} {
	set w2 [file writable tds2]
    } else {
	set w2 0
    }
    list [lsort [glob td*]] $a1 $a2 [file writable tds1] $w2
} [subst {{tdd1 tdd2 tds1 tds2} {1 {error renaming "tds1" to "[file join tdd1 tds1]": file already exists}} {1 {error renaming "tds2" to "[file join tdd2 tds2]": file already exists}} 1 0}]
test fCmd-9.10 {file rename: comprehensive: file to new name and dir} {notRoot testchmod} {
    cleanup
    createfile tf1
    createfile tf2
    file mkdir td1
    testchmod 444 tf2
    file rename tf1 [file join td1 tf3]
    file rename tf2 [file join td1 tf4]
    list [catch {glob tf*}] [lsort [glob -directory td1 t*]] \
    [file writable [file join td1 tf3]] [file writable [file join td1 tf4]]
} [subst {1 {[file join td1 tf3] [file join td1 tf4]} 1 0}]
test fCmd-9.11 {file rename: comprehensive: dir to new name and dir} {notRoot testchmod} {
    cleanup
    file mkdir td1
    file mkdir td2
    file mkdir td3
    if {$tcl_platform(platform) != "unix" && $tcl_platform(platform) != "macintosh"} {
	testchmod 555 td2
    }
    file rename td1 [file join td3 td3]
    file rename td2 [file join td3 td4]
    if {$tcl_platform(platform) != "unix" && $tcl_platform(platform) != "macintosh"} {
	set w4 [file writable [file join td3 td4]]
    } else {
        set w4 0
    }
    list [lsort [glob td*]] [lsort [glob -directory td3 t*]] \
    [file writable [file join td3 td3]] $w4
} [subst {td3 {[file join td3 td3] [file join td3 td4]} 1 0}]
test fCmd-9.12 {file rename: comprehensive: target exists} {notRoot testchmod} {
    cleanup
    file mkdir [file join td1 td2] [file join td2 td1]
    if {$tcl_platform(platform) != "macintosh"} {
    	testchmod 555 [file join td2 td1]
    }
    file mkdir [file join td3 td4] [file join td4 td3]
    file rename -force td3 td4
    set msg [list [file exists td3] [file exists [file join td4 td3 td4]] \
    [catch {file rename td1 td2} msg] $msg]
    if {$tcl_platform(platform) != "macintosh"} {
    	testchmod 755 [file join td2 td1]
    }
    set msg
} [subst {0 1 1 {error renaming "td1" to "[file join td2 td1]": file already exists}}]
test fCmd-9.13 {file rename: comprehensive: can't overwrite target} {notRoot} {
    cleanup
    file mkdir [file join td1 td2] [file join td2 td1 td4]
    list [catch {file rename -force td1 td2} msg] $msg
} [subst {1 {error renaming "td1" to "[file join td2 td1]": file already exists}}]
test fCmd-9.14 {file rename: comprehensive: dir into self} {notRoot} {
    cleanup
    file mkdir td1
    list [glob td*] [list [catch {file rename td1 td1} msg] $msg]
} [subst {td1 {1 {error renaming "td1" to "[file join td1 td1]": trying to rename a volume or move a directory into itself}}}]
test fCmd-9.15 {file rename: comprehensive: source and target incompatible} \
	{notRoot} {
    cleanup
    file mkdir td1
    createfile tf1
    list [catch {file rename -force td1 tf1} msg] $msg
} {1 {can't overwrite file "tf1" with directory "td1"}}
test fCmd-9.16 {file rename: comprehensive: source and target incompatible} \
	{notRoot} {
    cleanup
    file mkdir td1/tf1
    createfile tf1
    list [catch {file rename -force tf1 td1} msg] $msg
} [subst {1 {can't overwrite directory "[file join td1 tf1]" with file "tf1"}}]

test fCmd-10.1 {file copy: comprehensive: source doesn't exist} {notRoot} {
    cleanup
    list [catch {file copy tf1 tf2} msg] $msg
} {1 {error copying "tf1": no such file or directory}}
test fCmd-10.2 {file copy: comprehensive: file to new name} {notRoot testchmod} {
    cleanup
    createfile tf1 tf1
    createfile tf2 tf2
    testchmod 444 tf2
    file copy tf1 tf3
    file copy tf2 tf4
    list [lsort [glob tf*]] [contents tf3] [contents tf4] [file writable tf3] [file writable tf4]
} {{tf1 tf2 tf3 tf4} tf1 tf2 1 0}
test fCmd-10.3 {file copy: comprehensive: dir to new name} {notRoot unixOrPc testchmod} {
    cleanup
    file mkdir [file join td1 tdx]
    file mkdir [file join td2 tdy]
    testchmod 555 td2
    file copy td1 td3
    file copy td2 td4
    set msg [list [lsort [glob td*]] [glob -directory td3 t*] \
	    [glob -directory td4 t*] [file writable td3] [file writable td4]]
    if {$tcl_platform(platform) != "macintosh"} {
    	testchmod 755 td2
    	testchmod 755 td4
    }
    set msg
} [subst {{td1 td2 td3 td4} [file join td3 tdx] [file join td4 tdy] 1 0}]
test fCmd-10.4 {file copy: comprehensive: file to existing file} {notRoot testchmod} {
    cleanup
    createfile tf1
    createfile tf2
    createfile tfs1
    createfile tfs2
    createfile tfs3
    createfile tfs4
    createfile tfd1
    createfile tfd2
    createfile tfd3
    createfile tfd4
    testchmod 444 tfs3
    testchmod 444 tfs4
    testchmod 444 tfd2
    testchmod 444 tfd4
    set msg [list [catch {file copy tf1 tf2} msg] $msg]
    file copy -force tfs1 tfd1
    file copy -force tfs2 tfd2
    file copy -force tfs3 tfd3
    file copy -force tfs4 tfd4
    list [lsort [glob tf*]] $msg [file writable tfd1] [file writable tfd2] [file writable tfd3] [file writable tfd4] 
} {{tf1 tf2 tfd1 tfd2 tfd3 tfd4 tfs1 tfs2 tfs3 tfs4} {1 {error copying "tf1" to "tf2": file already exists}} 1 1 0 0}
test fCmd-10.5 {file copy: comprehensive: dir to empty dir} {notRoot testchmod} {
    cleanup
    file mkdir td1
    file mkdir [file join td2 td1]
    file mkdir tds1
    file mkdir tds2
    file mkdir tds3
    file mkdir tds4
    file mkdir [file join tdd1 tds1]
    file mkdir [file join tdd2 tds2]
    file mkdir [file join tdd3 tds3]
    file mkdir [file join tdd4 tds4]
    if {$tcl_platform(platform) != "macintosh"} {
	testchmod 555 tds3
	testchmod 555 tds4
	testchmod 555 [file join tdd2 tds2]
	testchmod 555 [file join tdd4 tds4]
    }
    set a1 [list [catch {file copy td1 td2} msg] $msg]
    set a2 [list [catch {file copy -force tds1 tdd1} msg] $msg]
    set a3 [catch {file copy -force tds2 tdd2}]
    set a4 [catch {file copy -force tds3 tdd3}]
    set a5 [catch {file copy -force tds4 tdd4}]
    list [lsort [glob td*]] $a1 $a2 $a3 $a4 $a5 
} [subst {{td1 td2 tdd1 tdd2 tdd3 tdd4 tds1 tds2 tds3 tds4} {1 {error copying "td1" to "[file join td2 td1]": file already exists}} {1 {error copying "tds1" to "[file join tdd1 tds1]": file already exists}} 1 1 1}]
test fCmd-10.6 {file copy: comprehensive: dir to non-empty dir} \
	{notRoot unixOrPc testchmod} {
    cleanup
    file mkdir tds1
    file mkdir tds2
    file mkdir [file join tdd1 tds1 xxx]
    file mkdir [file join tdd2 tds2 xxx]
    testchmod 555 tds2
    set a1 [list [catch {file copy -force tds1 tdd1} msg] $msg]
    set a2 [list [catch {file copy -force tds2 tdd2} msg] $msg]
    list [lsort [glob td*]] $a1 $a2 [file writable tds1] [file writable tds2]
} [subst {{tdd1 tdd2 tds1 tds2} {1 {error copying "tds1" to "[file join tdd1 tds1]": file already exists}} {1 {error copying "tds2" to "[file join tdd2 tds2]": file already exists}} 1 0}]
test fCmd-10.7 {file rename: comprehensive: file to new name and dir} {notRoot testchmod} {
    cleanup
    createfile tf1
    createfile tf2
    file mkdir td1
    testchmod 444 tf2
    file copy tf1 [file join td1 tf3]
    file copy tf2 [file join td1 tf4]
    list [lsort [glob tf*]] [lsort [glob -directory td1 t*]] \
    [file writable [file join td1 tf3]] [file writable [file join td1 tf4]]
} [subst {{tf1 tf2} {[file join td1 tf3] [file join td1 tf4]} 1 0}]
test fCmd-10.8 {file rename: comprehensive: dir to new name and dir} \
	{notRoot unixOrPc testchmod} {
    cleanup
    file mkdir td1
    file mkdir td2
    file mkdir td3
    testchmod 555 td2
    file copy td1 [file join td3 td3]
    file copy td2 [file join td3 td4]
    list [lsort [glob td*]] [lsort [glob -directory td3 t*]] \
    [file writable [file join td3 td3]] [file writable [file join td3 td4]]
} [subst {{td1 td2 td3} {[file join td3 td3] [file join td3 td4]} 1 0}]
test fCmd-10.9 {file copy: comprehensive: source and target incompatible} \
	{notRoot} {
    cleanup
    file mkdir td1
    createfile tf1
    list [catch {file copy -force td1 tf1} msg] $msg
} {1 {can't overwrite file "tf1" with directory "td1"}}
test fCmd-10.10 {file copy: comprehensive: source and target incompatible} \
	{notRoot} {
    cleanup
    file mkdir [file join td1 tf1]
    createfile tf1
    list [catch {file copy -force tf1 td1} msg] $msg
} [subst {1 {can't overwrite directory "[file join td1 tf1]" with file "tf1"}}]
test fCmd-10.11 {file copy: copy to empty file name} {
    cleanup
    createfile tf1
    list [catch {file copy tf1 ""} msg] $msg
} {1 {error copying "tf1" to "": no such file or directory}}
test fCmd-10.12 {file rename: rename to empty file name} {
    cleanup
    createfile tf1
    list [catch {file rename tf1 ""} msg] $msg
} {1 {error renaming "tf1" to "": no such file or directory}}
cleanup    

# old tests    

test fCmd-11.1 {TclFileRenameCmd: -- option } {notRoot} {
    catch {file delete -force -- -tfa1}
    set s [createfile -tfa1]
    file rename -- -tfa1 tfa2
    set result [expr [checkcontent tfa2 $s] && ![file exists -tfa1]]
    file delete tfa2
    set result
} {1}

test fCmd-11.2 {TclFileRenameCmd: bad option } {notRoot} {
    catch {file delete -force -- tfa1}
    set s [createfile tfa1]
    set r1 [catch {file rename -x tfa1 tfa2}]
    set result [expr $r1 && [checkcontent tfa1 $s] && ![file exists tfa2]]
    file delete tfa1
    set result
} {1}

test fCmd-11.3 {TclFileRenameCmd: bad \# args} {
    catch {file rename -- }
} {1}

test fCmd-11.4 {TclFileRenameCmd: target filename translation failing} {notRoot} {
     global env
     set temp $env(HOME)
     unset env(HOME)
     set result [catch {file rename tfa ~/foobar }]
     set env(HOME) $temp
     set result
 } {1}

test fCmd-11.5 {TclFileRenameCmd: > 1 source & target is not a dir} {notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfa3}
    createfile tfa1 
    createfile tfa2 
    createfile tfa3 
    set result [catch {file rename tfa1 tfa2 tfa3}]
    file delete tfa1 tfa2 tfa3
    set result
} {1}

test fCmd-11.6 {TclFileRenameCmd: : single file into directory} {notRoot} {
    catch {file delete -force -- tfa1 tfad}
    set s [createfile tfa1]
    file mkdir tfad
    file rename tfa1 tfad
    set result [expr [checkcontent tfad/tfa1 $s] && ![file exists tfa1]]
    file delete -force tfad
    set result
} {1}

test fCmd-11.7 {TclFileRenameCmd: : multiple files into directory} {notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfad}
    set s1 [createfile tfa1 ]
    set s2 [createfile tfa2 ]
    file mkdir tfad
    file rename tfa1 tfa2 tfad
    set r1 [checkcontent tfad/tfa1 $s1]
    set r2 [checkcontent tfad/tfa2 $s2]
    
    set result [expr $r1 && $r2 && ![file exists tfa1] && ![file exists tfa2]]
	    
    file delete -force tfad
    set result
} {1}

test fCmd-11.8 {TclFileRenameCmd: error renaming file to directory} {notRoot} {
    catch {file delete -force -- tfa tfad}
    set s [createfile tfa ]
    file mkdir tfad
    file mkdir tfad/tfa
    set r1 [catch {file rename tfa tfad}]
    set r2 [checkcontent tfa $s]
    set r3 [file isdir tfad]
    set result [expr $r1 && $r2 && $r3 ]
    file delete -force tfa tfad
    set result
} {1}

#
# Coverage tests for renamefile() ;
#
test fCmd-12.1 {renamefile: source filename translation failing} {notRoot} {
    global env
    set temp $env(HOME)
    unset env(HOME)
    set result [catch {file rename ~/tfa1 tfa2}]
    set env(HOME) $temp
    set result
} {1}

test fCmd-12.2 {renamefile: src filename translation failing} {notRoot} {
    global env
    set temp $env(HOME)
    unset env(HOME)
    set s [createfile tfa1]
    file mkdir tfad
    set result [catch {file rename tfa1 ~/tfa2 tfad}]
    set env(HOME) $temp
    file delete -force tfad
    set result
} {1}

test fCmd-12.3 {renamefile: stat failing on source} {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    set r1 [catch {file rename tfa1 tfa2}]
    expr {$r1 && ![file exists tfa1] && ![file exists tfa2]}
} {1}

test fCmd-12.4 {renamefile: error renaming file to directory} {notRoot} {
    catch {file delete -force -- tfa tfad}
    set s1 [createfile tfa ]
    file mkdir tfad
    file mkdir tfad/tfa
    set r1 [catch {file rename tfa tfad}]
    set r2 [checkcontent tfa $s1]
    set r3 [file isdir tfad/tfa]
    set result [expr $r1 && $r2 && $r3]
    file delete -force tfa tfad
    set result
} {1}

test fCmd-12.5 {renamefile: error renaming directory to file} {notRoot} {
    catch {file delete -force -- tfa tfad}
    file mkdir tfa
    file mkdir tfad
    set s [createfile tfad/tfa]
    set r1 [catch {file rename tfa tfad}]
    set r2 [checkcontent tfad/tfa $s]
    set r3 [file isdir tfad]
    set r4 [file isdir tfa]
    set result [expr $r1 && $r2 && $r3 && $r4 ]
    file delete -force tfa tfad
    set result
} {1}

test fCmd-12.6 {renamefile: TclRenameFile succeeding} {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    set s [createfile tfa1]
    file rename tfa1 tfa2
    set result [expr [checkcontent tfa2 $s] && ![file exists tfa1]]
    file delete tfa2
    set result
} {1}

test fCmd-12.7 {renamefile: renaming directory into offspring} {notRoot} {
    catch {file delete -force -- tfad}
    file mkdir tfad
    file mkdir tfad/dir
    set result [catch {file rename tfad tfad/dir}]
    file delete -force tfad 
    set result
} {1}

test fCmd-12.8 {renamefile: generic error} {unixOnly notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    file mkdir tfa/dir
    file attributes tfa -permissions 0555
    set result [catch {file rename tfa/dir tfa2}]
    file attributes tfa -permissions 0777
    file delete -force tfa
    set result
} {1}


test fCmd-12.9 {renamefile: moving a file across volumes} {unixOnly notRoot} {
    catch {file delete -force -- tfa /tmp/tfa}
    set s [createfile tfa ]
    file rename tfa /tmp
    set result [expr [checkcontent /tmp/tfa $s] && ![file exists tfa]]
    file delete /tmp/tfa
    set result
} {1}

test fCmd-12.10 {renamefile: moving a directory across volumes } \
	{unixOnly notRoot} {
    catch {file delete -force -- tfad /tmp/tfad}
    file mkdir tfad
    set s [createfile tfad/a ]
    file rename tfad /tmp
    set restul [expr [checkcontent /tmp/tfad/a $s] && ![file exists tfad]]
    file delete -force /tmp/tfad
    set result
} {1}

#
# Coverage tests for TclCopyFilesCmd()
#
test fCmd-13.1 {TclCopyFilesCmd: -force option} {notRoot} {
    catch {file delete -force -- tfa1}
    set s [createfile tfa1]
    file copy -force  tfa1 tfa2
    set result [expr [checkcontent tfa2 $s] && [checkcontent tfa1 $s]]
    file delete tfa1 tfa2
    set result
} {1}

test fCmd-13.2 {TclCopyFilesCmd: -- option} {notRoot} {
    catch {file delete -force -- tfa1}
    set s [createfile -tfa1]
    file copy --  -tfa1 tfa2
    set result [expr [checkcontent tfa2 $s] &&  [checkcontent -tfa1 $s]]
    file delete -- -tfa1 tfa2
    set result
} {1}

test fCmd-13.3 {TclCopyFilesCmd: bad option} {notRoot} {
    catch {file delete -force -- tfa1}
    set s [createfile tfa1]
    set r1 [catch {file copy -x tfa1 tfa2}]
    set result [expr $r1 && [checkcontent tfa1 $s] && ![file exists tfa2]]
    file delete tfa1
    set result
} {1}

test fCmd-13.4 {TclCopyFilesCmd: bad \# args} {notRoot} {
    catch {file copy -- }
} {1}

test fCmd-13.5 {TclCopyFilesCmd: target filename translation failing} {
     global env
     set temp $env(HOME)
    unset env(HOME)
     set result [catch {file copy tfa ~/foobar }]
     set env(HOME) $temp
     set result
 } {1}

test fCmd-13.6 {TclCopyFilesCmd: > 1 source & target is not a dir} {notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfa3}
    createfile tfa1 
    createfile tfa2 
    createfile tfa3 
    set result [catch {file copy tfa1 tfa2 tfa3}]
    file delete tfa1 tfa2 tfa3
    set result
} {1}

test fCmd-13.7 {TclCopyFilesCmd: single file into directory} {notRoot} {
    catch {file delete -force -- tfa1 tfad}
    set s [createfile tfa1]
    file mkdir tfad
    file copy tfa1 tfad
    set result [expr [checkcontent tfad/tfa1 $s] &&  [checkcontent tfa1 $s]]
    file delete -force tfad tfa1
    set result
} {1}

test fCmd-13.8 {TclCopyFilesCmd: multiple files into directory} {notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfad}
    set s1 [createfile tfa1 ]
    set s2 [createfile tfa2 ]
    file mkdir tfad
    file copy tfa1 tfa2 tfad
    set r1 [checkcontent tfad/tfa1 $s1]
    set r2 [checkcontent tfad/tfa2 $s2]
    set r3 [checkcontent tfa1 $s1]
    set r4 [checkcontent tfa2 $s2]
    set result [expr $r1 && $r2 && $r3 && $r4 ]
	    
    file delete -force tfad tfa1 tfa2
    set result
} {1}

test fCmd-13.9 {TclCopyFilesCmd: error copying file to directory} {notRoot} {
    catch {file delete -force -- tfa tfad}
    set s [createfile tfa ]
    file mkdir tfad
    file mkdir tfad/tfa
    set r1 [catch {file copy tfa tfad}]
    set r2 [expr [checkcontent tfa $s] && [file isdir tfad/tfa]]
    set r3 [file isdir tfad]
    set result [expr $r1 && $r2 && $r3 ]
    file delete -force tfa tfad
    set result
} {1}

#
# Coverage tests for copyfile()
# 
test fCmd-14.1 {copyfile: source filename translation failing} {notRoot} {
    global env
    set temp $env(HOME)
    unset env(HOME)
    set result [catch {file copy ~/tfa1 tfa2}]
    set env(HOME) $temp
    set result
} {1}

test fCmd-14.2 {copyfile: dst filename translation failing} {notRoot} {
    global env
    set temp $env(HOME)
    unset env(HOME)
    set s [createfile tfa1]
    file mkdir tfad
    set r1 [catch {file copy tfa1 ~/tfa2 tfad}]
    set result [expr $r1 && [checkcontent tfad/tfa1 $s]]
    set env(HOME) $temp
    file delete -force tfa1 tfad
    set result
} {1}

test fCmd-14.3 {copyfile: stat failing on source} {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    set r1 [catch {file copy tfa1 tfa2}]
    expr $r1 && ![file exists tfa1] && ![file exists tfa2]
} {1}

test fCmd-14.4 {copyfile: error copying file to directory} {notRoot} {
    catch {file delete -force -- tfa tfad}
    set s1 [createfile tfa ]
    file mkdir tfad
    file mkdir tfad/tfa
    set r1 [catch {file copy tfa tfad}]
    set r2 [checkcontent tfa $s1]
    set r3 [file isdir tfad]
    set r4 [file isdir tfad/tfa]
    set result [expr $r1 && $r2 && $r3 && $r4 ]
    file delete -force tfa tfad
    set result
} {1}

 test fCmd-14.5 {copyfile: error copying directory to file} {notRoot} {
     catch {file delete -force -- tfa tfad}
     file mkdir tfa
     file mkdir tfad
     set s [createfile tfad/tfa]
     set r1 [catch {file copy tfa tfad}]
     set r2 [checkcontent tfad/tfa $s]
     set r3 [file isdir tfad]
     set r4 [file isdir tfa]
     set result [expr $r1 && $r2 && $r3 && $r4 ]
     file delete -force tfa tfad
     set result
} {1}

test fCmd-14.6 {copyfile: copy file succeeding} {notRoot} {
    catch {file delete -force -- tfa tfa2}
    set s [createfile tfa]
    file copy tfa tfa2
    set result [expr  [checkcontent tfa $s] && [checkcontent tfa2 $s]]
    file delete tfa tfa2
    set result
} {1}

test fCmd-14.7 {copyfile: copy directory succeeding} {notRoot} {
    catch {file delete -force -- tfa tfa2}
    file mkdir tfa
    set s [createfile tfa/file]
    file copy tfa tfa2
    set result [expr [checkcontent tfa/file $s] && [checkcontent tfa2/file $s]]
    file delete -force tfa tfa2
    set result
} {1}

test fCmd-14.8 {copyfile: copy directory failing} {unixOnly notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa/dir/a/b/c
    file attributes tfa/dir -permissions 0000
    set r1 [catch {file copy tfa tfa2}]
    file attributes tfa/dir -permissions 0777
    set result $r1
    file delete -force tfa tfa2
    set result
} {1}

#
# Coverage tests for TclMkdirCmd()
#
test fCmd-15.1 {TclMakeDirsCmd: target filename translation failing} {notRoot} {
    global env
    set temp $env(HOME)
    unset env(HOME) 
    set result [catch {file mkdir ~/tfa}]
    set env(HOME) $temp
    set result
} {1}
#
# Can Tcl_SplitPath return argc == 0? If so them we need a
# test for that code.
#
test fCmd-15.2 {TclMakeDirsCmd - one directory } {notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    set result [file isdirectory tfa]
    file delete tfa
    set result
} {1}

test fCmd-15.3 {TclMakeDirsCmd: - two directories} {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    file mkdir tfa1 tfa2
    set result [expr [file isdirectory tfa1] && [file isdirectory tfa2]]
    file delete tfa1 tfa2
    set result
} {1}

test fCmd-15.4 {TclMakeDirsCmd - stat failing} {unixOnly notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    createfile tfa/file
    file attributes tfa -permissions 0000
    set result [catch {file mkdir tfa/file}]
    file attributes tfa -permissions 0777
    file delete -force tfa
    set result
} {1}

test fCmd-15.5 {TclMakeDirsCmd: - making a directory several levels deep} \
	{notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa/a/b/c
    set result [file isdir tfa/a/b/c]
    file delete -force tfa
    set result
} {1}

    
test fCmd-15.6 {TclMakeDirsCmd: - trying to overwrite a file} {notRoot} {
    catch {file delete -force -- tfa}
    set s [createfile tfa]
    set r1 [catch {file mkdir tfa}]
    set r2 [file isdir tfa]
    set r3 [file exists tfa]
    set result [expr $r1 && !$r2 && $r3 && [checkcontent tfa $s]]
    file delete tfa
    set result
} {1}

test fCmd-15.7 {TclMakeDirsCmd - making several directories} {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    file mkdir tfa1 tfa2/a/b/c
    set result [expr [file isdir tfa1] && [file isdir tfa2/a/b/c]]
    file delete -force tfa1 tfa2
    set result
} {1}

test fCmd-15.8 {TclFileMakeDirsCmd: trying to create an existing dir} {notRoot} {
    file mkdir tfa
    file mkdir tfa
    set result [file isdir tfa]
    file delete tfa
    set result
} {1}


# Coverage tests for TclDeleteFilesCommand()
test fCmd-16.1 {test the -- argument} {notRoot} {
    catch {file delete -force -- tfa}
    createfile tfa
    file delete -- tfa
    file exists tfa
} {0}

test fCmd-16.2 {test the -force and -- arguments} {notRoot} {
    catch {file delete -force -- tfa}
    createfile tfa
    file delete -force -- tfa
    file exists tfa
} {0}

test fCmd-16.3 {test bad option} {notRoot} {
    catch {file delete -force -- tfa}
    createfile tfa
    set result [catch {file delete -dog tfa}]
    file delete tfa
    set result
} {1}

test fCmd-16.4 {test not enough args} {notRoot} {
    catch {file delete}
} {1}

test fCmd-16.5 {test not enough args with options} {notRoot} {
    catch {file delete --}
} {1}

test fCmd-16.6 {delete: source filename translation failing} {notRoot} {
    global env
    set temp $env(HOME)
    unset env(HOME)
    set result [catch {file delete ~/tfa}]
    set env(HOME) $temp
    set result
} {1}

test fCmd-16.7 {remove a non-empty directory without -force } {notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    createfile tfa/a
    set result [catch  {file delete tfa }]
    file delete -force tfa
    set result
} {1}

test fCmd-16.8 {remove a normal file } {notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    createfile tfa/a
    set result [catch  {file delete tfa }]
    file delete -force tfa
    set result
} {1}

test fCmd-16.9 {error while deleting file } {unixOnly notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    createfile tfa/a
    file attributes tfa -permissions 0555
    set result [catch  {file delete tfa/a }]
    #######
    #######  If any directory in a tree that is being removed does not 
    #######  have write permission, the process will fail!
    #######  This is also the case with "rm -rf"
    #######
    file attributes tfa -permissions 0777
    file delete -force tfa
    set result
} {1}

test fCmd-16.10 {deleting multiple files} {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    createfile tfa1
    createfile tfa2
    file delete tfa1 tfa2
    expr ![file exists tfa1] && ![file exists tfa2]
} {1}

test fCmd-16.11 {TclFileDeleteCmd: removing a nonexistant file} {notRoot} {
    catch {file delete -force -- tfa}
    file delete tfa
    set result 1
} {1}

# More coverage tests for mkpath()
 test fCmd-17.1 {mkdir stat failing on target but not ENOENT} {unixOnly notRoot} {
     catch {file delete -force -- tfa1}
     file mkdir tfa1
     file attributes tfa1 -permissions 0555
     set result [catch {file mkdir tfa1/tfa2}]
     file attributes tfa1 -permissions 0777
     file delete -force tfa1
     set result
} {1}

test fCmd-17.2 {mkdir several levels deep - relative } {notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa/a/b
    set result [file isdir tfa/a/b ]
    file delete tfa/a/b tfa/a tfa
    set result
} {1}

test fCmd-17.3 {mkdir several levels deep - absolute } {notRoot} {
    catch {file delete -force -- tfa}
    set f [file join [pwd] tfa a ]
    file mkdir $f
    set result [file isdir $f ]
    file delete $f [file join [pwd] tfa]
    set result
} {1}

#
# Functionality tests for TclFileRenameCmd()
#

test fCmd-18.1 {TclFileRenameCmd: rename (first form) in the same directory} \
	{notRoot} {
    catch {file delete -force -- tfad}
    file mkdir tfad/dir
    cd tfad/dir
    set s [createfile foo ]
    file rename  foo bar
    file rename bar ./foo
    file rename ./foo bar
    file rename ./bar ./foo
    file rename foo ../dir/bar
    file rename ../dir/bar ./foo
    file rename ../../tfad/dir/foo ../../tfad/dir/bar
    file rename [file join [pwd] bar] foo
    file rename foo [file join [pwd] bar]
    set result [expr [checkcontent bar $s] && ![file exists foo]]
    cd ../..
    file delete -force tfad
    set result
} {1}

test fCmd-18.2 {TclFileRenameCmd: single dir to nonexistant} {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    file mkdir tfa1
    file rename tfa1 tfa2
    set result [expr [file exists tfa2] && ![file exists tfa1]]
    file delete tfa2
    set result
} {1}

test fCmd-18.3 {TclFileRenameCmd: mixed dirs and files into directory} {notRoot} {
    catch {file delete -force -- tfa1 tfad1 tfad2}
    set s [createfile tfa1 ]
    file mkdir tfad1 tfad2
    file rename tfa1 tfad1 tfad2
    set r1 [checkcontent  tfad2/tfa1 $s]
    set r2 [file isdir tfad2/tfad1]
    set result [expr $r1 && $r2 && ![file exists tfa1] && ![file exists tfad1]]
    file delete tfad2/tfa1
    file delete -force tfad2
    set result
} {1}

test fCmd-18.4 {TclFileRenameCmd: attempt to replace non-dir with dir} {notRoot} {
    catch {file delete -force -- tfa tfad}
    set s [createfile tfa ]
    file mkdir tfad
    set r1 [catch {file rename tfad tfa}]
    set r2 [checkcontent tfa $s]
    set r3 [file isdir tfad]
    set result [expr $r1 && $r2 && $r3 ]
    file delete tfa tfad
    set result
} {1}

test fCmd-18.5 {TclFileRenameCmd: attempt to replace dir with non-dir} {notRoot} {
    catch {file delete -force -- tfa tfad}
    set s [createfile tfa ]
    file mkdir tfad/tfa
    set r1 [catch {file rename tfa tfad}]
    set r2 [checkcontent tfa $s]
    set r3 [file isdir tfad/tfa]
    set result [expr $r1 && $r2 && $r3 ]
    file delete -force  tfa tfad
    set result
} {1}

#
# On Windows there is no easy way to determine if two files are the same
#
test fCmd-18.6 {TclFileRenameCmd: rename a file to itself} {macOrUnix notRoot} {
    catch {file delete -force -- tfa}
    set s [createfile tfa]
    set r1 [catch {file rename tfa tfa}]
    set result [expr $r1 && [checkcontent tfa $s]]
    file delete tfa
    set result
} {1}

test fCmd-18.7 {TclFileRenameCmd: rename dir on top of another empty dir w/o -force} \
	{notRoot} {
    catch {file delete -force -- tfa tfad}
    file mkdir tfa tfad/tfa
    set r1 [catch {file rename tfa tfad}]
    set result [expr $r1 && [file isdir tfa]]
    file delete -force tfa tfad
    set result
} {1}

test fCmd-18.8 {TclFileRenameCmd: rename dir on top of another empty dir w/ -force} \
	{notRoot} {
    catch {file delete -force -- tfa tfad}
    file mkdir tfa tfad/tfa
    file rename -force tfa tfad
    set result [expr ![file isdir tfa]]
    file delete -force tfad
    set result
} {1}

test fCmd-18.9 {TclFileRenameCmd: rename dir on top of a non-empty dir w/o -force} \
	{notRoot} {
    catch {file delete -force -- tfa tfad}
    file mkdir tfa tfad/tfa/file
    set r1 [catch {file rename tfa tfad}]
    set result [expr $r1 && [file isdir tfa] && [file isdir tfad/tfa/file]]
    file delete -force tfa tfad
    set result
} {1}

test fCmd-18.10 {TclFileRenameCmd: rename dir on top of a non-empty dir w/ -force} \
	{notRoot} {
    catch {file delete -force -- tfa tfad}
    file mkdir tfa tfad/tfa/file
    set r1 [catch {file rename -force tfa tfad}]
    set result [expr $r1 && [file isdir tfa] && [file isdir tfad/tfa/file]]
    file delete -force tfa tfad
    set result
} {1}

test fCmd-18.11 {TclFileRenameCmd: rename a non-existant file} {notRoot} {
    catch {file delete -force -- tfa1}
    set r1 [catch {file rename tfa1 tfa2}]
    set result [expr $r1 && ![file exists tfa1] && ![file exists tfa2]]
} {1}

test fCmd-18.12 {TclFileRenameCmd : rename a symbolic link to file} \
	{unixOnly notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfa3}
	
    set s [createfile tfa1]
    file link -symbolic tfa2 tfa1
    file rename tfa2 tfa3
    set t [file type tfa3]
    set result [expr {$t eq "link"}]
    file delete tfa1 tfa3
    set result
} {1}

test fCmd-18.13 {TclFileRenameCmd : rename a symbolic link to dir} \
	{unixOnly notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfa3}
	
    file mkdir tfa1
    file link -symbolic tfa2 tfa1
    file rename tfa2 tfa3
    set t [file type tfa3]
    set result [expr {$t eq "link"}]
    file delete tfa1 tfa3
    set result
} {1}

test fCmd-18.14 {TclFileRenameCmd : rename a path with sym link} \
	{unixOnly notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfa3}
	
    file mkdir tfa1/a/b/c/d
    file mkdir tfa2
    set f [file join [pwd] tfa1/a/b] 
    set f2 [file join [pwd] {tfa2/b alias}]
    file link -symbolic $f2 $f
    file rename {tfa2/b alias/c} tfa3
    set r1 [file isdir tfa3]
    set r2 [file exists tfa1/a/b/c]
    set result [expr $r1 && !$r2]
    file delete -force tfa1 tfa2 tfa3
    set result
} {1}

test fCmd-18.15 {TclFileRenameCmd : rename a file to a symlink dir} \
	{unixOnly notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfalink}
	
    file mkdir tfa1
    set s [createfile tfa2]
    file link -symbolic tfalink tfa1

    file rename tfa2 tfalink
    set result [checkcontent tfa1/tfa2 $s ]
    file delete -force tfa1 tfalink
    set result
} {1}

test fCmd-18.16 {TclFileRenameCmd: rename a dangling symlink} {unixOnly notRoot} {
    catch {file delete -force -- tfa1 tfalink}
	
    file mkdir tfa1
    file link -symbolic tfalink tfa1
    file delete tfa1 
    file rename tfalink tfa2
    set result [expr [string compare [file type tfa2] "link"] == 0]
    file delete tfa2
    set result
} {1}


#
# Coverage tests for TclUnixRmdir
#
test fCmd-19.1 {remove empty directory} {notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    file delete tfa
    file exists tfa
} {0}

test fCmd-19.2 {rmdir error besides EEXIST} {unixOnly notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    file mkdir tfa/a
    file attributes tfa -permissions 0555
    set result [catch {file delete tfa/a}]
    file attributes tfa -permissions 0777
    file delete -force tfa
    set result
} {1}

test fCmd-19.3 {recursive remove} {notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    file mkdir tfa/a
    file delete -force tfa
    file exists tfa
} {0}

#
# TclUnixDeleteFile and TraversalDelete are covered by tests from the 
# TclDeleteFilesCmd suite
#
#

#
# Coverage tests for TraverseUnixTree(), called from TclDeleteFilesCmd
#

test fCmd-20.1 {TraverseUnixTree : failure opening a subdirectory directory } \
	{unixOnly notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    file mkdir tfa/a
    file attributes tfa/a -permissions 0000
    set result [catch {file delete -force tfa}]
    file attributes tfa/a -permissions 0777
    file delete -force tfa
    set result
} {1}

test fCmd-20.2 {TraverseUnixTree : recursive delete of large directory: Bug 1034337} \
	{unix notRoot} {
    catch {file delete -force -- tfa}
    file mkdir tfa
    for {set i 1} {$i <= 200} {incr i} {createfile tfa/testfile_$i}
    set result [catch {file delete -force tfa} msg]
    while {[catch {file delete -force tfa}]} {}
    list $result $msg
} {0 {}}

#
# Feature testing for TclCopyFilesCmd
# 
test fCmd-21.1 {copy : single file to nonexistant } {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    set s [createfile tfa1]
    file copy tfa1 tfa2
    set result [expr [checkcontent tfa2 $s] && [checkcontent tfa1 $s]]
    file delete tfa1 tfa2
    set result
} {1}

test fCmd-21.2 {copy : single dir to nonexistant } {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    file mkdir tfa1
    file copy tfa1 tfa2
    set result [expr [file isdir tfa2] && [file isdir tfa1]]
    file delete tfa1 tfa2
    set result
} {1}

test fCmd-21.3 {copy : single file into directory  } {notRoot} {
    catch {file delete -force -- tfa1 tfad}
    set s [createfile tfa1]
    file mkdir tfad
    file copy tfa1 tfad
    set result [expr [checkcontent tfad/tfa1 $s] && [checkcontent tfa1 $s]]
    file delete -force tfa1 tfad
    set result
} {1}

test fCmd-21.4 {copy : more than one source and target is not a directory} \
	{notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfa3}
    createfile tfa1 
    createfile tfa2 
    createfile tfa3 
    set result [catch {file copy tfa1 tfa2 tfa3}]
    file delete tfa1 tfa2 tfa3
    set result
} {1}

test fCmd-21.5 {copy : multiple files into directory  } {notRoot} {
    catch {file delete -force -- tfa1 tfa2 tfad}
    set s1 [createfile tfa1 ]
    set s2 [createfile tfa2 ]
    file mkdir tfad
    file copy tfa1 tfa2 tfad
    set r1 [checkcontent tfad/tfa1 $s1]
    set r2 [checkcontent tfad/tfa2 $s2]
    set r3 [checkcontent tfa1 $s1]
    set r4 [checkcontent tfa2 $s2]
    set result [expr $r1 && $r2 && $r3 && $r4]
    file delete -force tfa1 tfa2 tfad
    set result
} {1}

test fCmd-21.6 {copy: mixed dirs and files into directory} \
	{notRoot notFileSharing} {
    catch {file delete -force -- tfa1 tfad1 tfad2}
    set s [createfile tfa1 ]
    file mkdir tfad1 tfad2
    file copy tfa1 tfad1 tfad2
    set r1 [checkcontent [file join tfad2 tfa1] $s]
    set r2 [file isdir [file join tfad2 tfad1]]
    set r3 [checkcontent tfa1 $s]
    set result [expr $r1 && $r2 && $r3 && [file isdir tfad1]]
    file delete -force tfa1 tfad1 tfad2
    set result
} {1}

test fCmd-21.7.1 {TclCopyFilesCmd: copy a dangling link} {unixOnly notRoot dontCopyLinks} {
    file mkdir tfad1
    file link -symbolic tfalink tfad1
    file delete tfad1
    set result [list [catch {file copy tfalink tfalink2} msg] $msg]
    file delete -force tfalink tfalink2 
    set result
} {1 {error copying "tfalink": the target of this link doesn't exist}}
test fCmd-21.7.2 {TclCopyFilesCmd: copy a dangling link} {unixOnly notRoot} {
    file mkdir tfad1
    file link -symbolic tfalink tfad1
    file delete tfad1
    file copy tfalink tfalink2
    set result [string match [file type tfalink2] link]
    file delete tfalink tfalink2 
    set result
} {1}

test fCmd-21.8.1 {TclCopyFilesCmd: copy a link } {unixOnly notRoot dontCopyLinks} {
    file mkdir tfad1
    file link -symbolic tfalink tfad1
    file copy tfalink tfalink2
    set r1 [file type tfalink]; # link
    set r2 [file type tfalink2]; # directory
    set r3 [file isdir tfad1]; # 1
    set result [expr {("$r1" == "link") && ("$r2" == "directory") && $r3}]
    file delete -force tfad1 tfalink tfalink2
    set result
} {1}
test fCmd-21.8.2 {TclCopyFilesCmd: copy a link } {unixOnly notRoot} {
    file mkdir tfad1
    file link -symbolic tfalink tfad1
    file copy tfalink tfalink2
    set r1 [file type tfalink]; # link
    set r2 [file type tfalink2]; # link
    set r3 [file isdir tfad1]; # 1
    set result [expr {("$r1" == "link") && ("$r2" == "link") && $r3}]
    file delete -force tfad1 tfalink tfalink2
    set result
} {1}

test fCmd-21.9 {TclCopyFilesCmd: copy dir with a link in it} {unixOnly notRoot} {
    file mkdir tfad1
    file link -symbolic tfad1/tfalink "[pwd]/tfad1"
    file copy tfad1 tfad2
    set result [string match [file type tfad2/tfalink] link]
    file delete -force tfad1 tfad2
    set result
} {1}

test fCmd-21.10 {TclFileCopyCmd: copy dir on top of another empty dir w/o -force} \
	{notRoot} {
    catch {file delete -force -- tfa tfad}
    file mkdir tfa [file join tfad tfa]
    set r1 [catch {file copy tfa tfad}]
    set result [expr $r1 && [file isdir tfa]]
    file delete -force tfa tfad
    set result
} {1}

test fCmd-21.11 {TclFileCopyCmd: copy dir on top of a dir w/o -force} {notRoot} {
    catch {file delete -force -- tfa tfad}
    file mkdir tfa [file join tfad tfa file]
    set r1 [catch {file copy tfa tfad}]
    set result [expr $r1 && [file isdir tfa] && [file isdir [file join tfad tfa file]]]
    file delete -force tfa tfad
    set result
} {1}

test fCmd-21.12 {TclFileCopyCmd: copy dir on top of a non-empty dir w/ -force} \
	{notRoot} {
    catch {file delete -force -- tfa tfad}
    file mkdir tfa [file join tfad tfa file]
    set r1 [catch {file copy -force tfa tfad}]
    set result [expr $r1 && [file isdir tfa] && [file isdir [file join tfad tfa file]]]
    file delete -force tfa tfad
    set result
} {1}

#
# Coverage testing for TclpRenameFile
#
test fCmd-22.1 {TclpRenameFile: rename and overwrite in a single dir} {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    set s [createfile tfa1]
    set s2 [createfile tfa2 q]
	
    set r1 [catch {rename tfa1 tfa2}]
    file rename -force tfa1 tfa2
    set result [expr $r1 && [checkcontent tfa2 $s]]
    file delete [glob tfa1 tfa2]
    set result
} {1}

test fCmd-22.2 {TclpRenameFile: attempt to overwrite itself} {macOrUnix notRoot} {
    catch {file delete -force -- tfa1}
    set s [createfile tfa1]	
    file rename -force tfa1 tfa1
    set result [checkcontent tfa1 $s]
    file delete tfa1 
    set result
} {1}

test fCmd-22.3 {TclpRenameFile: rename dir to existing dir} {notRoot} {
    catch {file delete -force -- d1 tfad}
    file mkdir d1 [file join tfad d1]
    set r1 [catch {file rename d1 tfad}]
    set result [expr $r1 && [file isdir d1] && [file isdir [file join tfad d1]]]
    file delete -force d1 tfad
    set result
} {1}

test fCmd-22.4 {TclpRenameFile: rename dir to dir several levels deep} {notRoot} {
    catch {file delete -force -- d1 tfad}
    file mkdir d1 [file join tfad a b c]
    file rename d1 [file join tfad a b c d1]
    set result [expr ![file isdir d1] && [file isdir [file join tfad a b c d1]]]
    file delete -force [glob d1 tfad]
    set result
} {1}


#
# TclMacCopyFile needs to be redone.
#
test fCmd-22.5 {TclMacCopyFile: copy and overwrite in a single dir} {notRoot} {
    catch {file delete -force -- tfa1 tfa2}
    set s [createfile tfa1]
    set s2 [createfile tfa2 q]

    set r1 [catch {file copy tfa1 tfa2}]
    file copy -force tfa1 tfa2
    set result [expr $r1 && [checkcontent tfa2 $s] && [checkcontent tfa1 $s]]
    file delete tfa1 tfa2
    set result
} {1}

#
# TclMacMkdir - basic cases are covered elsewhere.
# Error cases are not covered.
#

#
# TclMacRmdir
# Error cases are not covered.
#

test fCmd-23.1 {TclMacRmdir: trying to remove a nonempty directory} {notRoot} {
    catch {file delete -force -- tfad}
	
    file mkdir [file join tfad dir]
	
    set result [catch {file delete tfad}]
    file delete -force tfad 
    set result
} {1}

#
# TclMacDeleteFile	
# Error cases are not covered.
#
test fCmd-24.1 {TclMacDeleteFile: deleting a normal file} {notRoot} {
    catch {file delete -force -- tfa1}
	
    createfile tfa1
    file delete tfa1
    file exists tfa1
} {0}

#
# TclMacCopyDirectory
# Error cases are not covered.
#
test fCmd-25.1 {TclMacCopyDirectory: copying a normal directory} {notRoot notFileSharing} {
    catch {file delete -force -- tfad1 tfad2}
		
    file mkdir [file join tfad1 a b c]
    file copy tfad1 tfad2
    set result [expr [file isdir [file join tfad1 a b c]] && [file isdir [file join tfad2 a b c]]]
    file delete -force tfad1 tfad2
    set result
} {1}

test fCmd-25.2 {TclMacCopyDirectory: copying a short path normal directory} {notRoot notFileSharing} {
    catch {file delete -force -- tfad1 tfad2}
		
    file mkdir tfad1
    file copy tfad1 tfad2
    set result [expr [file isdir tfad1] && [file isdir tfad2]]
    file delete tfad1 tfad2
    set result
} {1}

test fCmd-25.3 {TclMacCopyDirectory: copying dirs between different dirs} {notRoot notFileSharing} {
    catch {file delete -force -- tfad1 tfad2}
		
    file mkdir [file join tfad1 x y z]
    file mkdir [file join tfad2 dir]
    file copy tfad1 [file join tfad2 dir]
    set result [expr [file isdir [file join tfad1 x y z]] && [file isdir [file join tfad2 dir tfad1 x y z]]]
    file delete -force tfad1 tfad2
    set result
} {1}

#
# Functionality tests for TclDeleteFilesCmd
#

test fCmd-26.1 {TclDeleteFilesCmd: delete symlink} {unixOnly notRoot} {
    catch {file delete -force -- tfad1 tfad2}
		
    file mkdir tfad1
    file link -symbolic tfalink tfad1
    file delete tfalink

    set r1 [file isdir tfad1]
    set r2 [file exists tfalink]
    
    set result [expr $r1 && !$r2]
    file delete tfad1
    set result
} {1}

test fCmd-26.2 {TclDeleteFilesCmd: delete dir with symlink} {unixOnly notRoot} {
    catch {file delete -force -- tfad1 tfad2}
		
    file mkdir tfad1
    file mkdir tfad2
    file link -symbolic [file join tfad2 link] tfad1
    file delete -force tfad2

    set r1 [file isdir tfad1]
    set r2 [file exists tfad2]
    
    set result [expr $r1 && !$r2]
    file delete tfad1
    set result
} {1}

test fCmd-26.3 {TclDeleteFilesCmd: delete dangling symlink} {unixOnly notRoot} {
    catch {file delete -force -- tfad1 tfad2}
		
    file mkdir tfad1
    file link -symbolic tfad2 tfad1
    file delete tfad1
    file delete tfad2

    set r1 [file exists tfad1]
    set r2 [file exists tfad2]
    
    set result [expr !$r1 && !$r2]
    set result
} {1}

test fCmd-27.2 {TclFileAttrsCmd - Tcl_TranslateFileName fails} {testsetplatform} {
    set platform [testgetplatform]
    testsetplatform unix
    list [catch {file attributes ~_totally_bogus_user} msg] $msg [testsetplatform $platform]
} {1 {user "_totally_bogus_user" doesn't exist} {}}
test fCmd-27.3 {TclFileAttrsCmd - all attributes} {
    catch {file delete -force -- foo.tmp}
    createfile foo.tmp
    list [catch {file attributes foo.tmp} msg] [expr {[llength $msg] > 0}] [file delete -force -- foo.tmp]
} {0 1 {}}
test fCmd-27.4 {TclFileAttrsCmd - getting one option} {
    catch {file delete -force -- foo.tmp}
    createfile foo.tmp
    set attrs [file attributes foo.tmp]
    list [catch {eval file attributes foo.tmp [lindex $attrs 0]}] [file delete -force -- foo.tmp]
} {0 {}}

# Find a group that exists on this Unix system, or else skip tests that
# require Unix groups.
if {$tcl_platform(platform) == "unix"} {
    ::tcltest::testConstraint foundGroup 0
    catch {
	set groupList [exec groups]
	set group [lindex $groupList 0]
	::tcltest::testConstraint foundGroup 1
    }
} else {
    ::tcltest::testConstraint foundGroup 1
}

test fCmd-27.5 {TclFileAttrsCmd - setting one option} {foundGroup} {
    catch {file delete -force -- foo.tmp}
    createfile foo.tmp
    set attrs [file attributes foo.tmp]
    list [catch {eval file attributes foo.tmp [lrange $attrs 0 1]} msg] $msg [file delete -force -- foo.tmp]
} {0 {} {}}
test fCmd-27.6 {TclFileAttrsCmd - setting more than one option} {foundGroup} {
    catch {file delete -force -- foo.tmp}
    createfile foo.tmp
    set attrs [file attributes foo.tmp]
    list [catch {eval file attributes foo.tmp [lrange $attrs 0 3]} msg] $msg [file delete -force -- foo.tmp]
} {0 {} {}}

if {[string equal $tcl_platform(platform) "windows"]} {
    if {[string index $tcl_platform(osVersion) 0] >= 5 \
      && ([lindex [file system [temporaryDirectory]] 1] == "NTFS")} {
	tcltest::testConstraint linkDirectory 1
	tcltest::testConstraint linkFile 1
    } else {
	tcltest::testConstraint linkDirectory 0
	tcltest::testConstraint linkFile 0
    }
} else {
    tcltest::testConstraint linkFile 1
    tcltest::testConstraint linkDirectory 1
}

test fCmd-28.1 {file link} {
    list [catch {file link} msg] $msg
} {1 {wrong # args: should be "file link ?-linktype? linkname ?target?"}}

test fCmd-28.2 {file link} {
    list [catch {file link a b c d} msg] $msg
} {1 {wrong # args: should be "file link ?-linktype? linkname ?target?"}}

test fCmd-28.3 {file link} {
    list [catch {file link abc b c} msg] $msg
} {1 {bad switch "abc": must be -symbolic or -hard}}

test fCmd-28.4 {file link} {
    list [catch {file link -abc b c} msg] $msg
} {1 {bad switch "-abc": must be -symbolic or -hard}}
cd [workingDirectory]

makeDirectory abc.dir
makeDirectory abc2.dir
makeFile contents abc.file
makeFile contents abc2.file

cd [temporaryDirectory]
test fCmd-28.5 {file link: source already exists} {linkDirectory} {
    cd [temporaryDirectory]
    set res [list [catch {file link abc.dir abc2.dir} msg] $msg]
    cd [workingDirectory]
    set res
} {1 {could not create new link "abc.dir": that path already exists}}

test fCmd-28.6 {file link: unsupported operation} {linkDirectory macOrWin} {
    cd [temporaryDirectory]
    set res [list [catch {file link -hard abc.link abc.dir} msg] $msg]
    cd [workingDirectory]
    set res
} {1 {could not create new link "abc.link" pointing to "abc.dir": illegal operation on a directory}}

test fCmd-28.7 {file link: source already exists} {linkFile} {
    cd [temporaryDirectory]
    set res [list [catch {file link abc.file abc2.file} msg] $msg]
    cd [workingDirectory]
    set res
} {1 {could not create new link "abc.file": that path already exists}}

test fCmd-28.8 {file link} {linkFile winOnly} {
    cd [temporaryDirectory]
    set res [list [catch {file link -symbolic abc.link abc.file} msg] $msg]
    cd [workingDirectory]
    set res
} {1 {could not create new link "abc.link" pointing to "abc.file": not a directory}}

test fCmd-28.9 {file link: success with file} {linkFile} {
    cd [temporaryDirectory]
    file delete -force abc.link
    set res [list [catch {file link abc.link abc.file} msg] $msg]
    cd [workingDirectory]
    set res
} {0 abc.file}

cd [temporaryDirectory]
catch {file delete -force abc.link}
cd [workingDirectory]

test fCmd-28.10 {file link: linking to nonexistent path} {linkDirectory} {
    cd [temporaryDirectory]
    file delete -force abc.link
    set res [list [catch {file link abc.link abc2.doesnt} msg] $msg]
    cd [workingDirectory]
    set res
} {1 {could not create new link "abc.link" since target "abc2.doesnt" doesn't exist}}

test fCmd-28.11 {file link: success with directory} {linkDirectory} {
    cd [temporaryDirectory]
    file delete -force abc.link
    set res [list [catch {file link abc.link abc.dir} msg] $msg]
    cd [workingDirectory]
    set res
} {0 abc.dir}

test fCmd-28.12 {file link: cd into a link} {linkDirectory} {
    cd [temporaryDirectory]
    file delete -force abc.link
    file link abc.link abc.dir
    set orig [pwd]
    cd abc.link
    set dir [pwd]
    cd ..
    set up [pwd]
    cd $orig
    # now '$up' should be either $orig or [file dirname abc.dir],
    # depending on whether 'cd' actually moves to the destination
    # of a link, or simply treats the link as a directory.
    # (on windows the former, on unix the latter, I believe)
    if {([file normalize $up] != [file normalize $orig]) \
      && ([file normalize $up] != [file normalize [file dirname abc.dir]])} {
	set res "wrong directory with 'cd $link ; cd ..'"
    } else {
	set res "ok"
    }
    cd [workingDirectory]
    set res
} {ok}

test fCmd-28.13 {file link} {linkDirectory} {
    # duplicate link throws error
    cd [temporaryDirectory]
    set res [list [catch {file link abc.link abc.dir} msg] $msg]
    cd [workingDirectory]
    set res
} {1 {could not create new link "abc.link": that path already exists}}

test fCmd-28.14 {file link: deletes link not dir} {linkDirectory} {
    cd [temporaryDirectory]
    file delete -force abc.link
    set res [list [file exists abc.link] [file exists abc.dir]]
    cd [workingDirectory]
    set res
} {0 1}

test fCmd-28.15.1 {file link: copies link not dir} {linkDirectory dontCopyLinks} {
    cd [temporaryDirectory]
    file delete -force abc.link
    file link abc.link abc.dir
    file copy abc.link abc2.link
    # abc2.linkdir was a copy of a link to a dir, so it should end up as
    # a directory, not a link (links trace to endpoint).
    set res [list [file type abc2.link] [file tail [file link abc.link]]]
    cd [workingDirectory]
    set res
} {directory abc.dir}
test fCmd-28.15.2 {file link: copies link not dir} {linkDirectory} {
    cd [temporaryDirectory]
    file delete -force abc.link
    file link abc.link abc.dir
    file copy abc.link abc2.link
    set res [list [file type abc2.link] [file tail [file link abc2.link]]]
    cd [workingDirectory]
    set res
} {link abc.dir}

cd [temporaryDirectory]
file delete -force abc.link
file delete -force abc2.link

file copy abc.file abc.dir
file copy abc2.file abc.dir
cd [workingDirectory]

test fCmd-28.16 {file link: glob inside link} {linkDirectory} {
    cd [temporaryDirectory]
    file delete -force abc.link
    file link abc.link abc.dir
    set res [lsort [glob -dir abc.link -tails *]]
    cd [workingDirectory]
    set res
} [lsort [list abc.file abc2.file]]

test fCmd-28.17 {file link: glob -type l} {linkDirectory} {
    cd [temporaryDirectory]
    set res [glob -dir [pwd] -type l -tails abc*]
    cd [workingDirectory]
    set res
} {abc.link}

test fCmd-28.18 {file link: glob -type d} {linkDirectory} {
    cd [temporaryDirectory]
    set res [lsort [glob -dir [pwd] -type d -tails abc*]]
    cd [workingDirectory]
    set res
} [lsort [list abc.link abc.dir abc2.dir]]

test fCmd-29.1 {weird memory corruption fault} {
    catch {set res [open [file join ~a_totally_bogus_user_id/foo bar]]}
} 1

cd [temporaryDirectory]
file delete -force abc.link
cd [workingDirectory]

removeFile abc2.file
removeFile abc.file
removeDirectory abc2.dir
removeDirectory abc.dir

# cleanup
cleanup
::tcltest::cleanupTests
return
