/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.trace.commons;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class EclipseUtil {
  public static class NoSourceFolderException extends Exception {
    public NoSourceFolderException(final JavaModelException exception) {
      super(exception);
    }

    public NoSourceFolderException() {
      super();
    }
  }

  public static List<IFolder> findSrcFoldersOf(final IJavaProject p) throws EclipseUtil.NoSourceFolderException {
    final IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
    final Function1<IClasspathEntry, IFolder> _function = (IClasspathEntry e) -> {
      return root.getFolder(e.getPath());
    };
    return ListExtensions.<IClasspathEntry, IFolder>map(EclipseUtil.findClassPathEntriesOf(p, IClasspathEntry.CPE_SOURCE), _function);
  }

  public static List<IClasspathEntry> findClassPathEntriesOf(final IJavaProject p, final int type) {
    try {
      final List<IClasspathEntry> res = new ArrayList<IClasspathEntry>();
      final IClasspathEntry[] entries = p.getResolvedClasspath(true);
      for (int i = 0; (i < entries.length); i++) {
        {
          final IClasspathEntry entry = entries[i];
          int _entryKind = entry.getEntryKind();
          boolean _equals = (_entryKind == IClasspathEntry.CPE_SOURCE);
          if (_equals) {
            res.add(entry);
          }
        }
      }
      return res;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Taken from http://www.mkyong.com/java/how-to-copy-directory-in-java/
   */
  public static void copyFolder(final File src, final File dest) throws FileNotFoundException, IOException {
    boolean _isDirectory = src.isDirectory();
    if (_isDirectory) {
      boolean _exists = dest.exists();
      boolean _not = (!_exists);
      if (_not) {
        dest.mkdir();
      }
      final String[] files = src.list();
      for (final String file : files) {
        {
          final File srcFile = new File(src, file);
          final File destFile = new File(dest, file);
          EclipseUtil.copyFolder(srcFile, destFile);
        }
      }
    } else {
      final InputStream in = new FileInputStream(src);
      final OutputStream out = new FileOutputStream(dest);
      final byte[] buffer = new byte[1024];
      int length = 0;
      while (((length = in.read(buffer)) > 0)) {
        out.write(buffer, 0, length);
      }
      in.close();
      out.close();
    }
  }

  public static Set<IResource> findAllFilesOf(final IFolder f) {
    final HashSet<IResource> result = new HashSet<IResource>();
    try {
      IResource[] _members = f.members();
      for (final IResource r : _members) {
        if ((r instanceof IFile)) {
          result.add(r);
        } else {
          if ((r instanceof IFolder)) {
            result.addAll(EclipseUtil.findAllFilesOf(((IFolder)r)));
          }
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof CoreException) {
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return result;
  }

  public static void error(final String msg, final String pluginID, final Throwable e) throws CoreException {
    Status _status = new Status(Status.ERROR, pluginID, msg, e);
    throw new CoreException(_status);
  }

  public static Set<EPackage> findAllEPackagesIn(final Set<IContainer> containers) {
    try {
      final Set<EPackage> inputMetamodel = new HashSet<EPackage>();
      final ResourceSetImpl rs = new ResourceSetImpl();
      for (final IContainer container : containers) {
        final IResourceVisitor _function = (IResource r) -> {
          try {
            if ((r instanceof IFile)) {
              boolean _endsWith = ((IFile)r).getName().toLowerCase().endsWith(".ecore");
              if (_endsWith) {
                final URI uri = URI.createPlatformResourceURI(((IFile)r).getFullPath().toString(), true);
                final Resource model = EMFUtil.loadModelURI(uri, rs);
                final Set<EPackage> result = new HashSet<EPackage>();
                EList<EObject> _contents = model.getContents();
                for (final EObject c : _contents) {
                  if ((c instanceof EPackage)) {
                    result.add(((EPackage) c));
                  }
                }
                inputMetamodel.addAll(result);
              }
            }
            return true;
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        };
        container.accept(_function);
      }
      return inputMetamodel;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
