"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const Observable_1 = require("rxjs/Observable");
const defer_1 = require("rxjs/observable/defer");
const empty_1 = require("rxjs/observable/empty");
const from_1 = require("rxjs/observable/from");
const of_1 = require("rxjs/observable/of");
const operators_1 = require("rxjs/operators");
const exception_1 = require("../exception/exception");
const action_1 = require("../tree/action");
const Noop = function () { };
class SimpleSinkBase {
    constructor() {
        this.preCommitAction = Noop;
        this.postCommitAction = Noop;
        this.preCommit = Noop;
        this.postCommit = Noop;
    }
    _fileAlreadyExistException(path) {
        throw new exception_1.FileAlreadyExistException(path);
    }
    _fileDoesNotExistException(path) {
        throw new exception_1.FileDoesNotExistException(path);
    }
    _validateOverwriteAction(action) {
        return this._validateFileExists(action.path)
            .pipe(operators_1.map(b => { if (!b) {
            this._fileDoesNotExistException(action.path);
        } }));
    }
    _validateCreateAction(action) {
        return this._validateFileExists(action.path)
            .pipe(operators_1.map(b => { if (b) {
            this._fileAlreadyExistException(action.path);
        } }));
    }
    _validateRenameAction(action) {
        return this._validateFileExists(action.path).pipe(operators_1.map(b => { if (!b) {
            this._fileDoesNotExistException(action.path);
        } }), operators_1.mergeMap(() => this._validateFileExists(action.to)), operators_1.map(b => { if (b) {
            this._fileAlreadyExistException(action.to);
        } }));
    }
    _validateDeleteAction(action) {
        return this._validateFileExists(action.path)
            .pipe(operators_1.map(b => { if (!b) {
            this._fileDoesNotExistException(action.path);
        } }));
    }
    validateSingleAction(action) {
        switch (action.kind) {
            case 'o': return this._validateOverwriteAction(action);
            case 'c': return this._validateCreateAction(action);
            case 'r': return this._validateRenameAction(action);
            case 'd': return this._validateDeleteAction(action);
            default: throw new action_1.UnknownActionException(action);
        }
    }
    commitSingleAction(action) {
        return empty_1.empty().pipe(operators_1.concat(new Observable_1.Observable(observer => {
            return this.validateSingleAction(action).subscribe(observer);
        })), operators_1.concat(new Observable_1.Observable(observer => {
            let committed = null;
            switch (action.kind) {
                case 'o':
                    committed = this._overwriteFile(action.path, action.content);
                    break;
                case 'c':
                    committed = this._createFile(action.path, action.content);
                    break;
                case 'r':
                    committed = this._renameFile(action.path, action.to);
                    break;
                case 'd':
                    committed = this._deleteFile(action.path);
                    break;
            }
            if (committed) {
                committed.subscribe(observer);
            }
            else {
                observer.complete();
            }
        })));
    }
    commit(tree) {
        const actions = from_1.from(tree.actions);
        return (this.preCommit() || empty_1.empty()).pipe(operators_1.concat(defer_1.defer(() => actions)), operators_1.concatMap((action) => {
            const maybeAction = this.preCommitAction(action);
            if (!maybeAction) {
                return of_1.of(action);
            }
            else if (action_1.isAction(maybeAction)) {
                return of_1.of(maybeAction);
            }
            else {
                return maybeAction;
            }
        }), operators_1.concatMap((action) => {
            return this.commitSingleAction(action).pipe(operators_1.ignoreElements(), operators_1.concat([action]));
        }), operators_1.concatMap((action) => this.postCommitAction(action) || empty_1.empty()), operators_1.concat(defer_1.defer(() => this._done())), operators_1.concat(defer_1.defer(() => this.postCommit() || empty_1.empty())));
    }
}
exports.SimpleSinkBase = SimpleSinkBase;
//# sourceMappingURL=data:application/json;base64,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