/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#include "mod_ftp.h"
#include "ftp_internal.h"

#define FTP_SERVER_STRING "mod_ftp/" MODFTP_VERSION

static ap_filter_rec_t *ftp_crlf_filter_handle;
static ap_filter_rec_t *ftp_data_out_filter_handle;
static ap_filter_rec_t *ftp_byterange_filter_handle;
ap_filter_rec_t *ftp_input_filter_handle;
ap_filter_rec_t *ftp_content_length_filter_handle;
ap_filter_rec_t *ftp_ssl_input_filter_handle;
ap_filter_rec_t *ftp_ssl_output_filter_handle;

/* Register callbacks for mod_log_config. */
static int ftp_pre_config(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptemp)
{
    APR_OPTIONAL_FN_TYPE(ap_register_log_handler) * log_pfn_register;

    /*
     * Find the handles of non-ftp filters we will be using
     * 
     * This must be done after register_hooks since SSL may not be loaded before
     * FTP, but before we actually process the config because we test those
     * filter handles to determine if SSL-related directives are valid.
     */
    ftp_byterange_filter_handle
        = ap_get_output_filter_handle("BYTERANGE");
    ftp_content_length_filter_handle
        = ap_get_output_filter_handle("CONTENT_LENGTH");
    ftp_ssl_input_filter_handle
        = ap_get_input_filter_handle(FTP_SSL_FILTER);
    ftp_ssl_output_filter_handle
        = ap_get_output_filter_handle(FTP_SSL_FILTER);

    /*
     * Register our custom log format flags
     */
    log_pfn_register = APR_RETRIEVE_OPTIONAL_FN(ap_register_log_handler);

    if (log_pfn_register) {
        log_pfn_register(p, "M", ftp_log_transfer_mode, 0);
        log_pfn_register(p, "F", ftp_log_action_flags, 0);
        log_pfn_register(p, "d", ftp_log_transfer_direction, 0);
        log_pfn_register(p, "W", ftp_log_accessed_anonymously, 0);
        log_pfn_register(p, "S", ftp_log_service_name, 0);
        log_pfn_register(p, "Z", ftp_log_auth_method, 0);
        log_pfn_register(p, "Y", ftp_log_auth_user_id, 0);
    }

#ifdef HAVE_FTP_LOWPORTD
    return lowportd_pre_config(p, plog, ptemp);
#else
    return OK;
#endif
}


static int ftp_post_config(apr_pool_t *p, apr_pool_t *plog,
                               apr_pool_t *ptemp, server_rec *s)
{
    server_rec *base = s;
    ftp_server_config *basefsc = ftp_get_module_config(s->module_config);
    int lowportd = 0;

    ap_add_version_component(p, FTP_SERVER_STRING);

    /*
     * Unless disabled, advertise that UTF8 filenames are preferred/permitted
     * RFC2640 never -requires- UTF8 names
     */
    if (!(basefsc->options & FTP_OPT_NO_UTF8_FEAT))
        ftp_feat_advert("UTF8");

    /* Finalize ftp_cmd_help and ftp_cmd_feat messages */
    ftp_cmd_finalize(p, ptemp);

    /*
     * Fixup global values, then base server and virtual host values
     */
    if (!basefsc->limitdbfile)
        basefsc->limitdbfile = ap_server_root_relative(p, FTP_DEFAULT_DBFILE);

    for (; s; s = s->next) {
        ftp_server_config *fsc = ftp_get_module_config(s->module_config);

        if (fsc->timeout_login == FTP_UNSPEC)
            fsc->timeout_login = FTP_TIMEOUT_LOGIN;
        if (fsc->timeout_idle == FTP_UNSPEC)
            fsc->timeout_idle = FTP_TIMEOUT_IDLE;
        if (fsc->timeout_data == FTP_UNSPEC)
            fsc->timeout_data = FTP_TIMEOUT_DATA;

        if (fsc->max_login_attempts == FTP_UNSPEC)
            fsc->max_login_attempts = FTP_MAX_LOGINS;

        if (fsc->active_min == FTP_UNSPEC)
            fsc->active_min = fsc->active_max = -1;
        else if (fsc->active_min < 1024)
            lowportd = 1;

        if (fsc->pasv_min == FTP_UNSPEC)
            fsc->pasv_min = fsc->pasv_max = 0;
        if (fsc->epsv_ignore_family == FTP_UNSPEC)
            fsc->epsv_ignore_family = 0;

        if (fsc->data_block_size == FTP_UNSPEC)
            fsc->data_block_size = FTP_DATA_BLOCK_SIZE;

        if (fsc->limit_peruser == FTP_UNSPEC)
            fsc->limit_peruser = 0;
        if (fsc->limit_perip == FTP_UNSPEC)
            fsc->limit_perip = 0;
        if (fsc->limit_perserver == FTP_UNSPEC)
            fsc->limit_perserver = 0;

        fsc->limitdbfile = basefsc->limitdbfile;
    }

    if (ftp_mutexdb_init(base, p) != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_CRIT, 0, base,
                     "Could not initialize FTP mutex");
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    apr_pool_cleanup_register(p, base, ftp_mutexdb_cleanup,
                              apr_pool_cleanup_null);

#ifdef HAVE_FTP_LOWPORTD
    if (lowportd)
        /* Initialized only if a server has at least one active_min < 1024 */
        return lowportd_post_config(p, plog, ptemp, base);
    else
#endif
        return OK;
}

static void ftp_child_init(apr_pool_t *p, server_rec *s)
{
    ftp_mutexdb_child_init(s, p);
}

static void *create_ftp_server_config(apr_pool_t *p, server_rec *s)
{
    ftp_server_config *fsc = apr_pcalloc(p, sizeof(*fsc));

    fsc->timeout_login = FTP_UNSPEC;
    fsc->timeout_idle = FTP_UNSPEC;
    fsc->timeout_data = FTP_UNSPEC;
    fsc->max_login_attempts = FTP_UNSPEC;

    fsc->active_min = FTP_UNSPEC;
    fsc->active_max = FTP_UNSPEC;
    fsc->pasv_min = FTP_UNSPEC;
    fsc->pasv_max = FTP_UNSPEC;
    fsc->epsv_ignore_family = FTP_UNSPEC;

    fsc->data_block_size = FTP_UNSPEC;

    fsc->limit_peruser = FTP_UNSPEC;
    fsc->limit_perip = FTP_UNSPEC;
    fsc->limit_perserver = FTP_UNSPEC;

    return fsc;
}

static void *merge_ftp_server_config(apr_pool_t *p, void *basev, void *addv)
{
    ftp_server_config *base = (ftp_server_config *) basev;
    ftp_server_config *add = (ftp_server_config *) addv;
    ftp_server_config *fsc = apr_palloc(p, sizeof(*fsc));

    /*
     * We default to the add config, so any directive not handled here won't
     * be inherited from the base server.
     */
    memcpy(fsc, add, sizeof(*fsc));

    if (fsc->timeout_login == FTP_UNSPEC)
        fsc->timeout_login = base->timeout_login;
    if (fsc->timeout_idle == FTP_UNSPEC)
        fsc->timeout_idle = base->timeout_idle;
    if (fsc->timeout_data == FTP_UNSPEC)
        fsc->timeout_data = base->timeout_data;

    if (fsc->max_login_attempts == FTP_UNSPEC)
        fsc->max_login_attempts = FTP_MAX_LOGINS;

    if (fsc->active_min == FTP_UNSPEC) {
        fsc->active_min = base->active_min;
        fsc->active_max = base->active_max;
    }
    if (fsc->pasv_min == FTP_UNSPEC) {
        fsc->pasv_min = base->pasv_min;
        fsc->pasv_max = base->pasv_max;
    }
    if (fsc->epsv_ignore_family == FTP_UNSPEC) {
        fsc->epsv_ignore_family = base->epsv_ignore_family;
    }

    if (fsc->data_block_size == FTP_UNSPEC) {
        fsc->data_block_size = base->data_block_size;
    }

    if (fsc->limit_peruser == FTP_UNSPEC)
        fsc->limit_peruser = base->limit_peruser;
    if (fsc->limit_perip == FTP_UNSPEC)
        fsc->limit_perip = base->limit_perip;
    if (fsc->limit_perserver == FTP_UNSPEC)
        fsc->limit_perserver = base->limit_perserver;

    if (!fsc->banner_message) {
        fsc->banner_message = base->banner_message;
        fsc->banner_message_isfile = base->banner_message_isfile;
    }
    if (!fsc->exit_message) {
        fsc->exit_message = base->exit_message;
        fsc->exit_message_isfile = base->exit_message_isfile;
    }

    return fsc;
}


static void *create_ftp_dir_config(apr_pool_t *p, char *dir)
{
    ftp_dir_config *conf = apr_pcalloc(p, sizeof(*conf));

    conf->fileperms = APR_OS_DEFAULT;
    conf->dirperms = APR_OS_DEFAULT;

    return conf;
}

static void *merge_ftp_dir_config(apr_pool_t *p, void *basev, void *addv)
{
    ftp_dir_config *base = (ftp_dir_config *) basev;
    ftp_dir_config *add = (ftp_dir_config *) addv;
    ftp_dir_config *conf = apr_palloc(p, sizeof(*conf));

    /* XXX: NO-INHERIT the FTPReadmeMessage? */
    conf->path = add->path;
    conf->readme = add->readme;
    conf->readme_isfile = add->readme_isfile;

    conf->fileperms = (add->fileperms == APR_OS_DEFAULT)
        ? base->fileperms : add->fileperms;
    conf->dirperms = (add->dirperms == APR_OS_DEFAULT)
        ? base->dirperms : add->dirperms;

    return conf;
}

int ftp_have_ssl(void)
{
    return (ftp_ssl_input_filter_handle
            && ftp_ssl_output_filter_handle);
}

static void ftp_insert_filter(request_rec *r)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    if (!fc)
        return;

    if (fc->datasock && (fc->filter_mask & FTP_NEED_DATA_OUT)) {
        ap_add_output_filter_handle(ftp_data_out_filter_handle, fc, r,
                                    r->connection);
    }
    if (fc->filter_mask & FTP_NEED_BYTERANGE) {
        ap_add_output_filter_handle(ftp_byterange_filter_handle, NULL,
                                    r, r->connection);
    }
    if (fc->filter_mask & FTP_NEED_CRLF) {
        /* CRLF filter has a single int context, "did we last have a CR?" */
        ap_add_output_filter_handle(ftp_crlf_filter_handle,
                                    apr_pcalloc(r->pool, sizeof(int *)),
                                    r, r->connection);
    }
}

/*
 * The create request hook.  Used when Apache does an internal redirect
 *
 * XXX: because we are connection-oriented, perhaps this can go away?
 *      left in place until we finish any request_rec refactoring
 */
static int ftp_create_request(request_rec *r)
{
    return OK;
}

#if ((AP_SERVER_MAJORVERSION_NUMBER < 3) && (AP_SERVER_MINORVERSION_NUMBER < 3))

/* This is something of a hack.  The idea is that we want to be able to look
 * at a string with a '*' wildcard, and a string that matches the pattern,
 * and determine which part of the second string matches the wildcard.  This
 * allows us to extract that information, which is later used to setup
 * the require line properly if we are ensuring that users can not switch
 * to other user's home directories.
 */
static char *find_dir(apr_pool_t *p, const char *wildcard,
                           const char *curr_file)
{
    char *dirname;
    size_t i;

    for (i = 0; i < strlen(wildcard); i++) {
        if (wildcard[i] == curr_file[i]) {
            /* So far the strings are equal */
            continue;
        }
        if (wildcard[i] == '*' && wildcard[i + 1] == '/') {
            /*
             * Now we have the portion we want to match against.  Find the
             * corresponding string in the current file.
             */
            size_t j = i;
            while (curr_file[j] && curr_file[j] != '/') {
                j++;
            }
            dirname = apr_pstrmemdup(p, curr_file + i, j - i);
            return dirname;
        }
        return NULL;
    }
    return NULL;
}

/*
 * The check_user_id hook.  This is a hack that allows us to ensure that only
 * the owner of the directory can get access to the current directory.  If the
 * user has configured:
 *         Require dir-name
 * for the current directory, then we require that the logged in user match
 * the name of the directory.
 */
static int ftp_check_user_id(request_rec *r)
{
    apr_array_header_t *newrequire;
    const apr_array_header_t *reqs_arr = ap_requires(r);
    require_line *reqs;
    const char *t, *w;
    core_dir_config *conf;
    int x;

    conf = (core_dir_config *) ap_get_module_config(r->per_dir_config,
                                                    &core_module);

    if (!reqs_arr) {
        return DECLINED;
    }

    newrequire = apr_array_make(r->pool, 2, sizeof(require_line));

    reqs = (require_line *) reqs_arr->elts;
    for (x = 0; x < reqs_arr->nelts; x++) {
        require_line *rl;

        rl = (require_line *) apr_array_push(newrequire);
        rl->method_mask = reqs[x].method_mask;

        t = reqs[x].requirement;
        w = ap_getword_white(r->pool, &t);
        if (!strcmp(w, "user")) {
            rl->requirement = apr_pstrdup(r->pool, w);
            while (strcmp(w = ap_getword_white(r->pool, &t), "")) {
                if (!strcmp(w, "dir-name")) {
                    char *dirname = find_dir(r->pool, conf->d, r->filename);
                    rl->requirement = apr_pstrcat(r->pool, rl->requirement,
                                                  " ", dirname, NULL);
                }
                else {
                    rl->requirement = apr_pstrcat(r->pool, rl->requirement,
                                                  " ", w, NULL);
                }
            }
        }
        else {
            rl->requirement = apr_pstrdup(r->pool, reqs[x].requirement);
        }
    }
    conf->ap_requires = newrequire;

    return DECLINED;
}
#endif                          /* AP_VERSION > 2.2 */

/*
 * Handle FTP module directives
 */

/*
 * Generic function used for setting values in the server config.
 * Adapted from ap_set_int_slot, since it only works with per directory
 * configurations.
 */
static const char *ftp_set_int_slot(cmd_parms *cmd, void *dummy,
                                         const char *arg)
{
    char *endptr;
    char *error_str = NULL;
    int offset = (int) (long) cmd->info;
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    *(int *) ((char *) fsc + offset) = strtol(arg, &endptr, 10);

    if ((*arg == '\0') || (*endptr != '\0')) {
        error_str = apr_psprintf(cmd->pool,
                         "Invalid value for directive %s, expected integer",
                                 cmd->directive->directive);
    }

    return error_str;
}


static const char *ftp_enable(cmd_parms *cmd, void *dummy, int arg)
{
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    fsc->enabled = arg;

    return NULL;
}

static const char *ftp_umask(cmd_parms *cmd, void *dconf, const char *arg)
{
    int umask, mode;
    char *endp;
    char *error_str = NULL;

    ftp_dir_config *d = dconf;

    umask = strtol(arg, &endp, 8);
    mode = umask & 0666;

    if ((*arg == '\0') || (*endp != '\0')) {
        error_str = apr_psprintf(cmd->pool,
                                 "%s is not valid for %s", arg,
                                 cmd->directive->directive);
    }
    else {
        d->fileperms = ftp_unix_mode2perms(mode);
    }
    return error_str;
}


static const char *ftp_dirumask(cmd_parms *cmd, void *dconf, const char *arg)
{
    int umask, xmask, mode;
    char *endp;
    char *error_str = NULL;

    ftp_dir_config *d = dconf;

    umask = strtol(arg, &endp, 8);

    /*
     * Here the r bits  of umask are taken to xmask all other bits set to 1.
     */
    xmask = umask & 0444;
    xmask = xmask >> 2;
    xmask |= 0666;

    /*
     * Here r bits are & with x bits. Effectively it confirms that x
     * permission is given to all who are given r permission. but allow to
     * deny r while x is given.
     */
    mode = umask & xmask;


    if ((*arg == '\0') || (*endp != '\0')) {
        error_str = apr_psprintf(cmd->pool,
                                 "%s is not valid for %s", arg,
                                 cmd->directive->directive);
    }
    else {
        d->dirperms = ftp_unix_mode2perms(mode);
    }
    return error_str;
}

static const char *ftp_implicit_ssl(cmd_parms *cmd, void *dummy, int arg)
{
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    if (!ftp_have_ssl()) {
        return "No SSL module found, cannot enable implicit SSL";
    }

    fsc->implicit_ssl = arg;
    return NULL;
}

static const char *ftp_set_jail(cmd_parms *cmd, void *dummy, int arg)
{
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    fsc->jailuser = arg;
    return NULL;
}

static const char *ftp_options(cmd_parms *cmd, void *dummy,
                                    const char *raw)
{
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    while (raw[0]) {
        int opt_mask = 0;
        char *op = ap_getword_conf(cmd->pool, &raw);

        if (!strcasecmp(op, "RequireSSL")) {
            opt_mask = FTP_OPT_REQUIRESSL;
        }
        else if (!strcasecmp(op, "CheckMaxClients")) {
            opt_mask = FTP_OPT_CHECKMAXCLIENTS;
        }
        else if (!strcasecmp(op, "RemoveUserGroup")) {
            opt_mask = FTP_OPT_REMOVEUSERGROUP;
        }
        else if (!strcasecmp(op, "NLSTShowDirs")) {
            opt_mask = FTP_OPT_NLSTSHOWDIRS;
        }
        else if (!strcasecmp(op, "NLSTIsLIST")) {
            opt_mask = FTP_OPT_NLSTISLIST;
        }
        else if (!strcasecmp(op, "LISTIsNLST")) {
            opt_mask = FTP_OPT_LISTISNLST;
        }
        else if (!strcasecmp(op, "CreateHomeDirs")) {
            opt_mask = FTP_OPT_CREATEHOMEDIRS;
        }
        else if (!strcasecmp(op, "ShowUnAuthorizedFiles")) {
            opt_mask = FTP_OPT_SHOWUNAUTH;
        }
        else if (!strcasecmp(op, "AllowProxyPORT")) {
            opt_mask = FTP_OPT_ALLOWPROXYPORT;
        }
        else if (!strcasecmp(op, "AllowProxyPASV")) {
            opt_mask = FTP_OPT_ALLOWPROXYPASV;
        }
        else if (!strcasecmp(op, "VirtualHostByUser")) {
            opt_mask = FTP_OPT_VHOST_BY_USER;
        }
        else if (!strcasecmp(op, "StripHostname")) {
            opt_mask = FTP_OPT_STRIP_HOSTNAME;
        }
        else if (!strcasecmp(op, "NoUTF8Feature")) {
            opt_mask = FTP_OPT_NO_UTF8_FEAT;
        }
        else {
            return apr_pstrcat(cmd->pool, "Illegal FTPOption ", op, NULL);
        }
        fsc->options |= opt_mask;
    }
    if ((fsc->options & FTP_OPT_LISTISNLST) && (fsc->options & FTP_OPT_NLSTISLIST)) {
        return "LISTISNLST and NLSTISLIST are mutually exclusive options";
    }
    return NULL;
}

static const char *ftp_set_pasv_addr(cmd_parms *cmd, void *dummy,
                                          const char *addr)
{
    apr_uint32_t ipaddr;
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    if ((ftp_inet_pton(AF_INET, addr, &ipaddr)) != 1) {
        return apr_pstrcat(cmd->pool, "Invalid IP address for ",
                           cmd->directive->directive, " (", addr, ")", NULL);
    }

    fsc->pasv_addr = apr_pstrdup(cmd->pool, addr);

    return NULL;
}

static const char *ftp_set_pasv_bindaddr(cmd_parms *cmd, void *dummy,
                                              const char *addr)
{
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);
#if APR_HAVE_IPV6
    struct in6_addr ipaddr;
#else
    struct in_addr ipaddr;
#endif

    if ((ftp_inet_pton(AF_INET, addr, &ipaddr)) == 1) {
        fsc->pasv_bindfamily = APR_INET;
    }
#if APR_HAVE_IPV6
    else if ((ftp_inet_pton(AF_INET6, addr, &ipaddr)) == 1) {
        fsc->pasv_bindfamily = APR_INET6;
    }
#endif
    else {
        return apr_pstrcat(cmd->pool, "Invalid IP address for ",
                           cmd->directive->directive, " (", addr, ")", NULL);
    }

    fsc->pasv_bindaddr = apr_pstrdup(cmd->pool, addr);

    return NULL;
}

static const char *ftp_set_message_generic(cmd_parms *cmd, const char *arg,
                                          const char **dest, int *file_flag)
{
    apr_finfo_t finfo;
    apr_status_t rv;

    if (*dest != NULL) {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0,
                   cmd->server, "Ignoring duplicate message file: %s", arg);
        return NULL;
    }

    if (!strncmp(arg, "file:", 5)) {
        rv = apr_stat(&finfo,
                      ap_server_root_relative(cmd->temp_pool, arg + 5),
                      APR_FINFO_TYPE, cmd->temp_pool);
        if (rv != APR_SUCCESS || finfo.filetype != APR_REG) {
            return apr_pstrcat(cmd->pool, "Invalid message file: ",
                               arg + 5, NULL);
        }

        *(file_flag) = 1;
        *(dest) = ap_server_root_relative(cmd->pool, arg + 5);
    }
    else {
        *(dest) = apr_pstrdup(cmd->pool, arg);
    }

    return NULL;
}

static const char *ftp_set_banner_message(cmd_parms *cmd, void *dummy,
                                               const char *arg)
{
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    return ftp_set_message_generic(cmd, arg, &fsc->banner_message,
                                   &fsc->banner_message_isfile);
}

static const char *ftp_set_exit_message(cmd_parms *cmd, void *dummy,
                                             const char *arg)
{
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    return ftp_set_message_generic(cmd, arg, &fsc->exit_message,
                                   &fsc->exit_message_isfile);
}

static const char *ftp_set_readme_message(cmd_parms *cmd, void *dconf,
                                               const char *arg)
{
    ftp_dir_config *d = dconf;

    d->path = apr_pstrdup(cmd->pool, cmd->path);
    return ftp_set_message_generic(cmd, arg, &d->readme, &d->readme_isfile);
}

static const char *ftp_set_pasv_range(cmd_parms *cmd, void *dummy,
                                           const char *min, const char *max)
{
    char *error_str = NULL;

    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    /* XXX: better error handling */
    fsc->pasv_min = (apr_port_t) atoi(min);
    fsc->pasv_max = (apr_port_t) atoi(max);

    if (fsc->pasv_min > fsc->pasv_max
        || fsc->pasv_min < 0 || fsc->pasv_max > 0xFFFF) {
        error_str = apr_psprintf(cmd->pool,
                                 "Invalid range for %s (%s > %s)",
                                 cmd->directive->directive, min, max);
    }

    return error_str;
}

static const char *ftp_set_active_ports(cmd_parms *cmd, void *dummy,
                                           const char *min, const char *max)
{
    char *error_str = NULL;

    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    /* XXX: better error handling */
    fsc->active_min = atoi(min);
    if (!max) {
        fsc->active_max = fsc->active_min;
    }
    else {
        /* XXX: better error handling */
        fsc->active_max = atoi(max);
    }

    if (fsc->active_min > fsc->active_max
        || fsc->active_min < 0 || fsc->active_max > 0xFFFF) {
        error_str = apr_psprintf(cmd->pool,
                                 "Invalid range for %s (%s > %s)",
                                 cmd->directive->directive, min, max);
    }

    return error_str;
}

static const char *ftp_set_homedir(cmd_parms *cmd, void *dummy,
                                        const char *dir)
{
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    if (*dir != '/') {
        return apr_pstrcat(cmd->pool, "Path for ", cmd->directive->directive,
                           " must be absolute (", dir, ")", NULL);
    }

    fsc->homedir = apr_pstrdup(cmd->pool, dir);

    return NULL;
}

static const char *ftp_set_docrootenv(cmd_parms *cmd, void *dummy,
                                           const char *var)
{
    ftp_server_config *fsc =
    ftp_get_module_config(cmd->server->module_config);

    fsc->docrootenv = apr_pstrdup(cmd->pool, var);

    return NULL;
}

static const char *ftp_set_limit_peruser(cmd_parms *cmd, void *dummy,
                                              const char *limit)
{
    char *error_str = NULL;
    ftp_server_config *fsc;

    fsc = ftp_get_module_config(cmd->server->module_config);

    /* XXX: better error handling */
    fsc->limit_peruser = (apr_port_t) atoi(limit);

    if (fsc->limit_peruser < 0) {
        error_str = apr_psprintf(cmd->pool,
                                 "%s value must be 0 or greater (%s)",
                                 cmd->directive->directive, limit);
    }

    return error_str;
}

static const char *ftp_set_limit_perip(cmd_parms *cmd, void *dummy,
                                            const char *limit)
{
    char *error_str = NULL;
    ftp_server_config *fsc;

    fsc = ftp_get_module_config(cmd->server->module_config);

    /* XXX: better error handling */
    fsc->limit_perip = (apr_port_t) atoi(limit);

    if (fsc->limit_perip < 0) {
        error_str = apr_psprintf(cmd->pool,
                                 "%s value must be 0 or greater (%s)",
                                 cmd->directive->directive, limit);
    }

    return error_str;
}

static const char *ftp_set_limit_perserver(cmd_parms *cmd, void *dummy,
                                                const char *limit)
{
    char *error_str = NULL;
    ftp_server_config *fsc;

    fsc = ftp_get_module_config(cmd->server->module_config);

    /* XXX: better error handling */
    fsc->limit_perserver = (apr_port_t) atoi(limit);

    if (fsc->limit_perserver < 0) {
        error_str = apr_psprintf(cmd->pool,
                                 "%s value must be 0 or greater (%s)",
                                 cmd->directive->directive, limit);
    }

    return error_str;
}

static const char *ftp_set_dbfile(cmd_parms *cmd, void *dummy,
                                       const char *dbfile)
{
    ftp_server_config *fsc;

    const char *err = ap_check_cmd_context(cmd, GLOBAL_ONLY);
    if (err != NULL) {
        return err;
    }

    fsc = ftp_get_module_config(cmd->server->module_config);

    fsc->limitdbfile = ap_server_root_relative(cmd->pool, dbfile);
    if (!fsc->limitdbfile) {
        return apr_pstrcat(cmd->pool, "Invalid FTPLimitDBFile value: ",
                           dbfile, NULL);
    }

    return NULL;
}

static const char *ftp_epsv_ignore_family(cmd_parms *cmd, void *dummy, int flag)
{
    ftp_server_config *fsc = ftp_get_module_config(cmd->server->module_config);
    fsc->epsv_ignore_family = flag;
    return NULL;
}

/*
 * Setup command table
 */
static const command_rec ftp_cmds[] = {
    AP_INIT_TAKE1("FTPLowPortSock", lowportd_set_socket, NULL, RSRC_CONF,
                  "name of the socket to use for creating low-numbered-port "
                  "connections from ftp (global only)"),
    AP_INIT_FLAG("FTP", ftp_enable, NULL, RSRC_CONF,
                 "Run an FTP server on this host"),
    AP_INIT_TAKE1("FTPTimeoutLogin", ftp_set_int_slot,
                  (void *) APR_OFFSETOF(ftp_server_config,
                                        timeout_login), RSRC_CONF,
                  "Idle time allowed when logging in"),
    AP_INIT_TAKE1("FTPTimeoutIdle", ftp_set_int_slot,
                  (void *) APR_OFFSETOF(ftp_server_config,
                                        timeout_idle), RSRC_CONF,
                  "Idle time allowed during a FTP session"),
    AP_INIT_TAKE1("FTPTimeoutData", ftp_set_int_slot,
                  (void *) APR_OFFSETOF(ftp_server_config,
                                        timeout_data), RSRC_CONF,
                  "Idle time allowed during a data transfer"),
    AP_INIT_TAKE1("FTPMaxLoginAttempts", ftp_set_int_slot,
                  (void *) APR_OFFSETOF(ftp_server_config,
                                        max_login_attempts), RSRC_CONF,
                  "Maximum number of login attempts"),
    AP_INIT_FLAG("FTPImplicitSSL", ftp_implicit_ssl, NULL, RSRC_CONF,
                 "Use SSL implicitly."),
    AP_INIT_RAW_ARGS("FTPOptions", ftp_options, NULL, RSRC_CONF,
                     "Set options for this server"),
    AP_INIT_TAKE1("FTPPASVaddr", ftp_set_pasv_addr, NULL, RSRC_CONF,
                  "Set the allowed PASV server IP address for the data "
                  "channel"),
    AP_INIT_TAKE1("FTPPASVbindaddr", ftp_set_pasv_bindaddr, NULL, RSRC_CONF,
                  "Set and bind the allowed PASV server IP "
                  "address for the data channel"),
    AP_INIT_TAKE2("FTPPASVrange", ftp_set_pasv_range, NULL, RSRC_CONF,
                  "Set the allowed PASV port range"),
    AP_INIT_FLAG("FTPEPSVIgnoreFamily", ftp_epsv_ignore_family,
                 NULL, RSRC_CONF, \
                 "Instructs EPSV handler to ignore the requested IPv4 or IPv6"
                 " address family (to accomodate network translation)"),
    AP_INIT_TAKE1("FTPBannerMessage", ftp_set_banner_message, NULL, RSRC_CONF,
                  "Set initial login message"),
    AP_INIT_TAKE1("FTPExitMessage", ftp_set_exit_message, NULL, RSRC_CONF,
                  "Set logout message"),
    AP_INIT_TAKE1("FTPHomeDir", ftp_set_homedir, NULL, RSRC_CONF,
                  "Set the path to directory containing user's "
                  "home directories"),
    AP_INIT_TAKE1("FTPDocRootEnv", ftp_set_docrootenv, NULL, RSRC_CONF,
                  "Set the DocumentRoot based on the given environment "
                  "variable, such as a per-user LDAP property"),
    AP_INIT_FLAG("FTPJailUser", ftp_set_jail, NULL, RSRC_CONF,
                 "Users are not allowed to leave their home directories"),
    AP_INIT_TAKE12("FTPActiveRange", ftp_set_active_ports, NULL, RSRC_CONF,
                 "Ports the server will use for connecting to the client."),
    AP_INIT_TAKE1("FTPLimitLoginUser", ftp_set_limit_peruser, NULL, RSRC_CONF,
                  "Set the maximum number of concurrent logins per user"),
    AP_INIT_TAKE1("FTPLimitLoginIP", ftp_set_limit_perip, NULL, RSRC_CONF,
              "Set the maximum number of concurrent logins per IP address"),
    AP_INIT_TAKE1("FTPLimitLoginServer", ftp_set_limit_perserver, NULL, RSRC_CONF,
                  "Set the maximum number of concurrent logins per server"),
    AP_INIT_TAKE1("FTPLimitDBFile", ftp_set_dbfile, NULL, RSRC_CONF,
                  "Set the location for the Login Limit DB file"),
    AP_INIT_TAKE1("FTPDataBlockSize", ftp_set_int_slot,
                  (void *) APR_OFFSETOF(ftp_server_config,
                                        data_block_size), RSRC_CONF,
                  "Block size in bytes to use during data transfers"),
    AP_INIT_TAKE1("FTPReadmeMessage", ftp_set_readme_message, NULL, OR_ALL,
                  "Set per-directory Readme file"),
    AP_INIT_TAKE1("FTPUmask", ftp_umask, NULL, OR_FILEINFO,
                  "Set the umask for created files"),
    AP_INIT_TAKE1("FTPDirUmask", ftp_dirumask, NULL, OR_FILEINFO,
                  "Set the umask for created directory"),
    {NULL}
};

static void register_hooks(apr_pool_t *p)
{
    ap_hook_pre_config(ftp_pre_config, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_post_config(ftp_post_config, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_child_init(ftp_child_init, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_process_connection(ftp_process_connection, NULL, NULL,
                               APR_HOOK_MIDDLE);
    ap_hook_create_request(ftp_create_request, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_insert_filter(ftp_insert_filter, NULL, NULL, APR_HOOK_MIDDLE);

#if ((AP_SERVER_MAJORVERSION_NUMBER < 3) && (AP_SERVER_MINORVERSION_NUMBER < 3))
    ap_hook_check_user_id(ftp_check_user_id, NULL, NULL, APR_HOOK_REALLY_FIRST);
#endif

    /* FTP filters */
    ftp_input_filter_handle = ap_register_input_filter("FTP_PROTOCOL",
                                                       ftp_protocol_filter,
                                                       NULL,
                                                       AP_FTYPE_PROTOCOL);
    ftp_crlf_filter_handle = ap_register_output_filter("FTP_CRLF",
                                                       ftp_crlf_filter,
                                                       NULL,
                                                     AP_FTYPE_RESOURCE - 1);
    ftp_data_out_filter_handle = ap_register_output_filter("FTP_DATA_OUT",
                                                        ftp_data_out_filter,
                                                           NULL,
                                                      AP_FTYPE_NETWORK - 2);

    /* Register all core command handlers */
    ftp_register_core_cmds(p);
}

module FTP_DECLARE_DATA ftp_module =
{
    STANDARD20_MODULE_STUFF,
    create_ftp_dir_config,      /* create per-directory config structure */
    merge_ftp_dir_config,       /* merge per-directory config structures */
    create_ftp_server_config,   /* create per-server config structure */
    merge_ftp_server_config,    /* merge per-server config structures */
    ftp_cmds,                   /* command apr_table_t */
    register_hooks              /* register hooks */
};
