/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.jobmaster.slotpool;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import javax.annotation.Nullable;
import org.apache.flink.runtime.clusterframework.types.AllocationID;
import org.apache.flink.runtime.clusterframework.types.ResourceID;
import org.apache.flink.runtime.jobmaster.SlotInfo;
import org.apache.flink.runtime.jobmaster.slotpool.AllocatedSlot;
import org.apache.flink.runtime.jobmaster.slotpool.AllocatedSlotPool;
import org.apache.flink.runtime.jobmaster.slotpool.SlotInfoWithUtilization;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultAllocatedSlotPool
implements AllocatedSlotPool {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultAllocatedSlotPool.class);
    private final Map<AllocationID, AllocatedSlot> registeredSlots = new HashMap<AllocationID, AllocatedSlot>();
    private final Map<AllocationID, Long> freeSlotsSince;
    private final Map<ResourceID, Set<AllocationID>> slotsPerTaskExecutor = new HashMap<ResourceID, Set<AllocationID>>();

    public DefaultAllocatedSlotPool() {
        this.freeSlotsSince = new HashMap<AllocationID, Long>();
    }

    @Override
    public void addSlots(Collection<AllocatedSlot> slots, long currentTime) {
        for (AllocatedSlot slot : slots) {
            this.addSlot(slot, currentTime);
        }
    }

    private void addSlot(AllocatedSlot slot, long currentTime) {
        Preconditions.checkState(!this.registeredSlots.containsKey(slot.getAllocationId()), "The slot pool already contains a slot with id %s", slot.getAllocationId());
        this.addSlotInternal(slot, currentTime);
        this.slotsPerTaskExecutor.computeIfAbsent(slot.getTaskManagerId(), resourceID -> new HashSet()).add(slot.getAllocationId());
    }

    private void addSlotInternal(AllocatedSlot slot, long currentTime) {
        this.registeredSlots.put(slot.getAllocationId(), slot);
        this.freeSlotsSince.put(slot.getAllocationId(), currentTime);
    }

    @Override
    public Optional<AllocatedSlot> removeSlot(AllocationID allocationId) {
        AllocatedSlot removedSlot = this.removeSlotInternal(allocationId);
        if (removedSlot != null) {
            ResourceID owner = removedSlot.getTaskManagerId();
            this.slotsPerTaskExecutor.computeIfPresent(owner, (resourceID, allocationIds) -> {
                allocationIds.remove(allocationId);
                if (allocationIds.isEmpty()) {
                    return null;
                }
                return allocationIds;
            });
            return Optional.of(removedSlot);
        }
        return Optional.empty();
    }

    @Nullable
    private AllocatedSlot removeSlotInternal(AllocationID allocationId) {
        AllocatedSlot removedSlot = this.registeredSlots.remove(allocationId);
        this.freeSlotsSince.remove(allocationId);
        return removedSlot;
    }

    @Override
    public AllocatedSlotPool.AllocatedSlotsAndReservationStatus removeSlots(ResourceID owner) {
        Set<AllocationID> slotsOfTaskExecutor = this.slotsPerTaskExecutor.remove(owner);
        if (slotsOfTaskExecutor != null) {
            ArrayList<AllocatedSlot> removedSlots = new ArrayList<AllocatedSlot>();
            HashMap<AllocationID, ReservationStatus> removedSlotsReservationStatus = new HashMap<AllocationID, ReservationStatus>();
            for (AllocationID allocationId : slotsOfTaskExecutor) {
                ReservationStatus reservationStatus = this.containsFreeSlot(allocationId) ? ReservationStatus.FREE : ReservationStatus.RESERVED;
                AllocatedSlot removedSlot = Preconditions.checkNotNull(this.removeSlotInternal(allocationId));
                removedSlots.add(removedSlot);
                removedSlotsReservationStatus.put(removedSlot.getAllocationId(), reservationStatus);
            }
            return new DefaultAllocatedSlotsAndReservationStatus(removedSlots, removedSlotsReservationStatus);
        }
        return new DefaultAllocatedSlotsAndReservationStatus(Collections.emptyList(), Collections.emptyMap());
    }

    @Override
    public boolean containsSlots(ResourceID owner) {
        return this.slotsPerTaskExecutor.containsKey(owner);
    }

    @Override
    public boolean containsSlot(AllocationID allocationId) {
        return this.registeredSlots.containsKey(allocationId);
    }

    @Override
    public boolean containsFreeSlot(AllocationID allocationId) {
        return this.freeSlotsSince.containsKey(allocationId);
    }

    @Override
    public AllocatedSlot reserveFreeSlot(AllocationID allocationId) {
        LOG.debug("Reserve free slot with allocation id {}.", (Object)allocationId);
        Preconditions.checkState(this.freeSlotsSince.remove(allocationId) != null, "The slot with id %s was not free.", allocationId);
        return this.registeredSlots.get(allocationId);
    }

    @Override
    public Optional<AllocatedSlot> freeReservedSlot(AllocationID allocationId, long currentTime) {
        AllocatedSlot allocatedSlot = this.registeredSlots.get(allocationId);
        if (allocatedSlot != null && !this.freeSlotsSince.containsKey(allocationId)) {
            this.freeSlotsSince.put(allocationId, currentTime);
            return Optional.of(allocatedSlot);
        }
        return Optional.empty();
    }

    @Override
    public Optional<SlotInfo> getSlotInformation(AllocationID allocationID) {
        return Optional.ofNullable(this.registeredSlots.get(allocationID));
    }

    @Override
    public Collection<AllocatedSlotPool.FreeSlotInfo> getFreeSlotsInformation() {
        HashMap<ResourceID, Integer> freeSlotsPerTaskExecutor = new HashMap<ResourceID, Integer>();
        for (AllocationID allocationId : this.freeSlotsSince.keySet()) {
            ResourceID owner = Preconditions.checkNotNull(this.registeredSlots.get(allocationId)).getTaskManagerId();
            int newCount = freeSlotsPerTaskExecutor.getOrDefault(owner, 0) + 1;
            freeSlotsPerTaskExecutor.put(owner, newCount);
        }
        ArrayList<AllocatedSlotPool.FreeSlotInfo> freeSlotInfos = new ArrayList<AllocatedSlotPool.FreeSlotInfo>();
        for (Map.Entry<AllocationID, Long> freeSlot : this.freeSlotsSince.entrySet()) {
            AllocatedSlot allocatedSlot = Preconditions.checkNotNull(this.registeredSlots.get(freeSlot.getKey()));
            ResourceID owner = allocatedSlot.getTaskManagerId();
            int numberOfSlotsOnOwner = this.slotsPerTaskExecutor.get(owner).size();
            int numberOfFreeSlotsOnOwner = (Integer)freeSlotsPerTaskExecutor.get(owner);
            double taskExecutorUtilization = (double)(numberOfSlotsOnOwner - numberOfFreeSlotsOnOwner) / (double)numberOfSlotsOnOwner;
            SlotInfoWithUtilization slotInfoWithUtilization = SlotInfoWithUtilization.from(allocatedSlot, taskExecutorUtilization);
            freeSlotInfos.add(DefaultFreeSlotInfo.create(slotInfoWithUtilization, freeSlot.getValue()));
        }
        return freeSlotInfos;
    }

    @Override
    public Collection<? extends SlotInfo> getAllSlotsInformation() {
        return this.registeredSlots.values();
    }

    private static enum ReservationStatus {
        FREE,
        RESERVED;

    }

    private static final class DefaultAllocatedSlotsAndReservationStatus
    implements AllocatedSlotPool.AllocatedSlotsAndReservationStatus {
        private final Collection<AllocatedSlot> slots;
        private final Map<AllocationID, ReservationStatus> reservationStatus;

        private DefaultAllocatedSlotsAndReservationStatus(Collection<AllocatedSlot> slots, Map<AllocationID, ReservationStatus> reservationStatus) {
            this.slots = slots;
            this.reservationStatus = reservationStatus;
        }

        @Override
        public boolean wasFree(AllocationID allocatedSlot) {
            return this.reservationStatus.get(allocatedSlot) == ReservationStatus.FREE;
        }

        @Override
        public Collection<AllocatedSlot> getAllocatedSlots() {
            return this.slots;
        }
    }

    private static final class DefaultFreeSlotInfo
    implements AllocatedSlotPool.FreeSlotInfo {
        private final SlotInfoWithUtilization slotInfoWithUtilization;
        private final long freeSince;

        private DefaultFreeSlotInfo(SlotInfoWithUtilization slotInfoWithUtilization, long freeSince) {
            this.slotInfoWithUtilization = slotInfoWithUtilization;
            this.freeSince = freeSince;
        }

        @Override
        public SlotInfoWithUtilization asSlotInfo() {
            return this.slotInfoWithUtilization;
        }

        @Override
        public long getFreeSince() {
            return this.freeSince;
        }

        private static DefaultFreeSlotInfo create(SlotInfoWithUtilization slotInfoWithUtilization, long idleSince) {
            return new DefaultFreeSlotInfo(Preconditions.checkNotNull(slotInfoWithUtilization), idleSince);
        }
    }
}

