/*
 * Decompiled with CFR 0.152.
 */
package org.apache.asterix.common.memory;

import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.apache.asterix.common.memory.FrameAction;
import org.apache.hyracks.api.exceptions.HyracksDataException;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class ConcurrentFramePool {
    private static final boolean DEBUG = false;
    private static final String ERROR_INVALID_FRAME_SIZE = "The size should be an integral multiple of the default frame size";
    private static final String ERROR_LARGER_THAN_BUDGET_REQUEST = "The requested frame size must not be greater than the allocated budget";
    private static final Logger LOGGER = LogManager.getLogger();
    private final String nodeId;
    private final int budget;
    private final int defaultFrameSize;
    private final ArrayDeque<ByteBuffer> pool;
    private final ArrayDeque<FrameAction> subscribers = new ArrayDeque();
    private final Map<Integer, ArrayDeque<ByteBuffer>> largeFramesPools;
    private int handedOut;
    private int created;

    public ConcurrentFramePool(String nodeId, long budgetInBytes, int frameSize) {
        this.nodeId = nodeId;
        this.defaultFrameSize = frameSize;
        this.budget = (int)(budgetInBytes / (long)frameSize);
        this.pool = new ArrayDeque(this.budget);
        this.largeFramesPools = new HashMap<Integer, ArrayDeque<ByteBuffer>>();
    }

    public int getMaxFrameSize() {
        return this.budget * this.defaultFrameSize;
    }

    public synchronized ByteBuffer get() {
        if (this.subscribers.isEmpty()) {
            return this.doGet();
        }
        return null;
    }

    private ByteBuffer doGet() {
        if (this.handedOut < this.budget) {
            ++this.handedOut;
            return this.allocate();
        }
        return null;
    }

    public int remaining() {
        return this.budget - this.handedOut;
    }

    private ByteBuffer doGet(int bufferSize) throws HyracksDataException {
        if (bufferSize % this.defaultFrameSize != 0) {
            throw new HyracksDataException(ERROR_INVALID_FRAME_SIZE);
        }
        int multiplier = bufferSize / this.defaultFrameSize;
        if (multiplier > this.budget) {
            throw new HyracksDataException(ERROR_LARGER_THAN_BUDGET_REQUEST);
        }
        if (this.handedOut + multiplier <= this.budget) {
            this.handedOut += multiplier;
            ArrayDeque<ByteBuffer> largeFramesPool = this.largeFramesPools.get(multiplier);
            if (largeFramesPool == null || largeFramesPool.isEmpty()) {
                if (this.created + multiplier > this.budget) {
                    this.freeup(multiplier);
                }
                this.created += multiplier;
                return ByteBuffer.allocate(bufferSize);
            }
            ByteBuffer buffer = largeFramesPool.poll();
            buffer.clear();
            return buffer;
        }
        return null;
    }

    public synchronized ByteBuffer get(int bufferSize) throws HyracksDataException {
        if (this.subscribers.isEmpty()) {
            return this.doGet(bufferSize);
        }
        return null;
    }

    private int freeup(int desiredNumberOfFreePages) {
        int needToFree = desiredNumberOfFreePages - (this.budget - this.created);
        int freed = 0;
        Iterator<Map.Entry<Integer, ArrayDeque<ByteBuffer>>> it = this.largeFramesPools.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<Integer, ArrayDeque<ByteBuffer>> entry = it.next();
            if (entry.getKey() == desiredNumberOfFreePages) continue;
            while (!entry.getValue().isEmpty()) {
                entry.getValue().pop();
                if ((freed += entry.getKey().intValue()) < needToFree) continue;
                this.created -= freed;
                return freed;
            }
            it.remove();
        }
        needToFree -= freed;
        while (needToFree > 0) {
            this.pool.pop();
            --needToFree;
            ++freed;
        }
        this.created -= freed;
        return freed;
    }

    private ByteBuffer allocate() {
        if (this.pool.isEmpty()) {
            if (this.created == this.budget) {
                this.freeup(1);
            }
            ++this.created;
            return ByteBuffer.allocate(this.defaultFrameSize);
        }
        ByteBuffer buffer = this.pool.pop();
        buffer.clear();
        return buffer;
    }

    public synchronized boolean get(Collection<ByteBuffer> buffers, int count) {
        if (this.handedOut + count <= this.budget) {
            this.handedOut += count;
            for (int i = 0; i < count; ++i) {
                buffers.add(this.allocate());
            }
            return true;
        }
        return false;
    }

    public String toString() {
        return "ConcurrentFramePool  [" + this.nodeId + "](consumed:" + this.handedOut + "/" + this.budget + ")";
    }

    public synchronized void release(Collection<ByteBuffer> buffers) throws HyracksDataException {
        for (ByteBuffer buffer : buffers) {
            this.release(buffer);
        }
    }

    public synchronized void release(ByteBuffer buffer) throws HyracksDataException {
        FrameAction frameAction;
        ByteBuffer freeBuffer;
        int multiples = buffer.capacity() / this.defaultFrameSize;
        this.handedOut -= multiples;
        if (multiples == 1) {
            this.pool.add(buffer);
        } else {
            ArrayDeque<ByteBuffer> largeFramesPool = this.largeFramesPools.get(multiples);
            if (largeFramesPool == null) {
                largeFramesPool = new ArrayDeque();
                this.largeFramesPools.put(multiples, largeFramesPool);
            }
            largeFramesPool.push(buffer);
        }
        while (!this.subscribers.isEmpty() && (freeBuffer = (frameAction = this.subscribers.peek()).getSize() == this.defaultFrameSize ? this.doGet() : this.doGet(frameAction.getSize())) != null) {
            int handedOutBeforeCall = this.handedOut;
            try {
                frameAction.call(freeBuffer);
            }
            catch (Exception e) {
                LOGGER.log(Level.ERROR, "Error while attempting to answer a subscription. Buffer will be reclaimed", (Throwable)e);
                if (this.handedOut == handedOutBeforeCall) {
                    this.release(freeBuffer);
                }
                throw e;
            }
            finally {
                this.subscribers.remove();
            }
        }
    }

    public synchronized boolean subscribe(FrameAction frameAction) throws HyracksDataException {
        if (this.subscribers.isEmpty()) {
            ByteBuffer buffer = frameAction.getSize() == this.defaultFrameSize ? this.doGet() : this.doGet(frameAction.getSize());
            if (buffer != null) {
                frameAction.call(buffer);
                return false;
            }
        } else {
            int multiplier = frameAction.getSize() / this.defaultFrameSize;
            if (multiplier > this.budget) {
                throw new HyracksDataException(ERROR_LARGER_THAN_BUDGET_REQUEST);
            }
        }
        this.subscribers.add(frameAction);
        return true;
    }

    public Collection<FrameAction> getSubscribers() {
        return this.subscribers;
    }
}

