/*-
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (c) 2019 Steven G. Kargl <kargl@FreeBSD.ORG>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
#include <complex.h>
#include <float.h>
#include <math.h>

#include "fpmath.h"
#include "math_private.h"
#include "k_expl.h"

/* XXX cexpl() should be converted to use bits likeo src/s_cexp.c. */

static const long double
cexp_ovfl = 2.27892930024498818830197576893019292e+04L,
exp_ovfl = 1.13565234062941439494919310779707649e+04L;

long double complex
cexpl(long double complex z)
{
	long double c, exp_x, s, x, y;

	x = creall(z);
	y = cimagl(z);

	/* cexp(x + I 0) = exp(x) + I 0 */
	if (y == 0)
		return (CMPLXL(expl(x), y));
	/* cexp(0 + I y) = cos(y) + I sin(y) */
	if (x == 0) {
		sincosl(y, &s, &c);
		return (CMPLXL(c, s));
	}

	if (!isfinite(y)) {
		if (isfinite(x) || isnan(x)) {
			/* cexp(finite|NaN +- I Inf|NaN) = NaN + I NaN */
			return (CMPLXL(y - y, y - y));
		} else if (isinf(x) && copysignl(1.L, x) < 0) {
			/* cexp(-Inf +- I Inf|NaN) = 0 + I 0 */
			return (CMPLXL(0.0, 0.0));
		} else {
			/* cexp(+Inf +- I Inf|NaN) = Inf + I NaN */
			return (CMPLXL(x, y - y));
		}
	}

	if (x > exp_ovfl && x < cexp_ovfl) {
		/*
		 * x is between exp_ovfl and cexp_ovfl, so we must scale to
		 * avoid overflow in exp(x).
		 */
		return (__ldexp_cexpl(z, 0));
	} else {
		/*
		 * Cases covered here:
		 *  -  x < exp_ovfl and exp(x) won't overflow (common case)
		 *  -  x > cexp_ovfl, so exp(x) * s overflows for all s > 0
		 *  -  x = +-Inf (generated by exp())
		 *  -  x = NaN (spurious inexact exception from y)
		 */
		exp_x = expl(x);
		sincosl(y, &s, &c);
		return (CMPLXL(exp_x * c, exp_x * s));
	}
}
